/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HasAssign.h
 *    Typetrait to test whether a class has a certain assignment operator.
 *    This trait is missing in boost:: and std::type_traits.
 *
 * @author Erik Einhorn
 * @date   2012/12/12
 */

#ifndef _MIRA_HASASSIGN_H_
#define _MIRA_HASASSIGN_H_

#include <type_traits>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace detail {

template <typename T, typename U>
class HasAssignHelper
{
private:
	struct Failure {};

	template <typename V>
	static V value() {}

	template <typename V>
	static auto testassign(T x, V y) -> decltype(x = y);
	static Failure testassign(...);

public:
	typedef decltype(testassign(value<T>(), value<U>())) AssignResultType;

	typedef std::is_same<AssignResultType,Failure> not_type;
};

}

///////////////////////////////////////////////////////////////////////////////

/**
 * Evaluates to std::true_type, if U can be assigned to T, i.e. if T has an
 * assignment T::operator=(const U&) that allows
 * \code
 * T t;
 * U u;
 * t = u;
 * \endcode
 *
 * Unfortunatelly, there is no way to detect, whether the assignment operator
 * is public accessible or private. If it is private, this type trait will
 * produce a compiler error.
 */
template <typename T, typename U>
class has_assign :
	public std::integral_constant<bool, !detail::HasAssignHelper<T,U>::not_type::value>
{};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
