/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file core.hpp
 *    Non-intrusive reflection for OpenCV core types.
 *
 * @author Christoph Weinrich
 * @date   2011/05/18
 */

#ifndef _MIRA_CORE_HPP_
#define _MIRA_CORE_HPP_

#include <opencv2/core/core.hpp>
#include <serialization/BinarySerializer.h>
#include <serialization/NoGenericReflect.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Size.
 * This methods implements the reflection of width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Size& d) {
	r.property("Width", d.width, "Size width");
	r.property("Height", d.height, "Size height");
}

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Point.
 * This methods implements the reflection of width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Point& d) {
	r.property("X", d.x, "Point x");
	r.property("Y", d.y, "Point y");
}

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Rect.
 * This methods implements the reflection of x, y, width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Rect& d) {
	r.property("X", d.x, "Rect x");
	r.property("Y", d.y, "Rect y");
	r.property("Width", d.width, "Rect width");
	r.property("Height", d.height, "Rect height");
}

///////////////////////////////////////////////////////////////////////////////

MIRA_NO_GENERIC_REFLECT(cv::Mat)

/** @brief Reflect method for binaryStream serialization.
 * This methods implements the reflection of width, height, type,
 * and the actual image data serialization.
 */
template<typename BinaryStream>
void reflect(BinarySerializer<BinaryStream>& r, cv::Mat& mat) {
	// serialization is only for 2 dimensional matrices until yet
	assert(mat.dims == 2);
	r.version(1);
	r.member("Cols", mat.cols, "The width of the image");
	r.member("Rows", mat.rows, "The height of the image");
	int type = mat.type();
	r.member("Type", type, "The data type including number of channels");

	std::size_t bytesPerLine = mat.elemSize() * mat.cols;

	// serialize the image data line by line
	const uchar* data = mat.data;
	for (int i = 0; i < mat.rows; ++i, data += mat.step)
		r.write(data, bytesPerLine);
}

/** @brief Reflect method for binaryStream deserialization.
 * This methods implements the reflection for width, height, type,
 * and the image data deserialization.
 */
template<typename BinaryStream>
void reflect(BinaryDeserializer<BinaryStream>& r, cv::Mat& mat) {
	int cols, rows, type;
	// serialization is only for 2 dimensional matrices until yet
	r.version(1);
	r.member("Cols", cols, "The width of the image");
	r.member("Rows", rows, "The height of the image");
	r.member("Type", type, "The data type including number of channels");

	// create the empty image
	mat.create(rows, cols, type);

	std::size_t bytesPerLine = mat.elemSize() * mat.cols;

	// and deserialize the image line by line
	uchar* data = mat.data;
	for (int i = 0; i < mat.rows; ++i, data += mat.step)
		r.read(data, bytesPerLine);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
