/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SplitReflect.h
 *    Provides MIRA_SPLIT_REFLECT macros.
 *
 * @author Erik Einhorn
 * @date   2010/09/23
 */

#ifndef _MIRA_SPLITREFLECT_H_
#define _MIRA_SPLITREFLECT_H_

#ifndef Q_MOC_RUN
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/identity.hpp>
#endif

namespace mira {

template <typename Derived>
class ReflectorInterface;

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
// default
template<typename DerivedReflector, typename T>
void reflectRead(mira::ReflectorInterface<DerivedReflector>& r, T& v)
{
	// #########################################################################
	// If you get a compiler error here, you used the splitReflect() method
	// or the MIRA_SPLIT_REFLECT macro, but forgot to specialize and implement
	// the reflectRead() and reflectWrite() methods for your type. Additionally,
	// make sure, that these functions are placed within the namespace "mira".
	// #########################################################################
	static_assert(sizeof(T)==0, "You must implement reflectRead() "
	                            "if you use the MIRA_SPLIT_REFLECT macro");
}

template<typename DerivedReflector, typename T>
void reflectWrite(mira::ReflectorInterface<DerivedReflector>& r, T& v)
{
	// #########################################################################
	// If you get a compiler error here, you used the splitReflect() method
	// or the MIRA_SPLIT_REFLECT macro, but forgot to specialize and implement
	// the reflectRead() and reflectWrite() methods for your type. Additionally,
	// make sure, that these functions are placed within the namespace "mira".
	// #########################################################################
	static_assert(sizeof(T)==0, "You must implement reflectWrite() "
	                            "if you use the MIRA_SPLIT_REFLECT macro");
}

struct GlobalReflectRead {
	template<typename Derived, typename T>
	static void invoke(mira::ReflectorInterface<Derived>& r, T& value) {
		Derived& reflector = static_cast<Derived&>(r);
		reflectRead(reflector, value);
	}
};

struct GlobalReflectWrite {
	template<typename Reflector, typename T>
	static void invoke(Reflector& r, T& value) {
		reflectWrite(r, value);
	}
};

template<typename Derived, typename T>
inline void splitReflect(mira::ReflectorInterface<Derived>& r, T& value)
{
	Derived& reflector = static_cast<Derived&>(r);
	typedef Derived Reflector;
	typedef typename boost::mpl::eval_if<
				typename Reflector::isReadOnly,
					boost::mpl::identity<GlobalReflectRead>,
					boost::mpl::identity<GlobalReflectWrite>
		>::type type;
	type::invoke(reflector, value);
}
///@endcond

/**
 * @ingroup SerializationModule
 *
 * Macro that splits the reflect() method into a reflectRead() and
 * reflectWrite() method. reflectRead() is called if the visiting
 * Reflector is read-only and reflectWrite() is called if the visiting
 * Reflector is writing only.
 * In contrast to the @ref MIRA_SPLIT_REFLECT_MEMBER macro this macro can be used
 * outside of classes only, for non-intrusive variants of reflect().
 *
 * @see @ref SerializationPage
 */
#define MIRA_SPLIT_REFLECT(Type)                          \
	template<typename Derived>                            \
	void reflect(mira::ReflectorInterface<Derived>& r, Type& value)\
	{                                                     \
		splitReflect(r, value);                           \
	}

/**
 * @ingroup SerializationModule
 *
 * Macro that splits the reflect() method into a reflectRead() and
 * reflectWrite() method. reflectRead() is called if the visiting
 * Reflector is read-only and reflectWrite() is called if the visiting
 * Reflector is writing only.
 * In contrast to the @ref MIRA_SPLIT_REFLECT macro this macro can be used
 * within classes.
 *
 * @see @ref SerializationPage
 */
#define MIRA_SPLIT_REFLECT_MEMBER                         \
	struct __reflectRead {                                \
		template<typename Reflector, typename Class>      \
		static void invoke(Reflector& r, Class* This) {   \
			This->reflectRead(r);                         \
		}                                                 \
	};                                                    \
	                                                      \
	struct __reflectWrite {                               \
		template<typename Reflector, typename Class>      \
		static void invoke(Reflector& r, Class* This) {   \
			This->reflectWrite(r);                        \
		}                                                 \
	};                                                    \
	template<typename Derived>                            \
	void reflect(mira::ReflectorInterface<Derived>& r)    \
	{                                                     \
		Derived& reflector = static_cast<Derived&>(r);    \
		typedef Derived Reflector;                        \
		typedef typename boost::mpl::eval_if<             \
				typename Reflector::isReadOnly,           \
				boost::mpl::identity<__reflectRead>,      \
				boost::mpl::identity<__reflectWrite>      \
			>::type type;                                 \
		type::invoke(reflector, this);                    \
	}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_SPLITREFLECT_H_ */
