/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Class.h
 *    $Definition of the Class which supports some kind of class reflection
 *    and acts like a class factory$.
 *
 * @author Ronny Stricker
 * @date   2010/10/20
 */

#ifndef _MIRA_CLASS_H_
#define _MIRA_CLASS_H_

#include <vector>
#include <map>
#include <cstdarg>

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

#include <thread/Thread.h>
#include <platform/Typename.h>
#include <utils/Foreach.h>
#include <error/Exceptions.h>
#include <serialization/SplitReflect.h>
#include <factory/TypeId.h>
#include <factory/Object.h>


namespace mira {

//////////////////////////////////////////////////////////////////////////////

class Class;
class ClassProxy;
class Object;

typedef ClassProxy ClassPtr;

template <typename CLASS>
inline CLASS* mira_factoryDynamicCast( Object* base );

// *****************************************************************************
// *** Class
// *****************************************************************************

/**
 * @brief Class object which supports some kind of class reflection.
 * Furthermore, the class object acts like a class factory, which is responsible
 * for all registered children of the class.
 * @ingroup ClassFactoryModule
 */
class MIRA_BASE_EXPORT Class
{
	friend class ClassFactory;

	template<typename CLASS>
		friend class TClass;
	friend class ClassProxy;

public:
	virtual ~Class();

	/**
	 * @brief Return identifier for the class.
	 * The identifiers probably contains the namespace of the class.
	 */
	virtual std::string const& getIdentifier() const;

	/**
	 * @brief Return name of the class.
	 * The name will not contain any informations about the class namespace
	 * (if the macro for automatic extraction of the name has been used).
	 * Otherwise the function simply returns the name given in the object macro.
	 */
	virtual std::string const& getName() const;

	/**
	 * @brief Return unique id for the class.
	 * Keep in mind that this id is only valid during execution time and
	 * only for this process.
	 */
	virtual int getTypeId() const = 0;

	/**
	 * @brief Returns the platform independent C++ typename of the class.
	 */
	virtual Typename getTypename() const = 0;

	/**
	 * @brief Return map with meta information.
	 * meta information is coded as key-value pair.
	 */
	std::map<std::string, std::string> const& getMetaInfo() const;

	/**
	 * @brief Returns meta information for the given meta key.
	 * This method is provided for convenience. If the specified meta
	 * information is not available an empty string is returned.
	 */
	std::string const& getMetaInfo(const std::string& meta) const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * The derived TClass and VacantClasses will return the derived class type
	 * using covariant return types.
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no default constructor
	 */
	virtual Object* newInstance() const = 0;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * The class is casted to the desired class type.
	 * @throw XBadCast If the cast fails
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no default constructor
	 */
	template <class CLASS>
	CLASS* newInstance() const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * A list of parameters for the constructor can be passed.
	 * The derived TClass and VacantClasses will NOT return the derived class type
	 * since the combination of va_list and covariant return types seems to be
	 * unsupported...
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no constructor matching the given
	 *        parameter list (or if the constructor is not registered)
	 */
	virtual Object* newInstance( int paramCount, ... ) const = 0;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * The derived TClass and VacantClasses will return the derived class type
	 * using covariant return types.
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no default constructor
	 */
	virtual Object* newInstance( std::string const& childIdentifier ) const = 0;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * The class is casted to the desired class type.
	 * @throw XBadCast If the cast fails
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no default constructor
	 */
	template <class CLASS>
	CLASS* newInstance( std::string const& childIdentifier ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * A list of parameters for the constructor can be passed.
	 * The derived TClass and VacantClasses will NOT return the derived class type
	 * since the combination of va_list and covariant return types seems to be
	 * unsupported...
	 * @throw XLogical If the class is abstract.
	 * @throw XLogical If the class has no constructor matching the given
	 *        parameter list (or if the constructor is not registered)
	 */
	virtual Object* newInstance( std::string const& childIdentifier,
	                             int paramCount, ... ) const = 0;

	/**
	 * @brief Return true if a class with the desired identifier is registered.
	 */
	bool isClassRegistered( std::string const& classIdentifier ) const;
	
	/**
	 * @brief Remove the given child class from the class.
	 * Calls eraseChild on all parents and removes the given class from its
	 * own child list.
	 */
	void eraseChild( Class const* const iClass );

	/**
	 * @brief Remove the given class from the list of parents.
	 * Calls eraseParent on all children and removes the given class from its
	 * own parent list.
	 */
	void eraseParent( Class const* const iClass );

	/**
	 * @brief Return the ClassProxy object for the desired Class.
	 * @throws XInvalidParameter If the identifier is unknown.
	 */
	ClassProxy getClassByIdentifier(
			std::string const& classIdentifier ) const;

	/**
	 * @brief Return vector of ClassProxy objects matching the meta criterion.
	 * The returned classes matches the given metakey and metavalue.
	 */
	std::vector<ClassProxy> getClassByMeta( std::string const& metaKey,
	                                        std::string const& metaValue ) const;

	/**
	 * @brief Return vector of Class objects returning true for the given
	 * comparison function.
	 * The comparison function has to take the meta information
	 * std::map<std::string, std::string> const& as argument and should
	 * return true if the meta information matches the desired criterion.
	 */
	template <class T>
	std::vector<ClassProxy> getClassByMeta( T funcPtr ) const;

	/**
	 * @brief Return registered children for the associated class to the given map.
	 */
	std::map<std::string, ClassProxy> getDerivedClasses() const;

	/**
	 * @brief Return map with the direct parents of this class.
	 */
	std::map<std::string, ClassProxy> getDirectParents() const;

	/**
	 * @brief Return true if given class is registered and derived from this class.
	 */
	bool isBaseOf( Class const* const derived ) const;

	/**
	 * @brief Return true if given class is registered and derived from this class.
	 */
	bool isBaseOf( ClassProxy derived ) const;

	/**
	 * @brief Return true if given class is registered and derived from this class
	 */
	bool isBaseOf( std::string const& identifier ) const;

	/**
	 * @brief Return true if given class is registered and parent of this class.
	 */
	bool isDerivedFrom( Class const* const base ) const;

	/**
	 * @brief Return true if given class is registered and parent of this class.
	 */
	bool isDerivedFrom( ClassProxy base ) const;

	/**
	 * @brief Return true if the given class is a direct parent.
	 */
	bool isDirectlyDerivedFrom( Class const* const base ) const;

	/**
	 * @brief Return true if the given class is a direct parent.
	 */
	bool isDirectlyDerivedFrom( ClassProxy base ) const;

	/**
	 * @brief Return true if the associated class is abstract.
	 */
	virtual bool isAbstract() const = 0;

	/**
	 * @brief Return true if the library which contains the associated class
	 * is loaded. TClass classes will return true and Vacant class classes will
	 * return false.
	 */
	bool isLibraryLoaded() const;

	MIRA_SPLIT_REFLECT_MEMBER

	/**
	 * @brief Implementation of class member reflection.
	 * Attention: the default implementation will reflect the abstract key in
	 * order to read the information from the object. However, the abstract key
	 * will be ignored when writing information to the object.
	 */
	template<typename Reflector>
	void reflectRead( Reflector& r );

	/**
	 * @brief Implementation of class member reflection.
	 * Attention: the default implementation will reflect the abstract key in
	 * order to read the information from the object. However, the abstract key
	 * will be ignored when writing information to the object.
	 */
	template<typename Reflector>
	void reflectWrite( Reflector& r );

	/**
	 * Returns true, if the two classes are identical.
	 */
	bool operator==(Class const& other) const {
		return this == &other;
	}

	/**
	 * Returns true, if the two classes are not identical.
	 */
	bool operator!=(Class const& other) const {
		return this != &other;
	}
	
	/**
	 * Return true, if the comparison of the identifiers returns true.
	 */
	bool operator <(Class const& other ) const {
		return mIdentifier < other.mIdentifier;
	}

protected:
	/**
	 * @brief Call the specific constructor depending on the number of
	 * parameters passed to the function.
	 */
	virtual Object* newVAInstance( int paramCount, std::va_list ) const = 0;

	/**
	 * @brief This function is called: the constructor
	 */
	Class( std::string const& identifier,
			std::string const& name,
			std::map<std::string, std::string> const& metaInfo,
			bool libLoaded );

	/**
	 * @brief This function is called: the constructor
	 */
	Class ( std::string const& identifier, std::string const& name,
			bool libLoaded );

private:
	/**
	 * @brief This function is called: the constructor
	 */
	Class() {}

protected:
	std::map<std::string, ClassProxy > mDerivedChildren;/**< map of children */
	std::map<std::string, ClassProxy> mDirectParents; /**< map of parents */
	std::string mIdentifier;                          /**< class identifier */
	std::string mName;                                /**< class name */
	std::string mLib;                                 /**< lib name */
	std::map<std::string, std::string> mMetaInfo;     /**< meta info of class */
	bool mLibLoaded;                                  /**< is associated lib loaded? */
};

/**
 * @brief The class proxy assures that the pointer to the class object is always
 * valid. This class supports the transparent change of a vacant class to a
 * TClass object. Furthermore, it is necessary for thread save access.
 * @ingroup ClassFactoryModule
 */
class MIRA_BASE_EXPORT ClassProxy
{
	friend class ClassFactory;
	friend class VacantClass;
	friend class Class;

	template<typename CLASS>
		friend class TClass;

public:
	ClassProxy() : mClass( NULL ), mThreadMutex( NULL )
	{
	}

	ClassProxy( ClassProxy const& other )
	{
		mClass = other.mClass;
		mThreadMutex = other.mThreadMutex;
	}

	bool operator== (ClassProxy const& other ) const {
		if ( !mClass || !*mClass || !other.mClass || !*other.mClass )
			MIRA_THROW( XLogical, "ClassProxy invalid!" );
		return **other.mClass == **mClass;
	}

	bool operator!= (ClassProxy const& other ) const {
		if ( !mClass || !*mClass || !other.mClass || !*other.mClass )
			MIRA_THROW( XLogical, "ClassProxy invalid!" );
		return **other.mClass != **mClass;
	}

	bool operator <(ClassProxy const& other ) const {
		if ( !mClass || !*mClass || !other.mClass || !*other.mClass )
			MIRA_THROW( XLogical, "ClassProxy invalid!" );
		return **mClass < **other.mClass;
	}

	/**
	 * @brief Return identifier for the class.
	 * @see mira::Class::getIdentifier()
	 */
	std::string const& getIdentifier() const;

	/**
	 * @brief Return name of the class.
	 * @see mira::Class::getName()
	 */
	std::string const& getName() const;

	/**
	 * @brief Return unique id for the class.
	 * @see mira::Class::getTypeId()
	 */
	int getTypeId() const;

	/**
	 * @brief Returns the platform independent C++ typename of the class.
	 * @see mira::Class::getTypename()
	 */
	Typename getTypename() const;

	/**
	 * @brief Return map with meta information.
	 * @see mira::Class::getMetaInfo()
	 */
	std::map<std::string, std::string> const& getMetaInfo() const;

	/**
	 * @brief Returns meta information for the given meta key.
	 * @see mira::Class::getMetaInfo( meta )
	 */
	std::string const& getMetaInfo(const std::string& meta) const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @see mira::Class::newInstance()
	 */
	Object* newInstance() const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @see mira::Class::newInstance()
	 */
	template <class CLASS>
	CLASS* newInstance() const
	{
		boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
		return (*mClass)->newInstance<CLASS>();
	}

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @see mira::Class::newInstance()
	 */
	 Object* newInstance( int paramCount, ... ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * @see mira::Class::newInstance()
	 */
	Object* newInstance( std::string const& childIdentifier ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * @see mira::Class::newInstance( std::string const& )
	 */
	template <class CLASS>
	CLASS* newInstance( std::string const& childIdentifier ) const
	{
		boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
		return (*mClass)->newInstance<CLASS>( childIdentifier );
	}

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * @see mira::Class::newInstance( std::string const&, int, ... )
	 */
	Object* newInstance( std::string const& childIdentifier,
			int paramCount, ... ) const;

	/**
	 * @brief Return true if a class with the desired identifier is registered.
	 * @see mira::Class::isClassRegistered()
	 */
	bool isClassRegistered( std::string const& classIdentifier ) const;
	
	/**
	 * @brief Remove the given child class from the class.
	 * Calls eraseChild on all parents and removed the given class from its
	 * own child list.
	 */
	void eraseChild( Class const* const iClass );

	/**
	 * @brief Remove the given class from the list of parents.
	 * Calls eraseParent on all children and removes the given class from its
	 * own parent list.
	 */
	void eraseParent( Class const* const iClass );

	/**
	 * @brief Return the Class object for the desired Class.
	 * @see mira::Class::getClassByIdentifier( classIdentifier )
	 */
	ClassProxy getClassByIdentifier(std::string const& classIdentifier ) const;

	/**
	 * @brief Return vector of Class objects matching the meta criterion.
	 * @see mira::Class::getClassByMeta( std::string const&, std::string const& )
	 */
	std::vector<ClassProxy> getClassByMeta(std::string const& metaKey,
	                                       std::string const& metaValue ) const;

	/**
	 * @brief Return vector of Class objects returning true for the given
	 * @see mira::Class::getClassByMeta()
	 */
	template <class T>
	std::vector<ClassProxy> getClassByMeta( T funcPtr ) const
	{
		boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
		return (*mClass)->getClassByMeta( funcPtr );
	}

	/**
	 * @brief Return registered children for the associated class to the given map.
	 * @see mira::Class::getDerivedClasses()
	 */
	std::map<std::string, ClassProxy > getDerivedClasses() const;

	/**
	 * @brief Return map with the direct parents of this class.
	 * @see mira::Class::getDirectParents()
	 */
	std::map<std::string, ClassProxy > getDirectParents() const;

	/**
	 * @brief Return true if given class is registered and derived from this class.
	 * @see mira::Class::isBaseOf( Class const* const )
	 */
	bool isBaseOf( Class const* const derived ) const;

	/**
	 * @brief Return true if given class is registered and derived from this class.
	 * @see mira::Class::isBaseOf( ClassProxy )
	 */
	bool isBaseOf( ClassProxy derived ) const;

	/**
	 * @brief Return true if given class is registered and derived from this class
	 * @see mira::Class::isBaseOf( std::string const& )
	 */
	bool isBaseOf( std::string const& identifier ) const;

	/**
	 * @brief Return true if given class is registered and parent of this class.
	 * @see mira::Class:isDerivedFrom( Class const* const )
	 */
	bool isDerivedFrom( Class const* const base ) const;

	/**
	 * @brief Return true if given class is registered and parent of this class.
	 * @see mira::Class::isDerivedFrom( ClassProxy )
	 */
	bool isDerivedFrom( ClassProxy base ) const;

	/**
	 * @brief Return true if the given class is a direct parent.
	 * @see mira::Class::isDirectlyDerivedFrom( Class const* const )
	 */
	bool isDirectlyDerivedFrom( Class const* const base ) const;

	/**
	 * @brief Return true if the given class is a direct parent.
	 * @see mira::Class::isDirectlyDerivedFrom( ClassProxy )
	 */
	bool isDirectlyDerivedFrom( ClassProxy base ) const;

	/**
	 * @brief Return true if the associated class is abstract.
	 */
	bool isAbstract() const;

	/**
	 * @brief Return true if the library which contains the associated class
	 * is loaded. TClass classes will return true and Vacant class classes will
	 * return false.
	 */
	bool isLibraryLoaded() const;

protected:
	Object* newVAInstance( int paramCount, std::va_list list ) const;

	ClassProxy( boost::shared_ptr<Class>* iClass );
	void setClass( boost::shared_ptr<Class>* iClass );

protected:
	boost::shared_ptr<Class>* mClass;
	boost::recursive_mutex* mThreadMutex;
};

///@cond INTERNAL

// *****************************************************************************
// *** ClassFactoryAbstractClassBuilder
// *****************************************************************************

/**
 * @brief This object is responsible for the construction of abstract classes.
 * Since it is quite hard to construct abstract classes it always throws.
 */
struct ClassFactoryAbstractClassBuilder {

	template<typename CLASS>
	static CLASS* invoke( ) {
		MIRA_THROW( XLogical, "Cannot create abstract class ("
		            + CLASS::CLASS().getIdentifier() + ")!" );
		return NULL;
	}

	template<int N, typename CLASS>
	static CLASS* invoke( std::va_list ) {
		MIRA_THROW( XLogical, "Cannot create abstract class ("
		            + CLASS::CLASS().getIdentifier() + ")!" );
		return NULL;
	}
};

// *****************************************************************************
// *** ClassFactoryNoDefaultConstClassBuilder
// *****************************************************************************

/**
 * @brief This is the default implementation for classes which have no default
 * constructor. Since a generic implementation for this case is not possible
 * it always return NULL;
 */
struct ClassFactoryNoDefaultConstClassBuilder {
	template<typename CLASS>
	static CLASS* invoke( ) {
		MIRA_THROW( XLogical, "No public default constructor available for class"
		            "creation (" + CLASS::CLASS().getIdentifier() + ")!" );
		// go home
		return NULL;
	}
};

// *****************************************************************************
// *** ClassFactoryDefaultConstClassBuilder
// *****************************************************************************

/**
 * @brief This is the default implementation for regular classes (with default
 * constructor). The implementation of the construction with default constructor
 * is straight forward. However, constructors with arbitrary parameters cannot
 * be handled by this class.
 */
struct ClassFactoryDefaultConstClassBuilder {

	template<typename CLASS>
	static CLASS* invoke( ) {
		return new CLASS;
	}

	template<int N, typename CLASS>
	static CLASS* invoke( std::va_list ) {
		MIRA_THROW( XLogical, "Desired parameter constructor not registered ("
		            + CLASS::CLASS().getIdentifier() + ")!" );
		return NULL;
	}
};

///@endcond

// *****************************************************************************
// *** Class implementation
// *****************************************************************************

template <class CLASS>
inline	CLASS* mira::Class::newInstance() const
{
	Object* tBase = newInstance();
	return mira_factoryDynamicCast<CLASS>( tBase );
}

template <class CLASS>
inline CLASS* mira::Class::newInstance( std::string const& childIdentifier ) const
{
	Object* tBase = newInstance( childIdentifier );
	return mira_factoryDynamicCast<CLASS>( tBase );
}

template <class T>
inline std::vector<ClassProxy> mira::Class::getClassByMeta( T funcPtr ) const
{
	std::vector<ClassProxy> tReturn;
	// apply matching function for each registered child
	std::map<std::string, ClassProxy >::const_iterator tIt = mDerivedChildren.begin();
	for( ; tIt != mDerivedChildren.end(); tIt++ ) {
		if ( funcPtr( tIt->second.getMetaInfo() ) ) {
			tReturn.push_back( tIt->second );
		}
	}
	return tReturn;
}




template<typename Reflector>
inline void mira::Class::reflectRead(Reflector& r)
{
	r.member( "Identifier" , mIdentifier, "" );
	r.member( "Name" , mName, "" );
	r.member( "Lib" , mLib, "" );
	r.member( "Meta" , mMetaInfo, "" );
	bool tAbstract = isAbstract();
	r.member( "Abstract", tAbstract, "" );
}

template<typename Reflector>
inline void mira::Class::reflectWrite(Reflector& r)
{
	r.member( "Identifier" , mIdentifier, "" );
	r.member( "Name" , mName, "" );
	r.member( "Lib" , mLib, "" );
	r.member( "Meta" , mMetaInfo, "" );
}

/**
 * @brief Auxiliary function which throws an XBadCast exception if the cast
 * fails and deletes the object.
 */
template <typename CLASS>
inline CLASS* mira_factoryDynamicCast( Object* base )
{
	CLASS* tClass = dynamic_cast<CLASS*> ( base );
	if ( !tClass ) {
		std::string tName = "NULL";
		if ( base ) {
			tName = base->getClass().getIdentifier();
		}
		delete base;
		MIRA_THROW( XBadCast, "You try to cast created object to incompatible"
		            " base class (" + tName + ")!");
	}
	return tClass;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_CLASS_H_ */

