/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <iostream>
#include <set>

#include <boost/algorithm/string.hpp>

#include <error/Exceptions.h>
#include <utils/ProgramOptions.h>
#include <platform/Process.h>
#include <platform/Console.h>

#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/map>
#include <utils/Path.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

#define SVNINDEXXML_FILENAME "svnindex.xml"

typedef std::map<std::string,std::string> Index;

Path rootPath;
std::string svnAuth;

///////////////////////////////////////////////////////////////////////////////

int reindex(const Path& rootPath);

void loadIndexEx(Index& index)
{
	XMLDom indexFile;
	indexFile.loadFromFile(rootPath / SVNINDEXXML_FILENAME);
	XMLDeserializer ds(indexFile);
	ds.deserialize("index", index);
}

void loadIndex(Index& index)
{
	try {
		loadIndexEx(index);
	} catch(XFileNotFound&) {
		std::cout << "Index file svnindex.xml does not exist. "
		          << "Trying to create the index now..." << std::endl;
		reindex(rootPath);

		// try again
		try {
			loadIndexEx(index);
		} catch(XFileNotFound&) {
			std::cout << "ERROR: failed to create the index in svnindex.xml. "
			          << "Aborting." << std::endl;
			throw;
		}
	}
}

void storeIndex(const Index& index)
{
	XMLDom indexFile;
	XMLSerializer s(indexFile);
	s.serialize("index", index);
	indexFile.saveToFile(rootPath / SVNINDEXXML_FILENAME);

	std::string stdOut;
	std::string errOut;
	if (executeProcess("svn propget svn:ignore "+rootPath.string(),
	                   &stdOut, &errOut) != -1)
	{
		// Check, whether svnindex.xml is already in svn:ignore
		if (stdOut.find("svnindex.xml") == std::string::npos) {
			boost::algorithm::trim(stdOut);
			stdOut += "\n";
			stdOut += "svnindex.xml";

			// Write content of ignore property to temporary file
			std::string ignoreFile = (rootPath / ".svnignore").string();
			std::fstream ignoreStrm;
			ignoreStrm.open(ignoreFile, std::fstream::out);
			ignoreStrm << stdOut;
			ignoreStrm.close();

			// Set svn:ignore property
			executeProcess("svn propset svn:ignore -F "+ignoreFile+" "+
			               rootPath.string(), &stdOut, &errOut);

			// Delete the temporary file
			boost::filesystem::remove(ignoreFile);
		}
	}
}

// recursively indexes the whole directory
void indexDirectory(const Path& path, Index& index, bool includeSelf=true)
{
	// process directories only
	if(!boost::filesystem::is_directory(path))
		return;

	// skip hidden directories
	if(path.filename().string()[0]=='.')
		return;

	if(path.filename().string()==".svn")
		return;

	// call svn info for this directory and get the info in xml
	std::string infoxmlStr;
	bool inWorkingCopy =
		executeProcess("svn info --xml " + svnAuth + path.string(), &infoxmlStr) == 0;

	// the above call will yield true, if we are in a working copy and false
	// if we are in some directory that svn knows anything about

	if(inWorkingCopy) {

		if(includeSelf) { // include ourselves into the index
			// parse the info xml stuff and add us to the index
			/*
			  XML format:
			  <info>
				<entry ...>
					<url>URL</url>
					...
				</entry>
			  </info>
			 */
			XMLDom infoxml;
			infoxml.loadFromString(infoxmlStr);
			// find entry tag
			XMLDom::iterator entryTag = std::find(infoxml.root().begin(),infoxml.root().end(),"entry");
			assert(entryTag!=infoxml.root().end());
			// find url tag
			XMLDom::iterator urlTag = std::find(entryTag.begin(),entryTag.end(),"url");
			assert(urlTag!=entryTag.end());

			// get url:
			std::string url = *urlTag.content_begin();

			// write the url and the path to the index
			index[path.string()] = url;

			console::setTextColor(console::GREEN);
			std::cout << "Added: " << path.string() << " [ " << url << " ]" << std::endl;
			console::setTextColor(console::DEFAULT);
		}

		////////////////////////////////////////////////////////////////////////
		// now recurse into all unversioned subdirs

		//std::cout << "STATUS: " << path << std::endl;
		// get svn status
		std::string status;
		executeProcess("svn st " + svnAuth + path.string(), &status);

		// get each status line
		std::list<std::string> lines;
		boost::split(lines, status, boost::is_any_of(std::string("\n")));

		// only use status lines of unversioned files / directories
		std::list<std::string> unversioned;
		foreach(const std::string& s, lines)
		{
			if(s.empty())
				continue;
			if(s[0]=='?')
				unversioned.push_back(s.substr(8));
		}

		// for each unversioned item
		foreach(const std::string& s, unversioned)
		{
			Path path(s);
			assert(path.is_complete());
			indexDirectory(path,index);
		}

	}
	// if we are not in a working copy, iterate through every subdir since we
	// may find another svn directory down there
	else {
		using namespace boost::filesystem;
		directory_iterator endit; // default construction yields past-the-end
		// iterate through all sub directories
		for(directory_iterator it(path); it!=endit; ++it)
			if(is_directory(it->status()))
				indexDirectory(it->path(),index);
	}
}

int reindex(const Path& rootPath)
{
	assert(rootPath.is_complete());
	console::setTextColor(console::LIGHTRED);
	std::cout << "Reindexing: " << rootPath.string() << std::endl;
	std::cout << "Please ignore any svn error messages below ..." << std::endl;
	console::setTextColor(console::DEFAULT);

	MIRA_CMDLINE.getDescriptions().add_options()
				("exclude-root","");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Index index;

	indexDirectory(rootPath, index, vmap.count("exclude-root")==0);

	storeIndex(index);
	console::setTextColor(console::GREEN);
	std::cout << "Reindexing finished successfully."<< std::endl;
	console::setTextColor(console::DEFAULT);
	return 0;
}


void svnExecute(const std::string& cmd, const std::string& args)
{
	if(cmd=="history") {
		std::cout << "Obsolete! Please use 'mirasvn changes' instead" << std::endl;
		return;
	}
	if(cmd=="changes") {
		executeProcess("svn " + svnAuth + " log -r BASE:HEAD " + args);
		return;
	}
	executeProcess("svn " + svnAuth + cmd + " " + args);
}

int svnCommand(const std::string& cmd)
{
	Index index;
	loadIndex(index);

	bool sequential = true;

	// for commit the default is non sequential
	if(cmd=="ci" || cmd=="commit")
		sequential = false;

	MIRA_CMDLINE.getDescriptions().add_options()
				("sequential","");
	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
    if(vmap.count("sequential")!=0)
    	sequential =true;

	if(sequential) { // sequential
		foreach(auto& v, index) {
			console::setTextColor(console::GREEN);
			std::cout << v.first << " [ " << v.second << " ]:" << std::endl;
			console::setTextColor(console::DEFAULT);
			svnExecute(cmd, v.first);
		}
	} else { // single step
		std::string dirs;
		foreach(auto& v, index) {
			console::setTextColor(console::GREEN);
			std::cout << v.first << " [ " << v.second << " ]:" << std::endl;
			console::setTextColor(console::DEFAULT);
			dirs += " " + v.first;
		}
		svnExecute(cmd, dirs);
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////////

void printUsage()
{
	std::cout <<
			"mirasvn <command> [directory] [options]\n\n"
			"Runs 'svn' in all indexed branches. All branches that are used are stored in the\n"
			"svnindex.xml file. This file can be generated automatically by calling 'mirasvn reindex'.\n"
			"If no directory is specified the current working directory is used to look for the\n"
			"svnindex.xml file\n"
			"\n"
			"General Options:\n"
			"  --auth: request username and password just ONCE and use it for all calls to svn.\n"
			"\n"
			"Available commands:\n"
			"    reindex     - rebuild the index of all branches and repositories\n"
			"\n"
			"                  Options:\n"
			"                    --exclude-root : exclude the specified root / current working directory\n"
			"\n"
			"    status (st) - status of all branches that are listed in the index\n"
			"    changes     - shows the changes on the server that were made since the last update\n"
			"    update (up) - update all branches that are listed in the index\n"
			"    commit (ci) - commit all changes (use with care!)\n"
			"\n"
			"                  Options:\n"
			"                    --sequential : perform commit sequentially for each branch\n"
			"                                   (instead of a single commit which is the default\n"
			"\n"
			"    all other svn commands are supported too, type 'svn help' for a complete list"
			<< std::endl;
}

///////////////////////////////////////////////////////////////////////////////

int main(int argc, char** argv)
{
	if(argc<2) {
		printUsage();
		return -1;
	}

	std::string cmd = argv[1];
	ProgramOptions options(argc, argv);

	rootPath = boost::filesystem::current_path();

	if(argc>2 && argv[2][0]!='-')
		rootPath = argv[2];

	rootPath = boost::filesystem::absolute(rootPath);

	MIRA_CMDLINE.getDescriptions().add_options()
				("auth","");
	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
	if(vmap.count("auth"))
	{
		std::string username, password;
		std::cout << "username: ";
		std::cin >> username;
		std::cout << "password: ";
		console::enableEcho(false);
		std::cin >> password;
		console::enableEcho(true);
		std::cout << std::endl;
		svnAuth = "--username \"" + username + "\" --password '" + password + "' ";
	} else {
		svnAuth = "";
	}

	try {
		if(cmd=="reindex") {
			return reindex(rootPath);
		} else {
			return svnCommand(cmd);
		}
	} catch(XFileNotFound&) {
		return -1;
	}

	return 0;
}
