/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ConfigurationEditor.C
 *    Allows to edit the current configuration and adding new units
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2011/12/08
 */

#include <serialization/Serialization.h>
#include <xml/XMLDomPreprocessor.h>

// include the view base class
#include <views/ConfigurationEditor.h>

// include Qt specific stuff
#include <QPushButton>

#include <widgets/SelectionListDialog.h>

#include <fw/MicroUnit.h>
#include <fw/UnitManager.h>


using namespace mira;

namespace mira { 

///////////////////////////////////////////////////////////////////////////////

ConfigurationEditor::ConfigurationEditor() : mWidget(NULL)
{

}

QWidget* ConfigurationEditor::createPartControl()
{
	mWidget = new QWidget(this);

	// TODO: add your content to mWidget here:
	QPushButton* bt = new QPushButton("Add Unit Config", mWidget);

	connect(bt,SIGNAL(clicked()), this, SLOT(addUnitConfig()));

	QHBoxLayout* hlayout = new QHBoxLayout(mWidget);
	hlayout->addWidget(bt);


	QVBoxLayout* vlayout = new QVBoxLayout;
	hlayout->addLayout(vlayout);

	mEdit = new QPlainTextEdit(mWidget);
	vlayout->addWidget(mEdit);

	vlayout->addWidget(new QLabel("Namespace:" , mWidget));
	mNamespace = new QLineEdit(mWidget);
	vlayout->addWidget(mNamespace);
	mNamespace->setText("/");

	bt = new QPushButton("Create Unit", mWidget);
	connect(bt,SIGNAL(clicked()), this, SLOT(createUnit()));
	vlayout->addWidget(bt);

	return mWidget;
}


void ConfigurationEditor::addUnitConfig()
{
	SelectionListDialog<ClassProxy> d(this);
	d.setWindowTitle(tr("Choose a unit"));
	d.setHeaderLabels(QStringList() << "Unit");
	d.resize(600,400);

	// query all view classes from the class factory:
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap unitClasses = MicroUnit::CLASS().getDerivedClasses( );

	foreach(ClassMap::value_type i, unitClasses)
	{
		if(i.second.isAbstract())
			continue;

		std::string name = i.first; // use class name

		/*
		std::string category = i.second.getMetaInfo("Category");
		if(category.empty())
			category = "General"; // use default category
		 */

		QStringList labels;
		//labels << name.c_str() << i.second.getMetaInfo("Description").c_str();
		labels << name.c_str();
		d.addItem(labels, i.second);
	}

	if(d.exec()==0)
		return;

	auto selection = d.selectedItems();
	if(selection.empty())
		return; // nothing chosen

	// get the parameters of the unit by creating it and deserializing it into xml
	const ClassProxy& unitClass = selection.begin()->second;
	MicroUnit* unit = unitClass.newInstance<MicroUnit>();
	XMLDom xml;
	XMLDom::iterator root = xml.root();
	root = "unit";
	root.add_attribute("id", unitClass.getName().c_str());
	XMLSerializer s(root);
	s.serialize("instance", unit, "");
	std::string content = xml.saveToString();
	mEdit->setPlainText(content.c_str());
	delete unit;

	if(mNamespace->text().isEmpty())
		mNamespace->setText("/");
}


void ConfigurationEditor::createUnit()
{
	std::string content = mEdit->toPlainText().toStdString();
	XMLDom xml;
	xml.loadFromString(content);
	XMLVariablesMap variables;
	preprocessXML(xml, variables);
	std::string name = xml.root().get_attribute<std::string>("id");
	XMLDeserializer d(xml.root());

	MicroUnit* unit = NULL;
	d.deserialize("instance", unit);
	MicroUnitPtr unitPtr(unit);
	std::string ns = mNamespace->text().toStdString();
	unit->checkin(ns, name);
	unit->start();
	MIRA_FW.getUnitManager()->addUnit(unitPtr, ns, name);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::ConfigurationEditor, mira::EditorPart );
