/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRACenter.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#include <QApplication>
#include <QFileDialog>
#include <QMdiArea>

#include <QDesktopServices>
#include <QUrl>

#include <error/LoggingAux.h>
#include <utils/PathFinder.h>
#include <factory/ManifestAgent.h>

#include <fw/FrameworkWithGui.h>
#include <fw/RemoteModule.h>

#include <app/MIRACenter.h>
#include <app/ErrorDialog.h>
#include <app/PreferencesDialog.h>

#include <ui_AddRemoteFrameworkDialog.h>

#include <ui_AboutDialog.h>
#include <ui_KeyboardShortcutsDialog.h>

#include <widgets/SelectionListDialog.h>
#include <widgets/QtUtils.h>

#include <app/private/EditorPartAction.h>
#include <app/private/CreateViewDialog.h>


// we are in a c file, so we can name the local macro without MIRA_ prefix
#define WORKSPACE_FILENAME "miracenter.workspace"

#if BOOST_VERSION < 105000
	namespace boost_filesystem_ns = boost::filesystem3;
#else
	namespace boost_filesystem_ns = boost::filesystem;
#endif

using namespace std;
using namespace mira;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

Path getDefaultWorkspace()
{
	Path defaultWorkspaceDir = WORKSPACE_FILENAME;
	try {
		defaultWorkspaceDir = getAppDataDirectory() / WORKSPACE_FILENAME;
	} catch(...) {}
	return defaultWorkspaceDir;
}

///////////////////////////////////////////////////////////////////////////////

class WorkspaceLoader :
	public ExplicitSingleton<WorkspaceLoader>,
	public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(WorkspaceLoader)

public:
	WorkspaceLoader()
	{
		workspace = getDefaultWorkspace();
	}

	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("workspace");
	}

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader)
	{
		workspace = node.get_attribute<std::string>("file");
	}

	Path workspace;
};

class InitialWorkspaceLoader :
	public ExplicitSingleton<InitialWorkspaceLoader>,
	public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(InitialWorkspaceLoader)

public:
	InitialWorkspaceLoader()
	{
		valid = false;
	}

	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("initial-workspace");
	}

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader)
	{
		workspace = node.get_attribute<std::string>("file");
		valid = true;
	}

	Path workspace;
	bool valid;
};

///////////////////////////////////////////////////////////////////////////////

MIRACenter::MIRACenter() :
	Workbench(false), // do not setup the GUI of the workbench immediately
	mAuthority("/","MIRACenter", Authority::ANONYMOUS | Authority::INTERNAL)
{
	mAutosaveTimer = -1;
	mStatusBarTimer = -1;
    mResourceTimer = -1;
}

void MIRACenter::setupGUI(boost::scoped_ptr<SplashScreen>& splash)
{
	Ui::MIRACenter::setupUi(this);

	//in qt5 this is not set automatically so we do it manually
	mMenuBar->setVisible(true);

	// FIX FOR Qt <4.6.1
#if (QT_VERSION < QT_VERSION_CHECK(4, 6, 1))
	// the following explicit call is necessary for some older Qt versions (<4.6.1)
	// where the uic compiler forgets to call this method for some reason (wtf?!)
	// http://bugreports.qt.nokia.com/browse/QTBUG-5824
	this->setCentralWidget(mCentralWidget);
#endif


	// set the application icon
	QPixmap pixmap(":/icons/Mira.png");
	QIcon icon(pixmap);
	setWindowIcon(icon);

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	if (vmap.count("no-statusbar") == 0) {
		QFrame* tStatusFrame = new QFrame();
		tStatusFrame->setFrameStyle(QFrame::Panel);
		tStatusFrame->setLineWidth(0);

		mLbTime = new QLabel(tStatusFrame);
		mLbTime->setFixedWidth(120);
		QFontMetrics tFM(mLbTime->font());
		mLbTime->setFixedWidth(tFM.width(tr("Time: 88:89:88.8"))+4);

		mLbMemoryUsage = new QLabel(tStatusFrame);
		mLbMemoryUsage->setFixedWidth(150);
		mLbMemoryUsage->setFixedWidth(tFM.width(tr("Memory: 888.8 MB"))+4);

		mLbCPUUsage = new QLabel(tStatusFrame);
		mLbCPUUsage->setFixedWidth(150);
		mLbCPUUsage->setFixedWidth(tFM.width(tr("CPU: 100.0 %"))+4);

		mLbNetworkIncomingBytes = new QLabel(tStatusFrame);
		mLbNetworkIncomingBytes->setFixedWidth(150);
		mLbNetworkIncomingBytes->setFixedWidth(tFM.width(tr("In: 100.000 kbytes"))+4);

		mLbNetworkOutgoingBytes = new QLabel(tStatusFrame);
		mLbNetworkOutgoingBytes->setFixedWidth(150);
		mLbNetworkOutgoingBytes->setFixedWidth(tFM.width(tr("Out: 100.000 kbytes"))+4);

		QHBoxLayout* tStatusLayout = new QHBoxLayout();
		tStatusLayout->addWidget(mLbTime);
		QFrame* tStatusVLine = new QFrame(tStatusFrame);
		tStatusVLine->setFrameStyle(QFrame::VLine | QFrame::Sunken);
		tStatusLayout->addWidget(tStatusVLine);
		tStatusLayout->addWidget(mLbMemoryUsage);
		tStatusVLine = new QFrame(tStatusFrame);
		tStatusVLine->setFrameStyle(QFrame::VLine | QFrame::Sunken);
		tStatusLayout->addWidget(tStatusVLine);
		tStatusLayout->addWidget(mLbCPUUsage);
		if (MIRA_FW.getRemoteModule())
		{
			tStatusVLine = new QFrame(tStatusFrame);
			tStatusVLine->setFrameStyle(QFrame::VLine | QFrame::Sunken);
			tStatusLayout->addWidget(tStatusVLine);
			tStatusLayout->addWidget(mLbNetworkIncomingBytes);
			tStatusLayout->addWidget(mLbNetworkOutgoingBytes);
		}
		tStatusLayout->setContentsMargins(4, 0, 4, 0);
		tStatusFrame->setLayout(tStatusLayout);

		statusBar()->addPermanentWidget(tStatusFrame);
		updateStatusBar();
	}

	Workbench::setupUi();

	// if workspace file specified via command line use it
	if (vmap.count("workspace") > 0) {
		mWorkspaceFilePath = resolvePath(vmap["workspace"].as<std::string>());
	} else {
		// will return the path found in xml or the default workspace file
		mWorkspaceFilePath = resolvePath(WorkspaceLoader::instance().workspace);
	}

	// this is the file we will save to later, no matter if we load another initially
	Path saveWorkspaceFilePath = mWorkspaceFilePath;

	if(vmap.count("initial-workspace") > 0) {
		mWorkspaceFilePath = resolvePath(vmap["initial-workspace"].as<std::string>());
	}  else {
		// check if user specified initial workspace in xml config file
		if (InitialWorkspaceLoader::instance().valid)
			mWorkspaceFilePath = resolvePath(InitialWorkspaceLoader::instance().workspace);
	}

	// "no-workspace" option has highest priority and overwrites any previous
	// setting of what workspace file to load.
	// Next priority is "default-workspace" and loading an existing workspace
	// has lowest priority.
	if (vmap.count("no-workspace") > 0)
		addPerspective(tr("Default").toLocal8Bit().data());
	else
	if (vmap.count("default-workspace") > 0) {
		MIRA_LOG(DEBUG) << "Creating default perspectives";
		populateDefaultPerspectives();
	} else {
		try {
			loadWorkspace(mWorkspaceFilePath.string());
		} catch(...) {
			// cannot load old workspace, so create a default one
			std::stringstream msg;

			// need to backup the file we will overwrite, not the one we tried to load!
			if (boost_filesystem_ns::exists(saveWorkspaceFilePath)) {
				std::string timestr = to_iso_string(Time::now());
				Path backup = saveWorkspaceFilePath.string() + ".bak"+timestr.substr(0,15);

				msg << "Cannot load workspace " << mWorkspaceFilePath << ". ";

				try {
					boost_filesystem_ns::copy(saveWorkspaceFilePath, backup);
					msg << "Created backup " << backup << " and generating default perspectives.";
				}
				catch (...)
				{
					msg << "Creating backup " << backup << " failed! Generating default perspectives.";
				}
			} else {
				msg << "Cannot load workspace " << mWorkspaceFilePath << ". Generating default perspectives.";
			}

			MIRA_LOG(WARNING) << msg.str();

			splash->close();
			QMessageBox::warning(this, windowTitle(), QString::fromStdString(msg.str()));

			populateDefaultPerspectives();
		}
	}

	mWorkspaceFilePath = saveWorkspaceFilePath;

	addAction(actionFullscreen);
	addAction(actionShowView);

	mMinimizeAllViews = new QAction(this);
	mMinimizeAllViews->setShortcut(Qt::Key_F10);
	mMinimizeAllViews->setShortcutContext(Qt::ApplicationShortcut);
	connect(mMinimizeAllViews, SIGNAL(triggered()), this, SLOT(minimizeAllViews()));
	addAction(mMinimizeAllViews);


	if (vmap.count("no-statusbar") == 0) {
		mCurrentCPUUsage = getCPUUsage();
		mCurrentMemoryUsage = getMemoryUsage();
		mCurrentIncomingData = 0;
		mCurrentOutgoingData = 0;
		mResourceTimer = startTimer(2000);
		mAutosaveTimer = -1;
		mStatusBarTimer = startTimer(100);
	}

	updateFromPreferences();

	statusBar()->showMessage(tr("MIRACenter started successfully."), 5000);

	if (vmap.count("minimized") > 0) {
		MIRA_LOG(DEBUG) << "minimizing miracenter main window";
		setWindowState(Qt::WindowMinimized);
	}
}

MIRACenter::~MIRACenter()
{
}

void MIRACenter::fileLoadConfiguration()
{
	QString name =
		QtUtils::getOpenFileName(this, tr("Open Configuration"), QString(),
		                         tr("Configuration (*.xml)"));
	if(name.isNull() || name.isEmpty())
		return;

	MIRA_FW.load(name.toLocal8Bit().data());

}

void MIRACenter::saveWorkspace()
{
	try {
		saveWorkspace(mWorkspaceFilePath.string());
	} catch(...) {} // shit happens, ignore it this time
}

void MIRACenter::saveWorkspace(const std::string& file)
{
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("MIRACenter", *this);
	xml.saveToFile(file);
}

void MIRACenter::loadWorkspace(const std::string& file)
{
	if(file.empty())
		return;

	try {
		XMLDom xml;
		xml.loadFromFile(file);
		XMLDeserializer s(xml);
		s.deserialize("MIRACenter", *this);

		// Reset all floating ViewParts / undocked EditorParts,
		// because restoring floating widgets is buggy right now.
		foreach(ViewPart* p, mViewParts) {
			p->setFloating(false);

			// This is necessary, otherwise the widget stays "gray"
			// and needs to be "recreated" to receive paint events again
			p->setUpdatesEnabled(true);
		}
		foreach(EditorPart* p, mEditorParts)
			p->dock();

		// Now we can set up the docking areas, which basically manage and show all the widgets
		updateFromPreferences();
	} catch(Exception& ex) {
		// catch all errors and clean up the dirt that might have been
		// created
		MIRA_LOG(WARNING) << "Failed to load workspace: " << ex.message() << ", cleaning up";
		cleanupWorkspace();
		throw; // rethrow
	}
	MIRA_LOG(DEBUG) << "Workspace successfully loaded";
}

void MIRACenter::fileAddRemoteFramework()
{
	Ui::AddRemoteFrameworkDialog dialog;
	QDialog* d = new QDialog(this);
	dialog.setupUi(d);

	foreach(const std::string& item, mRemoteFrameworkHistory)
		dialog.address->addItem(QString(item.c_str()));

	if(d->exec()==0)
		return;

	if(MIRA_FW.getRemoteModule()) {
		std::string address = dialog.address->currentText().toLocal8Bit().data();

		MIRA_FW.getRemoteModule()->addKnownFramework(address);

		auto it = std::find(mRemoteFrameworkHistory.begin(), mRemoteFrameworkHistory.end(), address);

		if(it==mRemoteFrameworkHistory.end()) // not yet in the list, create it at the beginning
			mRemoteFrameworkHistory.push_front(address);
		else // move chosen address to first position
			mRemoteFrameworkHistory.splice(mRemoteFrameworkHistory.begin(), mRemoteFrameworkHistory, it);
	}

}

void MIRACenter::fileClearWorkspace()
{
	cleanupWorkspace();
	// we need to create at least one empty perspective
	addPerspective(tr("Default").toLocal8Bit().data());
}

void MIRACenter::fileLoadWorkspace()
{
	QString name =
		QtUtils::getOpenFileName(this, tr("Open Workspace"), QString(),
		                         tr("Workspace (*.workspace)"));
	if(name.isNull() || name.isEmpty())
		return;

	loadWorkspace(name.toLocal8Bit().data());
}

void MIRACenter::fileLoadDefaultWorkspace()
{
	cleanupWorkspace();
	populateDefaultPerspectives();
}

void MIRACenter::fileSaveWorkspace()
{
	QString name = 
		QtUtils::getSaveFileName(this, tr("Save Workspace"), QString(),
		                         tr("Workspace (*.workspace)"));
	if(name.isNull() || name.isEmpty())
		return;
	Path fileName(name.toLocal8Bit().data());
	if (fileName.extension().empty())
		fileName.replace_extension(".workspace");
	saveWorkspace(fileName.string());
}

void MIRACenter::windowShowView()
{
	CreateViewDialog d(this);

	d.setWindowTitle(tr("Choose a view"));
	d.setHeaderLabels(QStringList() << "View" << "Description");
	d.resize(600,400);

	d.populateViewList();

	std::list<std::string> classes;
	foreach (const auto& p, mPartMemory)
		classes.push_back(p.first);
	d.setMemorizedClasses(classes);

	if(d.exec()==0)
		return;

	auto selection = d.selectedItems();
	if(selection.empty())
		return; // nothing chosen

	// create the selected part
	if (d.restoreIsChecked())
		restorePart(selection.begin()->second, *mPartMemory[selection.begin()->second.getIdentifier()]);
	else
		createPart(selection.begin()->second);
}

void MIRACenter::windowFullscreen(bool fullscreen)
{
	Qt::DockWidgetArea areas[4] = {Qt::LeftDockWidgetArea, Qt::RightDockWidgetArea,
	                               Qt::TopDockWidgetArea,Qt::BottomDockWidgetArea};

	if(fullscreen) {
		mBeforeFullscreenState.wasMaximized = isMaximized();

		mBeforeFullscreenState.restoreViews = Qt::NoDockWidgetArea;
		for(int i=0; i<4;++i) {
			if(!isDockAreaMinimized(areas[i])) {
				mBeforeFullscreenState.restoreViews |= areas[i];
				minimizeDockArea(areas[i]);
			}
		}

		this->menuBar()->hide();
		this->statusBar()->hide();
		showFullScreen();
	} else {
		this->menuBar()->show();
		this->statusBar()->show();

		// restore the view areas that were hidden when switching to fullscreen
		for(int i=0; i<4;++i) {
			if(mBeforeFullscreenState.restoreViews & areas[i]) {
				restoreDockArea(areas[i]);
			}
		}

		if(mBeforeFullscreenState.wasMaximized) {
			showNormal();
			showMaximized();
		} else
			showNormal();
	}
}

void MIRACenter::windowPreferences()
{
	PreferencesDialog d(this, mPreferences);
	if(d.exec()==QDialog::Rejected)
		return;

	mPreferences = d.getPreferences();
	updateFromPreferences();
}

void MIRACenter::windowEditorShow(EditorPart* editor)
{
	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
	assert(central!=NULL && "The central widget must be of type MdiCentralWidget!");
	central->activateSubWindow(editor);
	central->scrollToSubWindow(editor);
}

void MIRACenter::windowEditorClose(EditorPart* editor)
{
	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
	assert(central!=NULL && "The central widget must be of type MdiCentralWidget!");
	central->removeSubWindow(editor);
	delete editor;
}

void MIRACenter::helpContent()
{
	try {
		// check for local version of the help
		Path p = findProjectFile("doc/MIRACenterPage.html");
		QString url = QString("file://")+p.string().c_str();
		QDesktopServices::openUrl(QUrl(url));
	} catch(...) {
		// fallback: open online version
		QDesktopServices::openUrl(QUrl("http://www.mira-project.org/MIRA-doc/MIRACenterPage.html"));
	}
}

void MIRACenter::helpIndex()
{
	try {
		// check for local version of the help
		Path p = findProjectFile("doc/index.html");
		QString url = QString("file://")+p.string().c_str();
		QDesktopServices::openUrl(QUrl(url));
	} catch(...) {
		// fallback: open online version
		QDesktopServices::openUrl(QUrl("http://www.mira-project.org/MIRA-doc/index.html"));
	}
}

void MIRACenter::helpKeyboardShortcuts()
{
	QDialog dialog(this);
	Ui::KeyboardShortcutsDialog d;
	d.setupUi(&dialog);
	d.mTreeWidget->expandAll();
	d.mTreeWidget->resizeColumnToContents(0);
	d.mTreeWidget->resizeColumnToContents(1);
	dialog.exec();
}

void MIRACenter::helpAbout()
{
	QDialog dialog(this);
	Ui::AboutDialog d;
	d.setupUi(&dialog);
	dialog.exec();
}

void MIRACenter::minimizeAllViews()
{
	Qt::DockWidgetArea areas[4] = {Qt::LeftDockWidgetArea, Qt::RightDockWidgetArea,
		                           Qt::TopDockWidgetArea,Qt::BottomDockWidgetArea};

	for(int i=0; i<4;++i)
		minimizeDockArea(areas[i]);
}

void MIRACenter::onEditorCreated(EditorPart* editor)
{
	Workbench::onEditorCreated(editor);

	// if this is the first editor, add a separator to the menu first
	if (mEditorParts.size() == 1)
		mMenuView->addSeparator();

	QMenu* menu = mMenuView->addMenu("* " + editor->windowTitle());

	EditorPartAction* actionShow = new EditorPartAction("Show", menu, editor);
	connect(actionShow, SIGNAL(selected(EditorPart*)),
	        this, SLOT(windowEditorShow(EditorPart*)));
	menu->addAction(actionShow);

	EditorPartAction* actionClose = new EditorPartAction("Close", menu, editor);
	connect(actionClose, SIGNAL(selected(EditorPart*)),
	        this, SLOT(windowEditorClose(EditorPart*)));
	menu->addAction(actionClose);

}

void MIRACenter::onEditorDestroyed(EditorPart* editor)
{
	// find and remove the respective menu entry
	foreach (QAction* a, mMenuView->actions()) {
		if (!a->menu() || a->menu()->actions().isEmpty())
			continue;

		EditorPartAction* e = dynamic_cast<EditorPartAction*>(a->menu()->actions().first());
		if (!e)
			continue;

		if (e->editor() == editor) {
			mMenuView->removeAction(a);
			break;
		}
	}

	// if this is the last editor, remove the separator from the menu too
	if (mEditorParts.size() == 1)
		mMenuView->removeAction(mMenuView->actions().last());

	Workbench::onEditorDestroyed(editor);
}

void MIRACenter::onWindowTitleChanged(EditorPartWindow* window)
{
	Workbench::onWindowTitleChanged(window);

	foreach (QAction* a, mMenuView->actions()) {
		if (!a->menu() || a->menu()->actions().isEmpty())
			continue;

		EditorPartAction* e = dynamic_cast<EditorPartAction*>(a->menu()->actions().first());
		if (!e)
			continue;

		if ((EditorPartWindow*)(e->editor()) == window) {
			a->setText("* " + e->editor()->windowTitle());
			break;
		}
	}
}

void MIRACenter::onEditorWindowActivated(EditorPartWindow* window)
{
	Workbench::onEditorWindowActivated(window);

	foreach (QAction* a, mMenuView->actions()) {
		if (!a->menu() || a->menu()->actions().isEmpty())
			continue;

		EditorPartAction* e = dynamic_cast<EditorPartAction*>(a->menu()->actions().first());
		if (!e)
			continue;

		QString mark("   ");
		if ((EditorPartWindow*)(e->editor()) == window)
			mark = "* ";

		a->setText(mark + e->editor()->windowTitle());
	}
}

///////////////////////////////////////////////////////////////////////////////

void MIRACenter::timerEvent(QTimerEvent *event)
{
	if (event->timerId() == mResourceTimer)
	{
		mCurrentMemoryUsage = getMemoryUsage();
		mCurrentCPUUsage = getCPUUsage();
		if (MIRA_FW.getRemoteModule())
		{
			mCurrentIncomingData = MIRA_FW.getRemoteModule()->getIncomingBytesPerSecond();
			mCurrentOutgoingData = MIRA_FW.getRemoteModule()->getOutgoingBytesPerSecond();
		}
	}
	else if(event->timerId() == mAutosaveTimer) // Autosave
		saveWorkspace();
	else if(event->timerId() == mStatusBarTimer)
		updateStatusBar();
}

void MIRACenter::closeEvent(QCloseEvent* event)
{
	if (mPreferences.autosaveWorkspace)
		saveWorkspace();
	qApp->quit();
	//Workbench::closeEvent(event);
}

QWidget* MIRACenter::createDefaultPart(const std::string& className)
{
	try {
		return createPart(ClassFactory::instance().getClassByIdentifier(className));
	}
	catch (...)	{
		// ignore all errors
	}
	return NULL;
}

void MIRACenter::placeViewPart(QWidget* part, Qt::DockWidgetArea area)
{
	ViewPart* viewPart = dynamic_cast<ViewPart*>(part);
	if(viewPart!=NULL)
	{
		addDockWidget(area,viewPart);
		//viewPart->setEnabled(false); // why should we set the widget disabled??
	}
}

void MIRACenter::placeViewParts(QWidget* part1, QWidget* part2,
                               Qt::DockWidgetArea area,
                               Qt::Orientation orientation)
{
	ViewPart* viewPart1 = dynamic_cast<ViewPart*>(part1);
	if(viewPart1==NULL)
		return;

	ViewPart* viewPart2 = dynamic_cast<ViewPart*>(part2);
	if(viewPart2==NULL)
		return;

	placeViewPart(part1,area); // make sure the first part is placed in the correct area

	splitDockWidget(viewPart1, viewPart2, orientation);
}

void MIRACenter::placeViewPartTabbed(QWidget* placed, QWidget* part)
{
	ViewPart* placedPart = dynamic_cast<ViewPart*>(placed);
	if(placedPart==NULL)
		return;

	ViewPart* viewPart = dynamic_cast<ViewPart*>(part);
	if(viewPart==NULL)
		return;

	placeViewPart(part, dockWidgetArea(placedPart));

	tabifyDockWidget(placedPart, viewPart);
	placedPart->raise();
}

void MIRACenter::updateFromPreferences()
{
	mCentralWidget->setAutoHideDecorations(mPreferences.hideEditorWindowDecorations);
	updateDockCorners();

	if(mAutosaveTimer>=0)
		killTimer(mAutosaveTimer);

	if(mPreferences.autosaveWorkspace) {
		saveWorkspace();
		mAutosaveTimer = startTimer(mPreferences.autosaveWorkspaceInterval*1000);
	}

}

void MIRACenter::updateDockCorners()
{
	const Preferences& p = mPreferences;

	setCorner(Qt::TopLeftCorner,     p.corners[Preferences::TopLeft]     ? Qt::LeftDockWidgetArea : Qt::TopDockWidgetArea);
	setCorner(Qt::TopRightCorner,    p.corners[Preferences::TopRight]    ? Qt::RightDockWidgetArea: Qt::TopDockWidgetArea);
	setCorner(Qt::BottomLeftCorner,  p.corners[Preferences::BottomLeft]  ? Qt::LeftDockWidgetArea : Qt::BottomDockWidgetArea);
	setCorner(Qt::BottomRightCorner, p.corners[Preferences::BottomRight] ? Qt::RightDockWidgetArea: Qt::BottomDockWidgetArea);
}

QString memoryToString(uint64 mem)
{
	if (mem < 1000)
		return QString("%1 kB").arg(mem);
	if (mem < 1000000)
		return QString("%1 MB").arg(mem/1000.0f, 0, 'f', 1);
	return QString("%1 GB").arg(mem/1000000.0f, 0, 'f', 1);
}

void MIRACenter::updateStatusBar()
{
	Duration tNow = Time::now().toLocal().time_of_day();

	mLbTime->setText(QString(tr("Time: %1:%2:%3.%4")).
		arg(tNow.hours(), 2, 10, QChar('0')).
		arg(tNow.minutes(), 2, 10, QChar('0')).
		arg(tNow.seconds(), 2, 10, QChar('0')).
		arg(tNow.milliseconds()/100));

#ifdef MIRA_LINUX
	mLbMemoryUsage->setText(QString(tr("Memory: %1")).
	                           arg(memoryToString(mCurrentMemoryUsage.totalHeap+mCurrentMemoryUsage.totalStack)));
#else
	mLbMemoryUsage->setText(QString(tr("Memory: %1")).
	                           arg(memoryToString(mCurrentMemoryUsage.residentSetSize)));
#endif


	mLbCPUUsage->setText(QString(tr("CPU: %1 %")).
	                     arg(mCurrentCPUUsage, 0, 'f', 1));

	if (mCurrentIncomingData < 1024*1024)
		mLbNetworkIncomingBytes->setText(QString("In: %1 KBytes").arg(mCurrentIncomingData/1024.0, 0, 'f',3));
	else
		mLbNetworkIncomingBytes->setText(QString("In: %1 MBytes").arg(mCurrentIncomingData/1024.0/1024.0, 0, 'f',3));
	if (mCurrentOutgoingData < 1024*1024)
		mLbNetworkOutgoingBytes->setText(QString("Out: %1 KBytes").arg(mCurrentOutgoingData/1024.0, 0, 'f',3));
	else
		mLbNetworkOutgoingBytes->setText(QString("Out: %1 MBytes").arg(mCurrentOutgoingData/1024.0/1024.0, 0, 'f',3));
}

void MIRACenter::populateDefaultPerspectives()
{
	// DEFAULT "Development" perspective
	Perspective* p = addPerspective(tr("Development").toLocal8Bit().data());

	placeViewParts(
		createDefaultPart("mira::ChannelView"),
		createDefaultPart("mira::VisualizationControl"),
		Qt::LeftDockWidgetArea, Qt::Vertical);

	placeViewParts(
		createDefaultPart("mira::AuthorityView"),
		createDefaultPart("mira::PropertyView"),
		Qt::RightDockWidgetArea, Qt::Vertical);

	QWidget* bottomDock = createDefaultPart("mira::LogView");
	placeViewPart(bottomDock, Qt::BottomDockWidgetArea);
	placeViewPartTabbed(bottomDock, createDefaultPart("mira::RPCConsole"));
	placeViewPartTabbed(bottomDock, createDefaultPart("mira::TapePlayerView"));
	placeViewPartTabbed(bottomDock, createDefaultPart("mira::TapeRecorderView"));

	activatePerspective(p);
}

///////////////////////////////////////////////////////////////////////////////

}

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_REGISTER(WorkspaceLoader, mira::ConfigurationLoaderPlugin)
MIRA_CLASS_REGISTER(InitialWorkspaceLoader, mira::ConfigurationLoaderPlugin)

class MIRACenterApplication: public QApplication
{
public:

	MIRACenterApplication(int& argc, char** argv): QApplication(argc,argv)
	{}

	bool notify(QObject *rec, QEvent *ev)
	{
		try {
			return QApplication::notify(rec, ev);
		}
		catch (std::exception& ex) {
			MIRA_LOG_EXCEPTION(ERROR, ex) << "Unhandled exception:\n";
			if(ErrorDialog::showErrorDialog(NULL, ex)==ErrorDialog::Abort)
				abort();
		}
		catch (...) {
			MIRA_LOG(ERROR) << "Unhandled exception: <unknown exception>";
			if(ErrorDialog::showErrorDialog(NULL, "<unknown exception>")==ErrorDialog::Abort)
				abort();
		}
		return true;
	}
};

///////////////////////////////////////////////////////////////////////////////

int main(int argc, char** argv)
{
	MIRACenterApplication app(argc,argv);

	bool noSplash = false;
	for(int i=1; i<argc; ++i)
	{
		if(std::string(argv[i])=="--no-splash" ||
				std::string(argv[i])=="--help" ||
				std::string(argv[i])=="-?") {
			noSplash = true;
			break;
		}
	}

	QPixmap pixmap(":/images/Splash.png");

	boost::scoped_ptr<SplashScreen> splash;

	if(noSplash) {
		splash.reset(new SplashScreen(SplashScreen::DUMMY));
	} else {
		splash.reset(new SplashScreen(NULL, pixmap));
		splash->show();
		splash->setTextRect(QRect(10,10,splash->width()-20,20));
	}

	int res=0;
	try {
		splash->showMessage("Initializing framework ...");
		FrameworkWithGui fw(argc, argv, &app);

		// add our own command line parameters here
		MIRA_CMDLINE.getDescriptions().add_options()
			("no-workspace", "This option starts miracenter with a clean and empty "
			 "workspace (no default or last used workspace is created).")
			("default-workspace", "This option starts miracenter with the default "
			 "standard workspace.")
			("workspace", boost::program_options::value<std::string>(),
			 "The workspace to load (if not --no-workspace is set). "
			 "Overwrites any workspace setting given in config file.")
			("initial-workspace", boost::program_options::value<std::string>(),
			 "The workspace file loaded to initially populate the workspace. "
			 "Subsequent changes are not saved to this file, but rather to the default file.")
			("minimized", "This option starts miracenter as a minimized window.")
			("fontsize", boost::program_options::value<int>(),
			 "Sets the font size to use (e.g. for small robot screens: 10)")
			("no-splash", "Disables the start-up splash screen")
			("no-statusbar", "Disable status bar updates for cpu and memory usage.");

		splash->showMessage("Loading configuration and instantiating units ...");
		fw.load();

		ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
		if ( vmap.count("help") ) {
			std::cout << "Usage:" << std::endl << std::endl;
			std::cout << "    miracenter [options] [config file1] [config file2] ... [config fileN]" << std::endl << std::endl;
			std::cout << " Starts a MIRA process with a graphical frontend for controlling, inspection and debugging." <<  std::endl;
			std::cout << " Each configuration file can be specified as follows:" << std::endl;
			std::cout << "    <path>           : Relative or absolute path (mypath/myconfig.xml)" << std::endl;
			std::cout << "    <package>:<path> : Path relative to a package (mypackage:etc/myconfig.xml)" << std::endl;
			std::cout << "    ${find <path>}   : Searches for the specified file in all mira paths recursively\n"
					     "                                                       (\"\\${find somefile.xml}\")" << std::endl<< std::endl;

			std::cout << " Options:" << std::endl << std::endl;
			std::cout << MIRA_CMDLINE.getDescriptions();
			return 0;
		}

		MIRACenter center;

		if( vmap.count("fontsize")) {
			int fs = vmap["fontsize"].as<int>();
			QFont font = app.font();
			font.setPixelSize(fs);
			app.setFont(font);
		}

		splash->showMessage("Starting up framework ...");
		fw.start();

		splash->showMessage("Preparing user interface ...");
		center.setupGUI(splash);
		center.show();

		splash->showMessage("MIRACenter is up and running.");
		splash->closeOnClick();
		splash->closeTimer(500);

		res = fw.exec();
	}
	catch(std::exception& ex) {
		splash->close();
		MIRA_LOG_EXCEPTION(ERROR, ex) << "An exception has occured:\n";
		ErrorDialog::showErrorDialog(NULL, ex, ErrorDialog::Abort);
		return -1;
	}

	MIRA_LOG(NOTICE) << "Exiting application";
	return res;
}
