/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ErrorDialog.C
 *
 *
 * @author Erik Einhorn
 * @date   2011/11/30
 */

#include <app/ErrorDialog.h>

#include <utils/MakeString.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ErrorDialog::ErrorDialog(QWidget *parent, const QString& message,
        				const QString& details, Buttons buttons) : QDialog(parent),
        				mDetails(false)
{
	setupUi(this);

	mLbMessage->setText(message);
	if(!(buttons & Ignore))
		mBtIgnore->hide();

	if(!(buttons & Abort))
		mBtAbort->hide();

	if(details.isEmpty())
		mBtDetails->hide();
	else {
		mTeDetails->setPlainText(details);
	}

	QStyle *style = QApplication::style();
	int iconSize = style->pixelMetric(QStyle::PM_MessageBoxIconSize, 0);
	QIcon icon;
	icon = style->standardIcon(QStyle::SP_MessageBoxCritical, 0);

	if (!icon.isNull()) {
		QPixmap pm = icon.pixmap(iconSize, iconSize);
		mLbIcon->setPixmap(pm);
		mLbIcon->setFixedWidth(pm.width()+2);
	}

	QFont font("Monospace");
	font.setStyleHint(QFont::TypeWriter);
	mTeDetails->setFont(font);

	mDetailsWidget->hide();

	setMinimumHeight(100);
	resize(width(), 0);


}

void ErrorDialog::toggleDetails()
{
	mDetails = !mDetails;

	if(mDetails) {
		mDetailsWidget->show();
		resize(width(), mUpperWidget->minimumHeight()+400);
	} else {
		mDetailsWidget->hide();
		QApplication::processEvents();
		resize(width(), 0);
	}
}


ErrorDialog::Buttons ErrorDialog::showErrorDialog(QWidget *parent, const QString& message,
                                                  const QString& details, Buttons buttons)
{
	ErrorDialog dialog(parent, message, details, buttons);
	if(dialog.exec()==QDialog::Accepted)
		return Abort;
	else
		return Ignore;
}


ErrorDialog::Buttons ErrorDialog::showErrorDialog(QWidget *parent, const Exception& ex,
		                                          Buttons buttons)
{
	std::string cs = MakeString()
			<< typeName(ex) << ":\n"
			<< ex.what() << "\n\n" << "[CALLSTACK]:\n" << ex.callStack();
	return showErrorDialog(parent, QString::fromStdString(ex.message()),
	                               QString::fromStdString(cs), buttons);
}

ErrorDialog::Buttons ErrorDialog::showErrorDialog(QWidget *parent, const std::exception& ex,
                                                 Buttons buttons)
{
	try {
		const Exception& e = dynamic_cast<const Exception&>(ex);
		return showErrorDialog(parent, e, buttons);
	} catch(...) {
		return showErrorDialog(parent, ex.what(), QString::fromStdString(typeName(ex)), buttons);
	}

	// we never reach here
	return NoButton;
}





///////////////////////////////////////////////////////////////////////////////

} // namespace

