/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RectVisualization.C
 *    Visualization for rects
 *
 * @author Tim Langner
 * @date   2012/10/08
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/RectObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/**
 * A rect visualization.
 */
class Rect2fVisualization3D : public Visualization3DBasic<Rect2f>
{
	MIRA_META_OBJECT(Rect2fVisualization3D,
			("Name", "Rect2f")
			("Description", "Visualizes a 2D float rect in the xy-plane")
			("Category", "Geometry"))
public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	Rect2fVisualization3D() :
		mRect(NULL),
		mColor(Ogre::ColourValue::Black)
	{
	}

	/// The destructor.
	virtual ~Rect2fVisualization3D() {
		delete mRect;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Visualization3DBasic<Rect2f>::reflect(r);
		r.property("Color", mColor,
		           setter<Ogre::ColourValue>(&Rect2fVisualization3D::setColor, this),
		           "The color of the grid", Ogre::ColourValue::Black);
	}

	//@}

public:
	/** @name Setters */
	//@{

	/// Set the color of the grid.
	void setColor(const Ogre::ColourValue& c)
	{
		mColor = c;
		if (mRect != NULL)
			mRect->setColor(mColor);
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mRect = new RectObject(mgr, node);
		mRect->setColor(mColor);
		mRect->setVisible(this->isEnabled());
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3DBasic<Rect2f>::setEnabled(enabled);
		if (mRect != NULL)
			mRect->setVisible(enabled);
	}

	void dataChanged(ChannelRead<Rect2f> rect)
	{
		mRect->setRect(rect->value());
	}

	//@}

private:
	RectObject* mRect;
	Ogre::ColourValue mColor;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace


MIRA_CLASS_SERIALIZATION(mira::gui::Rect2fVisualization3D,
                         mira::Visualization3D);
