/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Rect2DVectorVisualization.C
 *    A visualization for a vector of 2D rects.
 *
 * @author Tim Langner, Christian Martin
 * @date   2011/03/04
 */

#include <QGraphicsRectItem>
#include <QGraphicsEllipseItem>

#include <serialization/adapters/Qt/QColor>

#include <geometry/Rect.h>

#include <visualization/Visualization2DBasic.h>

namespace mira { namespace gui {

//////////////////////////////////////////////////////////////////////////////

/// A visualization for a vector of 2D rects.
template<typename ItemType, typename DataType>
class Rect2DVectorVisualizationBase :
	public Visualization2DBasic<std::vector<DataType>>
{
public:
	typedef Visualization2DBasic<std::vector<DataType>> Base;

	/// Fill style mode.
	enum FillMode
	{
		UNFILLED = 0,   ///< Unfilled mode.
		FILLED_SOLID    ///< Solid fill mode.
	};

public:
	/** @name Constructor, destructor and reflect */
	//@{

	// The constructor.
	Rect2DVectorVisualizationBase() :
		mItemGroup(NULL),
		mColor(Qt::black),
		mLineWidth(1.0f),
		mFillMode(UNFILLED),
		mFillColor(Qt::yellow)
	{}

	/// The destructor.
	virtual ~Rect2DVectorVisualizationBase()
	{
		// Since destroyItemGroup will NOT delete the group item, we to do it!
		while(mItems.size() > 0)
		{
			mItemGroup->removeFromGroup(*mItems.begin());
			delete(*mItems.begin());
			mItems.pop_front();
		}

		// Since it is not sufficient to destroy mItemGroup (by calling delete
		// in Visualization2DBasic), we have to destroy the group by calling
		// destroyItemGroup and set the protected member of the base class
		// to NULL.
		if (mItemGroup && this->getSite() && this->getSite()->getSceneManager())
			this->getSite()->getSceneManager()->destroyItemGroup(mItemGroup);

		// prevent base class from deleting mItem again
		this->mItem = NULL;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("Color", mColor,
		           setter<QColor>(&Rect2DVectorVisualizationBase::setColor, this),
		           "The color.", Qt::black);
		r.property("LineWidth", mLineWidth,
		           setter<float>(&Rect2DVectorVisualizationBase::setLineWidth, this),
		           "The line width.", 1.0f);
		r.property("FillMode", mFillMode,
		           setter<FillMode>(&Rect2DVectorVisualizationBase::setFillMode, this),
		          "The fill mode.", UNFILLED,
		           mira::PropertyHints::enumeration("Unfilled;Solid"));
		r.property("FillColor", mFillColor,
		           setter<QColor>(&Rect2DVectorVisualizationBase::setFillColor, this),
		           "The fill color.", Qt::yellow);
	}

	//@}

public:
	/** @name Some setters */
	//@{

	/**
	 * @brief Set the color of the rect visualization
	 * @param[in] color The new color.
	 */
	void setColor(const QColor& color)
	{
		mColor = color;
		updatePen();
	}

	/**
	 * @brief Set the line width of the rect visualization
	 * @param[in] lineWidth The new line width.
	 */
	void setLineWidth(float lineWidth)
	{
		mLineWidth = lineWidth;
		updatePen();
	}

	/**
	 * @brief Set the fill style of the visualization.
	 * @param[in] fillMode The fill mode.
	 */
	void setFillMode(FillMode fillMode)
	{
		mFillMode = fillMode;
		updatePen();
	}

	/**
	 * @brief Set the fill color of the rect visualization
	 * @param[in] color The new color.
	 */
	void setFillColor(const QColor& color)
	{
		mFillColor = color;
		updatePen();
	}

	//@}

public:
	/** @name Public implementation of Visualization2DBasic */
	//@

	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr)
	{
		mItemGroup = mgr->createItemGroup(QList<QGraphicsItem*>());
		return mItemGroup;
	}

	//@}

protected:
	/** @name Protected implementation of Visualization2DBasic */
	//@{

	virtual void dataChanged(ChannelRead<std::vector<DataType>> data)
	{
		if (mItemGroup)
		{
			while(mItems.size() < data->size())
			{
				ItemType* i = new ItemType();
				mItems.push_back(i);
				mItemGroup->addToGroup(i);
			}
			while(mItems.size() > data->size())
			{
				mItemGroup->removeFromGroup(*mItems.begin());
				delete(*mItems.begin());
				mItems.pop_front();
			}

			auto r = mItems.begin();
			for(size_t i = 0; i < data->size(); ++i, ++r)
			{
				const DataType& value = data->value()[i];
				auto size = value.size();
				(*r)->setRect((qreal)value.minCorner.x(),
				              (qreal)value.minCorner.y(),
				              (qreal)size.width(), (qreal)size.height());
			}
			updatePen();
		}
	}

	//@}

private:
	void updatePen()
	{
		QPen pen(mColor, mLineWidth);

		QBrush brush;
		if (mFillMode == UNFILLED)
			brush = QBrush(mFillColor, Qt::NoBrush);
		else
		if (mFillMode == FILLED_SOLID)
			brush = QBrush(mFillColor, Qt::SolidPattern);

		auto r = mItems.begin();
		for(size_t i = 0; i < mItems.size(); ++i, ++r)
		{
			(*r)->setPen(pen);
			(*r)->setBrush(brush);
		}
	}

protected:
	QGraphicsItemGroup* mItemGroup;
	std::list<ItemType*> mItems;
	QColor mColor;
	float mLineWidth;
	FillMode mFillMode;
	QColor mFillColor;
};

//////////////////////////////////////////////////////////////////////////////

class Rect2iVectorVisualization2D :
	public Rect2DVectorVisualizationBase<QGraphicsRectItem, Rect2i>
{
	MIRA_META_OBJECT(Rect2iVectorVisualization2D,
		("Name", "Vector of rects")
		("Description", "Visualization of a vector of 2D int rectangles")
		("Category", "Geometry"))
};

class Rect2fVectorVisualization2D :
	public Rect2DVectorVisualizationBase<QGraphicsRectItem, Rect2f>
{
	MIRA_META_OBJECT(Rect2fVectorVisualization2D,
		("Name", "Vector of rects")
		("Description", "Visualization of a vector of 2D float rectangles")
		("Category", "Geometry"))
};

class Ellipse2iVectorVisualization2D :
	public Rect2DVectorVisualizationBase<QGraphicsEllipseItem, Rect2i>
{
	MIRA_META_OBJECT(Ellipse2iVectorVisualization2D,
		("Name", "Vector of ellipses")
		("Description", "Visualization of a vector of 2D int rectangles as ellipses")
		("Category", "Geometry"))
};

class Ellipse2fVectorVisualization2D :
	public Rect2DVectorVisualizationBase<QGraphicsEllipseItem, Rect2f>
{
	MIRA_META_OBJECT(Ellipse2fVectorVisualization2D,
		("Name", "Vector of ellipses")
		("Description", "Visualization of a vector of 2D float rectangles as ellipses")
		("Category", "Geometry"))
};

//////////////////////////////////////////////////////////////////////////////

}}

//////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::gui::Rect2iVectorVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2fVectorVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Ellipse2iVectorVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Ellipse2fVectorVisualization2D,
                         mira::Visualization2D);
