/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Z11Codec.C
 *    Implementation of Z11Codec.h.
 *
 * @author Erik Einhorn
 * @date   2011/10/01
 */

#include <codec/Z11Codec.h>
#include <opencv/highgui.h>

#include <serialization/Serialization.h>

#include <utils/TypedVoidPtr.h>

namespace mira { namespace codec {

///////////////////////////////////////////////////////////////////////////////

/*
 Encryption table:

 Differences are coded as follows:

 Bits       Difference (or [] sequences of differences)

        00  [0,0]  | 0

        01  -1     | A
        10   1     |

      0011   0     |
      0111  -2     | B
      1011   2     |

 0000 1111   3     |
 0010 1111  -3     |
 0100 1111   4     |
 0110 1111  -4     | C
 1000 1111   5     |
 1010 1111  -5     |
 1100 1111   6     |
 1110 1111  -6     |

 */

inline void encodeLine(const uint16* input, const uint16* inputEnd,
                       BinaryBufferOstream& stream)
{
	uint32 hold=0;
	uint32 bits=0;

	#define FLUSH                              \
		if(bits>=16) {                         \
			bits-=16;                          \
			stream << uint16(hold & 0xFFFF);   \
			hold>>=16;                         \
		}

	#define WRITE_A(value) {     \
		hold |= (value << bits); \
		bits += 2;               \
		FLUSH                    \
	}

	#define WRITE_B(value) {     \
		hold |= (value << bits); \
		bits += 4;               \
		FLUSH                    \
	}

	#define WRITE_C(absvalue, signbit) {                                \
		uint32 value = 0x0F | ((((absvalue-3) << 1) | (signbit)) << 5); \
		hold |= (value << bits);                                        \
		bits += 8;                                                      \
		FLUSH                                                           \
	}

	#define WRITE_D(value) {                        \
		uint32 val = 0x1F | ((value & 0x7FF) << 5); \
		hold |= (val << bits);                      \
		bits += 16;                                 \
		FLUSH                                       \
	}


	// write first value directly
	uint16 lastValue = *input;
	stream << lastValue;
	++input;
	bool lastDiffWasZero=false;

	for(;input<inputEnd;++input)
	{
		uint16 value = *input;
		int16  diff = (int16)value-lastValue;
		uint16 absdiff = std::abs(diff);

		if(lastDiffWasZero) {
			lastDiffWasZero = false;

			// [0,0] sequence
			if(diff==0) {
				WRITE_A(0x0)
				goto next;
			}

			// write last single zero
			WRITE_B(0x3)
		}

		if(diff==0)
			lastDiffWasZero = true;
		else if(diff==-1)
			WRITE_A(0x1)
		else if(diff==1)
			WRITE_A(0x2)
		else if(diff==-2)
			WRITE_B(0x7)
		else if(diff==2)
			WRITE_B(0xB)
		else if(absdiff>=3 && absdiff<=6)
			WRITE_C(absdiff, diff<0 ? 1 : 0)
		else
			WRITE_D(value)

	next:
		lastValue = value;
	}

	if(lastDiffWasZero)	// write last single zero
		WRITE_B(0x3)

	// write remaining bits
	if(bits>0) {
		assert(bits<16);
		stream << uint16(hold & 0xFFFF);
	}
}

inline void decodeLine(uint16* output, const uint16* outputEnd,
                       BinaryBufferIstream& stream)
{
	uint32 hold=0;
	uint32 bits=0;

	#define PULL {                 \
		uint16 v;                  \
		stream >> v;               \
		hold |= (uint32)v << bits; \
		bits+=16;                  \
	}

	#define NEEDBITS(n) { \
		if(bits < n)      \
            PULL          \
	}

	#define DROPBITS(n) { \
		hold >>= n;       \
		bits-=n;          \
	}

	#define BITS2 (hold & 0x3)
	#define BITS4 (hold & 0xF)
	#define BITS8 (hold & 0xFF)

	#define WRITE_OUTPUT(value) {   \
		assert(output < outputEnd); \
		*output = value; ++output;  \
	}


	// read first value directly
	uint16 lastValue;
	stream >> lastValue;
	WRITE_OUTPUT(lastValue)

	while(output < outputEnd)
	{
		NEEDBITS(2)
		if(BITS2 != 0x3) { // 0, A
			if(BITS2==0x0) {
				WRITE_OUTPUT(lastValue)
				WRITE_OUTPUT(lastValue)
			} else if(BITS2 == 0x1) {
				lastValue--;
				WRITE_OUTPUT(lastValue)
			} else {
				lastValue++;
				WRITE_OUTPUT(lastValue)
			}
			DROPBITS(2)
		} else {
			NEEDBITS(4)
			if(BITS4 == 0x3) { // 0011        0
				WRITE_OUTPUT(lastValue)
				DROPBITS(4)
			} else if(BITS4 == 0x7) { // 0111       -2
				lastValue-=2;
				WRITE_OUTPUT(lastValue)
				DROPBITS(4)
			} else if(BITS4 == 0xB) { // 1011        2     |
				lastValue+=2;
				WRITE_OUTPUT(lastValue)
				DROPBITS(4)
			} else {
				// C or E
				NEEDBITS(8)
				if(!(BITS8 & 0x10)) {
					// C
					uint32 nibble = BITS8 >> 5;
					uint32 sign   = nibble & 0x1;
					nibble >>= 1;
					int32 diff = nibble+3;
					if(sign)
						diff = -diff;
					lastValue+=diff;
					WRITE_OUTPUT(lastValue)
					DROPBITS(8)
				} else {
					// E
					NEEDBITS(16)
					lastValue = (hold >> 5) & 0x7FF;
					WRITE_OUTPUT(lastValue)
					DROPBITS(16)
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

TypeId Z11Codec::getSupportedTypeId() const
{
	return typeId<cv::Mat>();
}


Buffer<uint8> Z11Codec::encodeBuffer(TypedVoidConstPtr objectPtr)
{
	// get object from void pointer
	const cv::Mat* image = objectPtr;

	if(image->depth()!=CV_16U || image->channels()!=1)
		MIRA_THROW(XIO, "Z11Codec supports unsigned 16bit images with one channel only");

	Buffer<uint8> buf;
	BinaryBufferOstream stream(&buf);

	// write the image size
	stream << uint32(image->cols);
	stream << uint32(image->rows);

	// encode data
	for (int y=0; y<image->rows; ++y)
	{
		const uint16* input = image->ptr<uint16>(y);
		encodeLine(input, input+image->cols, stream);
	}

	return std::move(buf);
}

void Z11Codec::decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr)
{
	// get object from void pointer
	cv::Mat* image = ioObjectPtr;

	BinaryBufferIstream stream((Buffer<uint8>*)&data);

	// read the image size
	uint32 width,height;
	stream >> width;
	stream >> height;

	// create image with correct size and type, if necessary
	if(image->cols!=(int)width || image->rows!=(int)height ||
	   image->channels()!=1 || image->depth()!=CV_16U)
	{
		// need to create new matrix
		*image = cv::Mat(height, width, CV_16UC1);
	}

	// decode data
	for (int y=0; y<image->rows; ++y)
	{
		uint16* output = image->ptr<uint16>(y);
		decodeLine(output, output+image->cols, stream);
	}

	if(stream.fail())
		MIRA_THROW(XIO, "Premature end of data stream");
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::codec::Z11Codec, mira::BinarySerializerCodec);
