/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreWidget.C
 *    Implementation of OgreWidget.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/02
 */

#include <QApplication>
#include <QDesktopWidget>
#include <widgets/OgreWidget.h>

#include <boost/filesystem.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/assign/std/vector.hpp>

#include <utils/PathFinder.h>
#include <utils/Singleton.h>
#include <utils/ToString.h>
#include <utils/UUID.h>

#include <platform/Environment.h>

#include <widgets/OgreUtils.h>

#include <OGRE/OgreRoot.h>
#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreWindowEventUtilities.h>
#include <OGRE/OgreStringConverter.h>
#if OGRE_VERSION_MINOR >= 9
#include <OGRE/Overlay/OgreOverlaySystem.h>
#endif

#include <QMessageBox>

#if QT_VERSION < 0x050000
# if defined(Q_WS_X11)
#  include <QX11Info>
# endif
#endif

#ifdef MIRA_LINUX
# include <GL/glx.h>
# include <GL/glxext.h>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class OgreLogListener : public Ogre::LogListener
{
public:
	virtual void messageLogged(const Ogre::String& message,
	                           Ogre::LogMessageLevel lml,
	                           bool maskDebug, const Ogre::String &logName
#if OGRE_VERSION_MINOR >= 8
	                           , bool& skipThisMessage
#endif
	                           )
	{
		switch(lml)
		{
		case Ogre::LML_CRITICAL:
			MIRA_LOG(ERROR) << "[OGRE] " << message.c_str();
			break;
		case Ogre::LML_NORMAL:
			MIRA_LOG(NOTICE) << "[OGRE] " << message.c_str();
			break;
		default:
			MIRA_LOG(DEBUG) << "[OGRE] " << message.c_str();
			break;
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

class OgreSingleton : public LazySingleton<OgreSingleton>, private OgreSingletonHelper
{
public:

	OgreSingleton()
	{
#ifdef MIRA_LINUX
		/**
		 * If the display driver does not support direct rendering, loading
		 * the Ogre plugin RenderSystem_GL will end in a segfault. To avoid
		 * this crash, we check if DRI is available, before loading the plugin.
		 */

		/**
		 * The following code is partially taken from xdriinfo.c, but will
		 * fail on several machine, since glXGetScreenDriver is not implemented
		 * on all systems (especially on NVidia cards.).
		 */
		/*
		typedef const char* glXGetScreenDriverFuncType(Display *dpy, int scrNum);
		glXGetScreenDriverFuncType* getScreenDriver = NULL;
		getScreenDriver = (glXGetScreenDriverFuncType*)
				glXGetProcAddressARB((const GLubyte *)"glXGetScreenDriver");
		if (getScreenDriver != NULL) {
			Display* dpy = XOpenDisplay(NULL);
			const char* name = (*getScreenDriver)(dpy, 0);
			if (name != NULL) {
				MIRA_LOG(NOTICE) << "Using screen driver '" << name << "' for OgreWidget.";
			} else {
				MIRA_LOG(ERROR) << "The 3D view is not available, since screen is not direct rendering capable.";
				MIRA_THROW(XRuntime, "The 3D view is not available, since screen is not direct rendering capable.");
			}
			XCloseDisplay(dpy);
		}
		*/

		/**
		 * The following code is partially take from glxinfo.c and should work
		 * more reliable and on more machines.
		 */
		Display* dpy = XOpenDisplay(NULL);
		int attribSingle[] = {
			GLX_RGBA,
			GLX_RED_SIZE, 1,
			GLX_GREEN_SIZE, 1,
			GLX_BLUE_SIZE, 1,
			None };
		XVisualInfo* visinfo = glXChooseVisual(dpy, 0, attribSingle);
		GLXContext ctx = glXCreateContext(dpy, visinfo, NULL, true);

		bool hasDRI = false;
		std::string errMsg;
		if (ctx == NULL) {
			errMsg = "Creation of GLX context failed.";
		} else {
			hasDRI = glXIsDirect(dpy, ctx);
			if (!hasDRI)
				errMsg = "Screen is not direct rendering capable.";
			glXDestroyContext(dpy, ctx);
		}
		XCloseDisplay(dpy);

		if (!hasDRI) {
			MIRA_LOG(ERROR) << "Direct rendering detection failed: " << errMsg;
			if (QMessageBox::critical(NULL, tr("DRI detection failed"),
			        tr("Detection of direct rendering failed: %1\n\n"
			           "Opening the 3D view might cause an application crash!\n\n"
			           "Open the 3D view anyway?").arg(errMsg.c_str()),
			        QMessageBox::Yes | QMessageBox::No) == QMessageBox::No)
			{
				MIRA_THROW(XRuntime, "Direct rendering detection failed: " << errMsg);
			}
		}
#endif

		ogreLog = new Ogre::LogManager;
		log = ogreLog->createLog("ogre.log", true, false, true);
		log->addListener(&logListener);

		// create the main ogre object
		std::string empty;
		root = new Ogre::Root(empty,empty,empty);

		// find and load plugin
		Path pluginPath = findRenderSystemPlugin();

		if(pluginPath.string().empty()) {
			MIRA_LOG(ERROR) << "No Ogre GL rendering subsystems library found.";
			MIRA_THROW(XRuntime, "No Ogre GL rendering subsystems library found.");
		}

		MIRA_LOG(NOTICE) << "Using OpenGL plugin: " << pluginPath.string();
		root->loadPlugin(pluginPath.string());

		// load compiler plugin for cg vertex/fragment programs
		if(!loadPlugin(pluginPath, "CgProgramManager")) {
			MIRA_LOG(WARNING) << "No Ogre Cg plugin found, some shaders will "
					"not be available. Install the corresponding Ogre package, "
					"if you need them.";
		}

		// ensure, that the renderer was installed successfully
		const Ogre::RenderSystemList& renderers = root->getAvailableRenderers();
		if(renderers.empty()) {
			MIRA_LOG(ERROR) << "No Ogre rendering subsystems were found.";
			MIRA_THROW(XRuntime, "No Ogre rendering subsystems were found.");
		}

		Ogre::RenderSystem *renderSystem;
		renderSystem = root->getRenderSystemByName("OpenGL Rendering Subsystem");

		assert(renderSystem); // user might pass back a null renderer, which would be bad!

		root->setRenderSystem( renderSystem );

		// initialize without creating window
		root->getRenderSystem()->setConfigOption( "Full Screen", "No" );
		root->saveConfig();

#if OGRE_VERSION_MINOR >= 9
		Ogre::OverlaySystem *overlaySystem = new Ogre::OverlaySystem();
#endif

		root->initialise(false); // don't create a window

		// create a dummy GLWidget that is used for sharing the GL context
		sharedGLWidget = new QGLWidget();
		sharedGLWidget->setAttribute(Qt::WA_DeleteOnClose);
		sharedGLWidget->hide();

		connect(QApplication::instance(), SIGNAL(destroyed(QObject*)), this, SLOT(onAboutToQuit()));
	}

	QGLWidget* getSharedGLWidget()
	{
		assert(sharedGLWidget);
		return sharedGLWidget;
	}

	~OgreSingleton() {}

	void loadResources( std::string const& directory,
	                      std::string const& resourceGroup,
	                      std::string const& versionSuffix )
	{
		Ogre::ResourceGroupManager& mgr = Ogre::ResourceGroupManager::getSingleton();
		if (!mgr.resourceGroupExists(resourceGroup))
			mgr.createResourceGroup(resourceGroup);

		std::vector<Path> meshs = findProjectDirectories("resources/"+directory);
		foreach(const auto& p, meshs) {
			if ( boost::filesystem::exists(p.string()+versionSuffix) )
				OgreUtils::loadResource(p.string()+versionSuffix, resourceGroup);
			else
				OgreUtils::loadResource(p.string(), resourceGroup);
		}
	}

	void initializeUtils()
	{
		// Load resources

		// Search for meshs, fonts, materials and textures folders and load the associated resources.
		// If a folder with the suffix of the installed ogre version exists, this folder is loaded
		// instead. e.g. if the installed ogre version is 1.9 and an mesh-1_9 folder exists, the
		// mesh folder is ignored and the mesh-1_9 folder is loaded
		std::string versionSuffix = "-"+toString<int>(OGRE_VERSION_MAJOR)+"_"+toString<int>(OGRE_VERSION_MINOR);

		loadResources( "meshs", "MIRAMeshs", versionSuffix );
		loadResources( "fonts", "MIRAFonts", versionSuffix );
		loadResources( "materials", "MIRAMaterials", versionSuffix );
		loadResources( "textures", "MIRATextures", versionSuffix );
	}

private:

	virtual void onAboutToQuit()
	{
		// On Qt5 closing this widget will cause a crash when the application
		// is closed. For now, we simply skip this. Maybe, we will find a more
		// cleaner way in the future.
#if QT_VERSION < 0x050000
		// close and destroy the shared widget first (immediate deletion fails on Windows)
		if(sharedGLWidget)
			sharedGLWidget->close();
#endif
		sharedGLWidget = NULL;
		//QApplication::processEvents();

		// delete ogres root which will trigger the Ogre shutdown
		delete root;

		// logging must be destroyed AFTER the root
		ogreLog->destroyLog(log);
		delete ogreLog;

		MIRA_LOG(DEBUG) << "Finished Ogre shutdown";
	}

private:

	Path findRenderSystemPlugin()
	{
		#define TRY_FINDMIRAFILE(file)        \
			try {                             \
				return findProjectFile(file); \
			} catch(...) {}

		#define TRY_FINDFILE(paths,file)     \
			try {                            \
				return findFile(paths,file); \
			} catch(...) {}

		// try to find rendersystem first within the MIRA_PATH
		TRY_FINDMIRAFILE("lib/RenderSystem_GL.so")
		#ifndef NDEBUG
			TRY_FINDMIRAFILE("lib/RenderSystem_GL_d.dll")
		#endif
		TRY_FINDMIRAFILE("lib/RenderSystem_GL.dll")

		 // otherwise look within system dirs
		PathVector paths;

		try { // first look in the path specified by the OGRE_HOME environment variable (if there is such)
			Path ogreHome = resolveEnvironmentVariable("OGRE_HOME");
			paths.push_back(ogreHome);
			paths.push_back(ogreHome/"lib");
			paths.push_back(ogreHome/"lib/OGRE");
			paths.push_back(ogreHome/"bin");
			paths.push_back(ogreHome/"bin/release/");
			paths.push_back(ogreHome/"bin/debug/");
		} catch(...) {}

		boost::assign::push_back(paths) =
				"/usr/lib/OGRE*/",
				"/usr/local/lib/OGRE*/",
				"/usr/lib64/OGRE*/",
				"/usr/lib/i386-linux-gnu/OGRE*",
				"/usr/lib/x86_64-linux-gnu/OGRE*",
				"/usr/lib/arm-linux-gnueabihf/OGRE*",
				"/usr/local/lib64/OGRE*/",
				"C:/opt/OgreSDK*/bin/release",
				"C:/opt/OgreSDK*/bin/debug",
				"C:/opt/Ogre*/bin",
				"C:/opt/OGRE*/bin";

		TRY_FINDFILE(paths,"RenderSystem_GL.so")
		#ifndef NDEBUG
			TRY_FINDFILE(paths,"RenderSystem_GL_d.dll");
		#endif
		TRY_FINDFILE(paths,"RenderSystem_GL.dll");

		return Path();
	}

	bool loadPlugin(const Path& renderSystemPath, const std::string& plugin)
	{
		Path pluginRoot = renderSystemPath.parent_path();

		// try linux library
		Path pluginPath = pluginRoot / ("Plugin_" + plugin + ".so");
		if(boost::filesystem::is_regular_file(pluginPath)) {
			MIRA_LOG(NOTICE) << "Loading Ogre plugin: " << pluginPath.string();
			root->loadPlugin(pluginPath.string());
			return true;
		}

		// try windows library
		#ifdef NDEBUG
			pluginPath = pluginRoot / ("Plugin_" + plugin + ".dll");
		#else
			pluginPath = pluginRoot / ("Plugin_" + plugin + "_d.dll");
		#endif

		if(boost::filesystem::is_regular_file(pluginPath)) {
			MIRA_LOG(NOTICE) << "Loading Ogre plugin: " << pluginPath.string();
			root->loadPlugin(pluginPath.string());
			return true;
		}

		return false;
	}

public:

	Ogre::Root* root;
	Ogre::LogManager* ogreLog;
	Ogre::Log* log;
	OgreLogListener logListener;

	QGLWidget* sharedGLWidget;
};

///////////////////////////////////////////////////////////////////////////////

OgreWidget::OgreWidget(QWidget *parent) :
		QGLWidget(parent, OgreSingleton::instance().getSharedGLWidget()),
		mRenderWindow(0)
{
	setAutoBufferSwap(true);
	setAttribute(Qt::WA_OpaquePaintEvent);
}

OgreWidget::~OgreWidget()
{
	if (mRenderWindow) {
		mRenderWindow->removeAllViewports();
		OgreSingleton::instance().root->detachRenderTarget(mRenderWindow);

		MIRA_LOG(DEBUG) << "Destroying RenderWindow: '" << mRenderWindow->getName() << "'";
		OgreSingleton::instance().root->getRenderSystem()->destroyRenderWindow(mRenderWindow->getName());
	}
}

void OgreWidget::paintGL()
{
	if(isHidden())
		return;

	// Render Ogre
	try {
		OgreSingleton::instance().root->_fireFrameStarted();
		mRenderWindow->update();
		OgreSingleton::instance().root->_fireFrameEnded();
	} catch(...) {
		// ignore all errors, e.g. Vertex Buffer out of memory, etc.
	}
}

void OgreWidget::initializeGL()
{
	// initialize ogre singleton with ogre root
	OgreSingleton::instance();

	Ogre::NameValuePairList params;
	Ogre::String externalWindowHandleParams;

#ifdef MIRA_WINDOWS
	//positive integer for W32 (HWND handle) - According to Ogre Docs
	//externalWindowHandleParams = Ogre::StringConverter::toString((unsigned int)(this->parentWidget()->winId()));
	externalWindowHandleParams = Ogre::StringConverter::toString((unsigned int)(this->winId()));
#endif

#ifdef MIRA_LINUX
# if QT_VERSION >= 0x050000
	QDesktopWidget* desktop = QApplication::desktop();
	// Since QX11Info was removed in Qt5, we simply use "0:" here.
	externalWindowHandleParams  = "0:";
	externalWindowHandleParams += Ogre::StringConverter::toString((unsigned int)(desktop->screenNumber(this)));
	externalWindowHandleParams += ":";
	externalWindowHandleParams += Ogre::StringConverter::toString((unsigned long)(winId()));
# else
	QX11Info info = x11Info();
	externalWindowHandleParams  = Ogre::StringConverter::toString((unsigned long)(info.display()));
	externalWindowHandleParams += ":";
	externalWindowHandleParams += Ogre::StringConverter::toString((unsigned int)(info.screen()));
	externalWindowHandleParams += ":";
	externalWindowHandleParams += Ogre::StringConverter::toString((unsigned long)(winId()));
# endif
#endif

	//Add the external window handle parameters to the existing params set.
#ifdef MIRA_WINDOWS
	params["externalWindowHandle"] = externalWindowHandleParams;
	//params["parentWindowHandle"] = externalWindowHandleParams;
#endif

#ifdef MIRA_LINUX
	params["parentWindowHandle"] = externalWindowHandleParams;
#endif

	params["externalGLControl"] = "true"; // important for windows
	params["currentGLContext"] = "true";

	//Finally create our window using unique name based on uuid.
	std::string windowName = "OgreWindow_" + toString(boost::uuids::random_generator()());
	mRenderWindow = OgreSingleton::instance().root->createRenderWindow(windowName, width(), height(), false, &params);

	//mRenderWindow->setActive(true);
	mRenderWindow->setVisible(true);

	OgreSingleton::instance().initializeUtils();

	initOgre();
	mRenderWindow->setActive(true);
}

void OgreWidget::resizeGL(int w, int h)
{
	if (mRenderWindow) {
		mRenderWindow->resize(w, h);
		mRenderWindow->windowMovedOrResized();

		for(int i = 0; i < mRenderWindow->getNumViewports(); ++i)
		{
			Ogre::Viewport* vp = mRenderWindow->getViewport(i);
			Ogre::Camera* camera = vp->getCamera();
			camera->setAspectRatio((Ogre::Real)vp->getActualWidth() / (Ogre::Real)vp->getActualHeight());
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
