/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServicePropertyDelegate.C
 *    Implementation of ServicePropertyDelegate.h.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2012/03/26
 */

#include <visualization/private/ServicePropertyDelegate.h>

#include <fw/Framework.h>

#include <widgets/PropertyEditor.h>
#include <widgets/SelectionListDialog.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class ServicePropertyDelegate : public PropertyEditor::Delegate
{
	MIRA_OBJECT(ServicePropertyDelegate)

public:

	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<ServiceProperty>(SupportedType::TEXT |
		                                          SupportedType::COLOR |
		                                          SupportedType::EDITOR |
		                                          SupportedType::DIALOG);
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const TypedPropertyNode<ServiceProperty>* p =
				property->toTyped<ServiceProperty>();
		assert(p!=NULL);
		ServiceProperty ch = p->get();

		if(ch.getService().empty())
			return "<not assigned>";
		else
			return ch.getService();
	}

	virtual QColor getBackgroundColor(const PropertyNode* property)
	{
		const TypedPropertyNode<ServiceProperty>* p =
				property->toTyped<ServiceProperty>();
		assert(p!=NULL);
		ServiceProperty ch = p->get();

		/*if(!ch.isValid())
			return PropertyEditor::lightRed;
		else*/
			return QColor(); // use default color
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent)
	{
		ServicePropertyDelegateLineEdit* e =
				new ServicePropertyDelegateLineEdit(property, parent);
		e->setFrame(false);
		return e;
	}

	virtual bool execDialog(PropertyNode* property, QWidget* parent)
	{
		TypedPropertyNode<ServiceProperty>* p = property->toTyped<ServiceProperty>();
		assert(p!=NULL);

		ServiceProperty ch = p->get();

		SelectionListDialog<int> d(parent);
		d.setWindowTitle(d.tr("Select service"));
		d.setHeaderLabels(QStringList() << "Service");

		std::list<std::string> services =
				MIRA_FW.getRPCManager().queryServicesForInterface(ch.getInterface());

		foreach(const std::string s, services)
			d.addItem(QStringList() << s.c_str());

		if(d.exec()==0)
			return false;

		auto selection = d.selectedItems();
		if(selection.empty())
			return false; // nothing chosen

		ch = selection.front().first.toStdString();
		p->set(ch);

		return true;
	}
};

///////////////////////////////////////////////////////////////////////////////

ServicePropertyDelegateLineEdit::ServicePropertyDelegateLineEdit(
		PropertyNode* p, QWidget* parent) :
	QLineEdit(parent)
{
	property = p->toTyped<ServiceProperty>();
	ServiceProperty ch = property->get();
	setText(ch.getService().c_str());

	connect(this, SIGNAL(editingFinished()), this, SLOT(slotSetValue()));
}

void ServicePropertyDelegateLineEdit::slotSetValue()
{
	ServiceProperty ch = property->get();
	ch = text().toStdString();
	property->set(ch);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::ServicePropertyDelegate,
                    mira::PropertyEditor::Delegate);
