/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CameraFlyTool.C
 *    Implementation of CameraFlyTool.
 *
 * @author Erik Einhorn
 * @date   2011/07/01
 */

#include <QMouseEvent>

#include <visualization/VisualizationTool3D.h>
#include <visualization/3d/MeshObject.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
class SmoothValue
{
public:

	SmoothValue(T value=0) {
		reset(value);
	}


	SmoothValue& operator=(T value) {
		mTargetValue = value;
		return *this;
	}

	T value() const {
		return mValue;
	}

	T target() const {
		return mTargetValue;
	}

	operator T() const {
		return value();
	}

	void reset(T value=0) {
		mValue = value;
		mTargetValue = value;
	}

	void update(float alpha = 0.2) {
		T dvalue = mTargetValue - mValue;
		mValue += alpha  * dvalue;
	}

private:

	T mValue;
	T mTargetValue;

};

typedef SmoothValue<float> SmoothValuef;

///////////////////////////////////////////////////////////////////////////////

class CameraFlyTool : public VisualizationTool3D
{
	MIRA_META_OBJECT(CameraFlyTool,
			("Name", "Fly")
			("Description", "Look around and fly through the scene")
			("Category", "Camera")
			("Order", "1"))

public:

	CameraFlyTool() {
		reset();
	}

	virtual QIcon getIcon() {
		QPixmap pixmap(":/icons/ToolCameraFly.png");
		return QIcon(pixmap);
	}

	/// Called if the tool is active and a mouse button was pressed.
	virtual void onMousePressed(QMouseEvent* e) {
		mMouseDownPos = e->globalPos();
	}

	virtual void onMouseReleased(QMouseEvent* e) {
		mYawSpeed = 0.0f;
		mPitchSpeed = 0.0f;
		mRightSpeed = 0.0f;
		mUpSpeed = 0.0f;
	}

	virtual void onMouseMoved(QMouseEvent* e) {
		// obtain the mouse movement since the last event
		QPoint delta = e->globalPos() - mMouseDownPos;

		IVisualization3DSite::Camera cam = getSite()->getCamera();

		switch (e->buttons()) {
		case Qt::LeftButton:
			mPitchSpeed = delta.y()/300.0f;
			mYawSpeed = -delta.x()/300.0f;
			break;
		case Qt::RightButton:
			mRightSpeed = -delta.x()/150.0f;
			mUpSpeed = -delta.y()/150.0f;
			break;
		default:
			break;
		}

	}

	virtual void onMouseWheel(QWheelEvent * e) {

		Time wheelCurTime = Time::now();
		float wheelTime = (wheelCurTime - mWheelLastTime).totalMilliseconds() / 1000.0f;
		mWheelLastTime = wheelCurTime;
		wheelTime = 1.0f + (10.0f - (20.0f * std::min(wheelTime, 0.5f)));

		float ds = wheelTime*0.1f;
		if(e->delta()<0)
			ds = -ds;

		const float cEpsSpeed = 0.01f;
		mSpeed = mSpeed.target()+ ds;
		if(mSpeed.target()<0.0f && mSpeed>cEpsSpeed)
			mSpeed=0.0f;
		else if(mSpeed.target()>0.0f && mSpeed<-cEpsSpeed)
			mSpeed=0.0f;

	}

	void activate()
	{
		mWheelLastTime = Time::now();
		reset();
	}

	void deactivate()
	{
		reset();
	}

	void setupScene(IVisualization3DSite* site)
	{
		VisualizationTool3D::setupScene(site);
	}

	virtual void update(Duration time) {
		float dt = time.totalMilliseconds() / 1000.0f;


		mYawSpeed.update(dt*5.0f);
		mPitchSpeed.update(dt*5.0f);
		mSpeed.update(dt*5.0f);
		mUpSpeed.update(dt*5.0f);
		mRightSpeed.update(dt*5.0f);

		IVisualization3DSite::Camera cam = getSite()->getCamera();
		Eigen::Quaternionf q = quaternionFromYawPitchRoll(cam.yaw,cam.pitch,cam.roll);

		Eigen::Quaternionf ry(Eigen::AngleAxisf(mPitchSpeed*dt, Eigen::Vector3f::UnitY()));
		q *= ry;

		boost::tie(cam.yaw, cam.pitch, cam.roll) = quaternionToYawPitchRoll(q);
		cam.yaw += mYawSpeed * dt;

		Eigen::Vector3f direction = q * Eigen::Vector3f::UnitX();

		cam.position += direction * mSpeed * dt;

		Eigen::Vector3f up    = q * Eigen::Vector3f::UnitZ();
		Eigen::Vector3f right = q * Eigen::Vector3f::UnitY();

		cam.position += up * mUpSpeed * dt + right * mRightSpeed * dt;

		getSite()->setCamera(cam);
	}

	void reset()
	{
		mYawSpeed.reset(0.0f);
		mPitchSpeed.reset(0.0f);

		mSpeed.reset(0.0f);
		mRightSpeed.reset(0.0f);
		mUpSpeed.reset(0.0f);
	}

private:
	QPoint mMouseDownPos;

	Time mWheelLastTime;

	SmoothValuef mSpeed;

	SmoothValuef mYawSpeed;
	SmoothValuef mPitchSpeed;

	SmoothValuef mRightSpeed;
	SmoothValuef mUpSpeed;

};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::CameraFlyTool, mira::VisualizationTool3D)
