/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization2DView.C
 *    Implementation of Visualization2DView.h.
 *
 * @author Erik Einhorn
 * @date   2011/01/15
 */

#include <views/Visualization2DView.h>

#include <QBasicTimer>
#include <QTimeLine>
#include <QGraphicsItem>
#include <QGraphicsScene>
#include <QDragMoveEvent>
#include <QScrollBar>

#include <views/VisualizationControlPage.h>
#include <visualization/VisualizationTool2D.h>

#include <widgets/QtUtils.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION( mira::Visualization2DView, mira::VisualizationViewTransformable);

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Visualization2DView::UI::UI(Visualization2DView* parent) : QGraphicsView(parent->mScene, parent), vis(parent)
{
	setAlignment(Qt::AlignCenter);
	setAcceptDrops(true);
	setDragMode(QGraphicsView::NoDrag);
	setInteractive(true);
	setFrameStyle(QGraphicsView::NoFrame);
	setMouseTracking(true); // Necessary to receive mouseMoveEvents (at all)
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setTransformationAnchor(QGraphicsView::NoAnchor);
	setViewportUpdateMode(QGraphicsView::FullViewportUpdate);
	setSceneRect(-1.0e10,-1.0e10,2.0*1e10,2.0*1.0e10);
	vis->startUpdateTimer();
}

void Visualization2DView::UI::dragEnterEvent(QDragEnterEvent *event)	{
	setAcceptDrops(true);
	vis->dragEnterEventImpl(event);
}

void Visualization2DView::UI::dropEvent(QDropEvent *event) {
	vis->dropEventImpl(event, this);
}

void Visualization2DView::UI::dragMoveEvent(QDragMoveEvent* event) {
	event->accept();
}

void Visualization2DView::UI::mousePressEvent(QMouseEvent* e) {
	if(vis->getActiveTool()!=NULL)
		vis->getActiveTool()->onMousePressed(e);
}

void Visualization2DView::UI::mouseReleaseEvent(QMouseEvent* e) {
	if(vis->getActiveTool()!=NULL)
		vis->getActiveTool()->onMouseReleased(e);
}

void Visualization2DView::UI::mouseMoveEvent(QMouseEvent* e)
{
	if(vis->getActiveTool()!=NULL)
		vis->getActiveTool()->onMouseMoved(e);
}

void Visualization2DView::UI::wheelEvent(QWheelEvent * e)
{
	if(vis->getActiveTool()!=NULL) {
		vis->getActiveTool()->onMouseWheel(e);
		e->accept();
	}
}

void Visualization2DView::UI::readdVisualizationItems()
{
	std::list<Visualization*> visualizations = vis->getVisualizations();
	if (visualizations.size() < 2)
		return;

	foreach (Visualization* v, visualizations) {
		Visualization2D* vis2d = dynamic_cast<Visualization2D*>(v);
		assert (vis2d!=NULL);

		QGraphicsItem* item = vis2d->getItem();
		assert (item!=NULL);

		vis->getSceneManager()->removeItem(item);
		vis->getSceneManager()->addItem(item);
	}
}

///////////////////////////////////////////////////////////////////////////////

Visualization2DView::Visualization2DView() : ui(NULL),
                                             mSaveContentMinPictureSize(2048)
{
	mScene = new QGraphicsScene(this);
	mScene->setItemIndexMethod(QGraphicsScene::NoIndex);

	setUpdateInterval(40);
	setBackgroundColor(Qt::white);
}

void Visualization2DView::setupVisualizations()
{
	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void Visualization2DView::setupTools()
{
	assert(getSceneManager()!=NULL);

	// initialize all existing tools
	foreach(VisualizationTool* tool, getTools())
		tool->init(this);
}

void Visualization2DView::setBackgroundColor(const QColor& color)
{
	mScene->setBackgroundBrush(QBrush(color));
	mBackgroundColor = color;
}

void Visualization2DView::updateCamera()
{
	QTransform t;

	// By inverting the scale of the y axis, we flip the view vertically.
	// This allows us to have the same coordinate system as the 3D view.
	// Only caveat is that some calculations need to be adjusted to account
	// for this, e.g. in the 2D view tools.
	t.scale(mCameraSettings.scale, -mCameraSettings.scale);
	t.rotate(mCameraSettings.orientation.value(), Qt::ZAxis);
	t.translate(mCameraSettings.position(0), mCameraSettings.position(1));

	ui->setTransform(t, false);
}

const Visualization2DView::Camera& Visualization2DView::getCamera() const
{
	return mCameraSettings;
}

void Visualization2DView::setCamera(const Camera& camera)
{
	mCameraSettings = camera;
	updateCamera();
}

float Visualization2DView::getToolBaseZValue() const {
	return 100.0f; // Arbitrary but should do
}

void Visualization2DView::addVisualization(Visualization* vis)
{
	Visualization2D* v = dynamic_cast<Visualization2D*>(vis);
	assert(v!=NULL);
	v->init(this);
	VisualizationView::addVisualization(vis);
}

void Visualization2DView::moveUpVisualization(Visualization* vis)
{
	VisualizationView::moveUpVisualization(vis);
	ui->readdVisualizationItems();
}

void Visualization2DView::moveDownVisualization(Visualization* vis)
{
	VisualizationView::moveDownVisualization(vis);
	ui->readdVisualizationItems();
}

void Visualization2DView::addTool(VisualizationTool* tool)
{
	VisualizationTool2D* t = dynamic_cast<VisualizationTool2D*>(tool);
	assert(t!=NULL);

	if(getSceneManager()!=NULL)
		t->init(this);

	VisualizationView::addTool(tool);
}

const Class& Visualization2DView::supportedVisualizationClass() const
{
	return Visualization2D::CLASS();
}

const Class& Visualization2DView::supportedVisualizationToolClass() const
{
	return VisualizationTool2D::CLASS();
}

void Visualization2DView::update(Duration dt)
{
	assert(ui!=NULL);
	updateCamera();
}

QWidget* Visualization2DView::createVisualizationPart()
{
	ui = new UI(this);

	if(getVisualizations().empty())
		populateDefaultVisualizations();
	else
		setupVisualizations();

	setupTools();

	updateCamera();

	return ui;
}

void Visualization2DView::resetView()
{
	auto result = QMessageBox::question(NULL, trUtf8("Reset perspective"), trUtf8("Are you sure you want to reset the view perspective?"),
	                                    QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
	if(result == QMessageBox::Yes) {
		mCameraSettings.scale = 1.0f;
		mCameraSettings.position = Point2f();
		mCameraSettings.orientation = Degreef(0.0f);
		updateCamera();
	}
}

void Visualization2DView::saveContentToFile()
{
	QStringList extList;
	extList << "png" << "bmp" << "jpg" << "jpeg";
	static QString saveDir = ".";
	QString filename = QtUtils::getSaveFileName(NULL, trUtf8("Save content as image"), saveDir, "Images (*.png *.bmp *.jpg *.jpeg)",
	                                            NULL, QFileDialog::DontUseNativeDialog, extList);
	if(filename.isEmpty())
		return;

	auto mgr = getViewManager();

	const int w = mgr->width();
	const int h = mgr->height();
	const float ar = static_cast<float>(w) / static_cast<float>(h);

	uint targetW = w;
	uint targetH = h;
	if(mSaveContentMinPictureSize > std::min<uint>(targetW, targetH))
	{
		if(w >= h)
		{
			targetH = mSaveContentMinPictureSize;
			targetW = floor(targetH * ar + 0.5f);
		}
		else
		{
			targetW = mSaveContentMinPictureSize;
			targetH = floor(targetH / ar + 0.5f);
		}
	}

	QImage image(targetW, targetH, QImage::Format_RGB32);
	QPainter painter(&image);
	painter.setRenderHint(QPainter::Antialiasing);
	mScene->render(&painter);
	image.save(filename);

	// memorize the selected location
	saveDir = QFileInfo(filename).path();
}

QGraphicsScene* Visualization2DView::getSceneManager()
{
	return mScene;
}

QGraphicsView* Visualization2DView::getViewManager()
{
	return ui;
}

void Visualization2DView::createDefaultVisualization(const std::string& className,
                                                     const std::string& displayName)
{
	try {
		if (ClassFactory::instance().isClassRegistered(className)) {
			ClassProxy c = ClassFactory::instance().getClassByIdentifier(className);
			Visualization2D* vis = c.newInstance<Visualization2D>();
			vis->setName(displayName);
			addVisualization(vis);
		}
	} catch(...) {} // ignore errors
}

void Visualization2DView::populateDefaultVisualizations()
{
	createDefaultVisualization("mira::AxesVisualization2D", "Axes");
}

///////////////////////////////////////////////////////////////////////////////

}
