/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SignalBinder.h
 *    Class that allows to bind a function to a Qt slot that is called upon
 *    a connected signal.
 *
 * @author Erik Einhorn
 * @date   2011/04/06
 */

#ifndef _MIRA_SIGNALBINDER_H_
#define _MIRA_SIGNALBINDER_H_

#ifndef Q_MOC_RUN
#include <boost/bind.hpp>
#include <boost/function.hpp>
#endif

#include <QObject>

#include <widgets/GuiWidgetsExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Class that provides different slots with different parameters. For each such
 * slot a boost function can be bound that is called if the slot is invoked.
 * This mechanism can be used as workaround since Qt MOC is not able to handle
 * templates.
 *
 * Instead of Qt's connect:
 * <code>
 * connect(button, SIGNAL(clicked()), this, SLOT(method());
 * </code>
 * you can use:
 * <code>
 * SignalBinder::connect(button, SIGNAL(clicked()), &MyClass::method, this);
 * </code>
 *
 * Example how to use arbitrary bound methods:
 * <code>
 * SignalBinder* s = new SignalBinder(this);
 * s.bind(myMethod);
 * connect(someWidget,SIGNAL(someSignal(int)), s, SLOT(slot(int)));
 * </code>
 */
class MIRA_GUI_WIDGETS_EXPORT SignalBinder : public QObject
{
	Q_OBJECT
public:
	SignalBinder(QObject* parent) :
		QObject(parent) {}

public:

	/// binds the given function to the slot "slot()"
	void bind(boost::function<void()> f)    { mVoid = f; }

	/// binds the given function to the slot "slot(int)"
	void bindInt(boost::function<void(int)> f) { mInt = f; }

	/// binds the given function to the slot "slot(const QString&)"
	void bindQString(boost::function<void(const QString&)> f) { mQString = f; }

public slots:

	void slot()                   { if(mVoid) mVoid(); }
	void slot(int val)            { if(mInt)  mInt(val); }
	void slot(const QString& val) { if(mQString) mQString(val); }

public:


	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified function.
	 * This variant is for method with the signature 'void method()'.
	 */
	static SignalBinder* connect(QObject* sender, const char* signal,
	                             boost::function<void()> function)
	{
		SignalBinder* binder = new SignalBinder(sender);
		binder->bind(function);
		binder->QObject::connect(sender,signal, binder, SLOT(slot()));
		return binder;
	}

	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified 'method' of the given class 'instance'.
	 * This variant is for method with the signature 'void method()'.
	 */
	template<typename Class>
	static SignalBinder* connect(QObject* sender, const char* signal,
	             void (Class::*method)(), Class* instance) {
		return connect(sender,signal,(boost::function<void()>)boost::bind(method, instance));
	}

	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified function.
	 * This variant is for method with the signature 'void method(int)'.
	 */
	static SignalBinder* connect(QObject* sender, const char* signal,
	                             boost::function<void(int)> function)
	{
		SignalBinder* binder = new SignalBinder(sender);
		binder->bindInt(function);
		binder->QObject::connect(sender,signal, binder, SLOT(slot(int)));
		return binder;
	}

	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified 'method' of the given class 'instance'.
	 * This variant is for method with the signature 'void method(int)'.
	 */
	template<typename Class>
	static SignalBinder* connect(QObject* sender, const char* signal,
	             void (Class::*method)(int), Class* instance) {
		return connect(sender,signal,(boost::function<void(int)>)boost::bind(method, instance, _1));
	}

	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified function.
	 * This variant is for method with the signature 'void method(const QString&)'.
	 */
	static SignalBinder* connect(QObject* sender, const char* signal,
	                             boost::function<void(const QString&)> function)
	{
		SignalBinder* binder = new SignalBinder(sender);
		binder->bindQString(function);
		binder->QObject::connect(sender,signal, binder, SLOT(slot(const QString&)));
		return binder;
	}

	/**
	 * Create a SignalBinder instance and binds the specified 'signal' of
	 * the 'sender' to the specified 'method' of the given class 'instance'.
	 * This variant is for method with the signature 'void method(const QString&)'.
	 */
	template<typename Class>
	static SignalBinder* connect(QObject* sender, const char* signal,
	             void (Class::*method)(const QString&), Class* instance) {
		return connect(sender,signal,(boost::function<void(const QString&)>)boost::bind(method, instance, _1));
	}

private:

	boost::function<void()> mVoid;
	boost::function<void(int)> mInt;
	boost::function<void(const QString&)> mQString;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
