/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelProperty.h
 *    Description of ChannelProperty and related classes.
 *
 * @author Erik Einhorn
 * @date   2010/12/22
 */

#ifndef _MIRA_CHANNELPROPERTY_H_
#define _MIRA_CHANNELPROPERTY_H_

#include <QLineEdit>

#include <serialization/ReflectorInterface.h>
#include <serialization/IsObjectTrackable.h>
#include <serialization/SplitReflect.h>
#include <serialization/GetterSetter.h>
#include <serialization/PropertySerializer.h>

#include <fw/Framework.h>
#include <fw/AuthorityProvider.h>

#include <visualization/GuiVisualizationExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for ChannelProperty template class.
 */
class ChannelPropertyBase
{
public:

	ChannelPropertyBase() : mAuthorityProvider(NULL), mOptional(false) {}
	virtual ~ChannelPropertyBase() {}

public:

	/// Returns the channelID that was set by the user
	const std::string& getID() const {
		return mChannelID;
	}

	/// Sets the name of the property
	void setName(const std::string& name) {
		mName = name;
	}

	/// Returns the name of the property
	const std::string& getName() const {
		return mName;
	}

public:

	/// Returns true, if an existing channel is set. Must be implemented in derived class.
	virtual bool isValid() const = 0;
	
	/// Returns the number of data updates the channel has received
	virtual uint32 getDataUpdateCount() const = 0;

	/**
	 * Returns list of (channelID,typename) pairs containing the channels that
	 * match the desired type. Must be implemented in derived class.
	 */
	virtual std::list<std::pair<std::string, Typename>> getAvailableChannels() = 0;

public:

	/// For internal use only
	void setOptional() {
		mOptional = true;
	}

	/// For internal use only
	bool isOptional() const {
		return mOptional;
	}

public:
	/**
	 * Tell the property to update the assigned channel
	 * @throw XRuntime If no channel is set or if the channel does not exist.
	 */
	virtual void update() = 0;

	/**
	 * Sets the channelID and a necessary authority provider that is used to
	 * obtain the authority for subscribing the set channel.
	 */
	virtual void set(const std::string& id, IAuthorityProvider* authorityProvider)
	{
		mChannelID = id;
		mAuthorityProvider = authorityProvider;
	}

protected:
	std::string mName;
	std::string mChannelID;
	IAuthorityProvider* mAuthorityProvider;
	bool mOptional;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Proxy class that is returned/set by the getter and setter methods of
 * ChannelProperty. It acts as an interface between the ChannelProperty
 * template class and the PropertyEditor. There's a PropertyEditor delegate
 * for this ChannelPropertyProxy class.
 *
 * For internal use only!
 */
class ChannelPropertyProxy
{

public:

	ChannelPropertyProxy() : property(NULL) {}
	ChannelPropertyProxy(const std::string& id, ChannelPropertyBase* p) :
		channelID(id), property(p) {}

public:

	template <typename Reflector>
	void reflect(Reflector& r) {
		r.member("Channel",channelID, "The ID of the channel");
	}

public:

	/// Returns true, if an existing channel is set.
	bool isValid() const {
		assert(property);
		return property->isValid();
	}

	/**
	 * Returns list of (channelID,typename) pairs containing the channels that
	 * match the desired type.
	 */
	std::list<std::pair<std::string, Typename>> getAvailableChannels()
	{
		assert(property);
		return property->getAvailableChannels();
	}

public:

	std::string channelID;
	ChannelPropertyBase* property;

};

template<>
class IsObjectTrackable<ChannelPropertyProxy> : public std::false_type {};

///////////////////////////////////////////////////////////////////////////////

/**
 * The concrete typed ChannelProperty template class.
 * If you want to use channels that can be set using the property editor, you
 * must use this ChannelProperty<T> class instead of Channel<T>.
 * In your reflect method you need to call the channelProperty() method
 * of this class which will install special getter and setter methods. These
 * getter and setter methods will use the ChannelPropertyProxy as an interface
 * to the PropertyEditor.
 *
 * The set or chosen channel can be obtained using the getChannel() method
 * which will subscribe the channel and return the corresponding Channel<T>
 * object, if the channel exists.
 */
template <typename T>
class ChannelProperty : public ChannelPropertyBase
{

public:
	ChannelProperty() :
		mDataChanged(false),
		mDataUpdates(0),
		mDataChangedFn(NULL),
		mDataIntervalChangedFn(NULL),
		mChannelChangedFn(NULL)
	{}

public: // implementation of ChannelPropertyBase

	virtual bool isValid() const {
		return mChannel.isValid();
	}

	virtual uint32 getDataUpdateCount() const {
		return mDataUpdates;
	}

	virtual std::list<std::pair<std::string, Typename>> getAvailableChannels() {
		return MIRA_FW.getChannelManager().getChannelsOfType<T>();
	}

public:

	// overwritten from ChannelPropertyBase
	// will unsubscribe a set channel
	virtual void set(const std::string& id, IAuthorityProvider* authorityProvider)
	{
		ChannelPropertyBase::set(id,authorityProvider);
		mDataUpdates = 0;

		// unsubscribe and reset our channel
		if(mChannel.isValid()) {
			assert(mAuthorityProvider!=NULL);
			Authority& authority = mAuthorityProvider->getAuthority();
			authority.unsubscribe<T>(mChannel.getID());
			mChannel.reset();
		}
		if ( mChannelChangedFn )
			mChannelChangedFn();
	}

public:
	void setDataChangedCallback(boost::function<void (ChannelRead<T>)> dataChangedFn)
	{
		mDataChangedFn = dataChangedFn;
		mDataIntervalChangedFn = NULL;
	}

	template<typename Class>
	void setDataChangedCallback(void (Class::*f)(ChannelRead<T>), Class* obj)
	{
		mDataChangedFn = boost::bind(f, obj, _1);
		mDataIntervalChangedFn = NULL;
	}

	void setDataIntervalChangedCallback(boost::function<void (ChannelReadInterval<T>)> dataChangedFn)
	{
		mDataChangedFn = NULL;
		mDataIntervalChangedFn = dataChangedFn;
	}

	template<typename Class>
	void setDataIntervalChangedCallback(void (Class::*f)(ChannelReadInterval<T>), Class* obj)
	{
		mDataChangedFn = NULL;
		mDataIntervalChangedFn = boost::bind(f, obj, _1);
	}

	void setChannelChangedCallback(boost::function<void ()> channelChangedFn)
	{
		mChannelChangedFn = channelChangedFn;
	}

	template<typename Class>
	void setChannelChangedCallback(void (Class::*f)(), Class* obj)
	{
		mChannelChangedFn = boost::bind(f, obj);
	}

	/**
	 * Obtains the set or chosen channel.
	 * Will subscribe the channel (if not yet subscribed) and return the
	 * corresponding Channel<T> object, if the channel exists.
	 *
	 * @throw XRuntime If no channel is set or if the channel does not exist.
	 */
	Channel<T> getChannel()
	{
		if(mChannelID.empty())
			MIRA_THROW(XRuntime, "No channel specified.");

		if(!mChannel.isValid() && mAuthorityProvider!=NULL) {
			Authority& authority = mAuthorityProvider->getAuthority();

			// here we will set the channel by subscribing the channel with
			// the specified id, however, we will subscribe only if the channel
			// exists to prevent creating a new channel
			if(MIRA_FW.getChannelManager().hasChannel(mChannelID))
			{
				mChannel = authority.subscribe<T>(mChannelID, &ChannelProperty::callback, this);
				mDataUpdates = MIRA_FW.getChannelManager().getNrOfDataChanges(mChannelID);
			}
			else
				MIRA_THROW(XRuntime, "The channel '" << mChannelID << "' does not exist.");
			mDataChanged = true;
		}
		return mChannel;
	}

	virtual void update()
	{
		getChannel();
		if (mDataChanged && (mDataChangedFn || mDataIntervalChangedFn) )
		{
			if ( mDataChangedFn )
				mDataChangedFn(mChannel.read());
			else {
				if ( !mLastUpdate.isValid() ) {
					try {
						mLastUpdate = mChannel.read()->timestamp-Duration::milliseconds(1);
					}
					catch ( XInvalidRead& ) {
						return;
					}
				}
				ChannelReadInterval<T> tInterval = mChannel.readInterval(mLastUpdate);
				// we have to check if there actually is new data, even though mDataChanged is true:
				// it happens we already read the newest data (triggered by a previous callback),
				// then callback for the new data is executed _afterwards_ and sets mDataChanged to true
				if (!tInterval.empty())
				{
					mDataIntervalChangedFn( tInterval );
					typename ChannelReadInterval<T>::const_iterator it = tInterval.end();
					it--;
					mLastUpdate = ((ChannelRead<T>)it)->timestamp;
				}
			}
			mDataChanged = false;
		}

	}

	void callback(ChannelRead<T> channel)
	{
		++mDataUpdates;
		mDataChanged = true;
	}

public:


	/**
	 * Special property-method that should be called within the reflect method
	 * to specify a ChannelProperty. Callbacks for channel data updates or changed channel selection
	 * can be set using the methods above.
	 */
	template<typename Reflector>
	static void channelProperty(Reflector& r, const std::string& name,
			ChannelProperty& channel,
		const std::string& comment, IAuthorityProvider* authorityProvider)
	{
		r.property(name.c_str(),
			getter<ChannelPropertyProxy>(
					boost::bind(proxyGetter, &channel, authorityProvider)),
			setter<ChannelPropertyProxy>(
					boost::bind(proxySetter, &channel, _1, authorityProvider)),
			comment.c_str());
	}

private:

	/// Special setter method that takes a ChannelPropertyProxy object and sets the ChannelProperty accordingly
	static void proxySetter(ChannelProperty* This, const ChannelPropertyProxy& proxy,
							IAuthorityProvider* authorityProvider) {
		This->set(proxy.channelID, authorityProvider);
	}

	/// Special getter method that returns a ChannelPropertyProxy object for the ChannelProperty
	static ChannelPropertyProxy proxyGetter(ChannelProperty* This,
			IAuthorityProvider* authorityProvider) {
		This->mAuthorityProvider = authorityProvider;
		return ChannelPropertyProxy(This->mChannelID, This);
	}

private:

	Channel<T>  mChannel;
	bool mDataChanged;
	uint32 mDataUpdates;
	boost::function<void (ChannelRead<T>)> mDataChangedFn;
	boost::function<void (ChannelReadInterval<T>)> mDataIntervalChangedFn;
	boost::function<void ()> mChannelChangedFn;
	Time mLastUpdate;
};


///////////////////////////////////////////////////////////////////////////////

/// Special LineEdit for ChannelProperty delegate for Property Editor
class ChannelPropertyDelegateLineEdit : public QLineEdit
{
Q_OBJECT
public:
	ChannelPropertyDelegateLineEdit(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue();
public:
	TypedPropertyNode<ChannelPropertyProxy>* property;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
