/*
 * Copyright (C) 2017 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SceneQueryTool.h
 *    Base class for 3D tools querying scene
 *
 * @author Christof Schröter
 * @date   2017/02/07
 */

#ifndef _MIRA_SCENEQUERYTOOL3D_H_
#define _MIRA_SCENEQUERYTOOL3D_H_

#include <visualization/VisualizationTool3D.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/*
 * Base class for any tools that want to query the scene in a Visualization3DView
 * (in contrast to tools just affecting the camera)
 */
class MIRA_GUI_VISUALIZATION_EXPORT SceneQueryTool : public VisualizationTool3D
{
public:
	enum BackgroundMode {
		IGNORE_BACKGROUND = 0,
		PREFER_OBJECTS,
		IGNORE_OBJECTS
	};

public:
	SceneQueryTool();

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationTool3D::reflect(r);
		r.property("Background mode", mBackgroundMode,
		           "Objects only = select first object hit or nothing\n"
		           "Prefer objects = select background if no object hit\n"
		           "Background only = always select background, regardless of objects hit",
		           IGNORE_BACKGROUND,
		           PropertyHints::enumeration("Hit objects only;Prefer objects over background;Hit background only"));
		r.property("Background height", mBackgroundHeight,
		           "Height of (virtual) background plane to assume for determining coordinates clicked on background", 0.f);
	}

public:

	virtual void onMousePressed(QMouseEvent* e, float devicePixelRatio);
	virtual void onMouseReleased(QMouseEvent* e, float devicePixelRatio);
	virtual void onMouseDoubleClick(QMouseEvent* e, float devicePixelRatio);
	virtual void onMouseMoved(QMouseEvent* e, float devicePixelRatio);
	virtual void onMouseWheel(QWheelEvent * e, float devicePixelRatio);

protected:

	virtual void setupScene(IVisualization3DSite* site);

	/*
	 * Get intersection of a click into the view with objects in the view.
	 * @param x Viewport x coordinate
	 * @param y Viewport y coordinate
	 * @param mask If not empty, find only objects with name matching filter.
	 * @param exactMatch If true, name must match filter exactly.
	 *                   Otherwise, name must contain filter as substring.
	 * @param transform Apply transformVector to the coordinates
	 * @return Pair of object name and hit position. Returned name is empty if neither object nor background hit.
	 */
	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& filter = "",
	                                                      bool exactMatch = false,
	                                                      bool transform = false);

protected:

	Ogre::RaySceneQuery* mRaySceneQuery;

	BackgroundMode mBackgroundMode;
	float mBackgroundHeight;

	float mDevicePixelRatio;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
