/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ThreadLoader.C
 *    Loader plugin that parses the thread tag.
 *
 * @author Tim Langner
 * @date   2011/11/26
 */

#include <loader/Loader.h>
#include <fw/UnitManager.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin that supports grouping units in thread groups via
 * the thread tag.
 */
class ThreadLoader : public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(ThreadLoader)
public:
	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("thread");
	}

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader)
	{
		if ( loader->getContext().count("unit_id") > 0 )
			MIRA_THROW(XInvalidConfig, "Error in <thread>. "
			           "Thread groups cannot be declared inside <unit> tags");
		if ( loader->getContext().count("thread") > 0 )
			MIRA_THROW(XInvalidConfig,"Error in <thread>. "
			           "Thread groups cannot be declared inside <thread> tags");

		auto iName = node.find_attribute("name");
		if (iName == node.attribute_end())
			MIRA_THROW(XInvalidConfig, "Error in <thread>. "
			           "No name attribute specified. "
			           "Usage <thread name=\"MyThreadGroup\">...</thread>.");

		std::string ns = loader->getContext()["namespace"];
		std::string name = iName.value();

		std::string id = MIRA_FW.getUnitManager()->addThreadGroup(ns, name);
		loader->getContext()["thread"] = id;

		// parse the content
		loader->parse(node);

		// leave the thread group
		loader->getContext().erase("thread");
	}
};

///////////////////////////////////////////////////////////////////////////////
}

MIRA_CLASS_REGISTER(mira::ThreadLoader, mira::ConfigurationLoaderPlugin)
