/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceInterfaceHandler.C
 *    Implementation of ServiceInterfaceHandler.h.
 *
 * @author Tim Langner
 * @date   2012/03/28
 */

#include <fw/ServiceInterfaceHandler.h>

#include <error/Logging.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void ServiceInterfaceHandler::run(DispatcherThread* dispatcher)
{
	// handle all queued new service interfaces
	while ( !boost::this_thread::interruption_requested() && !mNewServiceQueue.empty() )
	{
		boost::mutex::scoped_lock lock(mMutex);
		std::pair<std::string, std::string> service = mNewServiceQueue.front();
		mNewServiceQueue.pop();
		lock.unlock();

		auto cbrange = mCallbacks.equal_range(service.first);
		for(auto it=cbrange.first; it!=cbrange.second; ++it)
		{
			try {
				it->second(service.first, service.second);
				if(getDiagnosticsModule())
					getDiagnosticsModule()->ok("ServiceInterfaceHandler");
			}
			catch(Exception& ex) {
				if (!getDiagnosticsModule() ||
					getDiagnosticsModule()->error("ServiceInterfaceHandler", ex.message(),
					                              "Exception while invoking service interface handler")) {
					MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
				}
			}
			catch(std::exception& ex) {
				if (!getDiagnosticsModule() ||
					getDiagnosticsModule()->error("ServiceInterfaceHandler", ex.what(),
					                              "Exception while invoking service interface handler")) {
					MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
				}
			}
		}
	}
}

void ServiceInterfaceHandler::registerCallback(const std::string& interface, Callback cb)
{
	boost::mutex::scoped_lock lock(mMutex);
	mCallbacks.insert(std::make_pair(interface, cb));
}

void ServiceInterfaceHandler::newServiceInterface(const std::string& interface,
                                                  const std::string& service)
{
	boost::mutex::scoped_lock lock(mMutex);
	mNewServiceQueue.push(std::make_pair(interface, service));
	// signal the dispatcher to process the handlers (where we are one of them)
	mDispatcher->insertRunnable(shared_from_this(),true);
}

///////////////////////////////////////////////////////////////////////////////

}
