/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteModule.C
 *    Implementation of RemoteModule.h.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#include <fw/RemoteModule.h>

#include <boost/array.hpp>

#include <communication/MulticastSender.h>
#include <stream/BinaryStream.h>
#include <thread/Atomic.h>

#include <fw/Framework.h>
#include <fw/UnitManager.h>
#include <fw/FrameworkMessage.h>
#include <fw/FrameworkDefines.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void RemoteModule::AuthSettings::clearAuth()
{
	mKeyFile.reset();
	mKey.reset();
	mPassword.reset();
}

const std::string& RemoteModule::AuthSettings::getPassword() const
{
	if(!mPassword)
		MIRA_THROW(XInvalidParameter, "No key authentication set");
	return *mPassword;
}

void RemoteModule::AuthSettings::setPassword(const boost::optional<std::string>& passwd)
{
	clearAuth();
	mPassword = passwd;
}

const RSAKey& RemoteModule::AuthSettings::getKey() const
{
	if(!mKey)
		MIRA_THROW(XInvalidParameter, "No key authentication set");
	return *mKey;
}

void RemoteModule::AuthSettings::setKey(const boost::optional<std::string>& str)
{
	clearAuth();
	std::string key = *str;
	std::string s = "PRIVATE:" + toString(key.size()/2) + ":" + key + ";";
	mKey = fromString<RSAKey>(s);
}

void RemoteModule::AuthSettings::setKeyFile(const boost::optional<std::string>& file)
{
	clearAuth();
	std::ifstream is(file->c_str());
	RSAKey key;
	is >> key;
	mKey = key;
}

///////////////////////////////////////////////////////////////////////////////


RemoteModule::RemoteModule() :
	mIncomingData(0),
	mOutgoingData(0),
	mCyclicRunnable(boost::bind(&RemoteModule::process, this),
	                Duration::milliseconds(100), "#RemoteModule"),
	mPort(0),
	mEnablePTPSync(true),
	mEnablePingTimeout(false),
	mPingInterval(Duration::seconds(1)),
	mPingTimeout(Duration::seconds(10)),
	mIOThreadCount(1),
	mExitOnDisconnect(false)
{
}

uint32 RemoteModule::getCurrentVersion()
{
	return MIRA_PROTOCOL_VERSION;
}

uint16 RemoteModule::getPort() const
{
	if (mServer)
		return mServer->getPort();
	return mPort;
}

void RemoteModule::setAuthGroup(const std::string& group)
{
	mAuthSettings.group = group;
}

void RemoteModule::setNoAuth()
{
	mAuthSettings.clearAuth();
}

void RemoteModule::setAuthPassword(const std::string& password)
{
	mAuthSettings.setPassword(password);
}

void RemoteModule::setAuthKey(const std::string& key)
{
	mAuthSettings.setKey(key);
}

void RemoteModule::setAuthKeyFile(const std::string& keyfile)
{
	mAuthSettings.setKeyFile(keyfile);
}

const RemoteModule::AuthSettings& RemoteModule::getAuthSettings() const
{
	return mAuthSettings;
}

ServiceLevel RemoteModule::getServiceLevel(const std::string& channelID)
{
	if (mChannelServiceLevels.count(channelID) == 0)
		return ServiceLevel(channelID);
	return mChannelServiceLevels[channelID];
}

void RemoteModule::start()
{
	mID = boost::uuids::random_generator()();

	// resolve all names of channels that use service level agreements
	// (can only be done here, since at the time mChannelServiceLevels
	// is deserialized the config file is not fully loaded)
	foreach(const auto& serviceLevel, mServiceLevels)
	{
		std::string channelID = MIRA_FW.getNameRegistry().resolve(serviceLevel.channelID, "/");
		mChannelServiceLevels[channelID] = serviceLevel;
		mChannelServiceLevels[channelID].channelID = channelID;
	}

	mServer.reset(new RemoteServer(mPort));
	mServer->run(mIOThreadCount, false);

	MIRA_LOG(DEBUG) << "Framework: Started remote framework server ID='"
		<< mID << "' on port " << getPort();

	mDiscoverService.reset();
	if(MIRA_CMDLINE.getOptions().count("autodiscover"))
	{
		try	
		{
			mDiscoverService.reset(new DiscoverService());
		} catch (...)
		{
			MIRA_LOG(WARNING) << "Failed to start Framework Discovery Service";
			mDiscoverService.reset();
		}
		MIRA_LOG(DEBUG) << "Auto discovery of remote frameworks is enabled";
	}
	MIRA_LOG(DEBUG) << "Starting remote frameworks cyclic runnable: " << &mCyclicRunnable;
	mThread = boost::thread(boost::bind(&CyclicRunnable::operator(), &mCyclicRunnable));

	if(mDiscoverService)
	{
		DiscoverService::AnnounceMessage data;
		data.port = getPort();
		data.id = mID;
		MulticastSender announceService(MIRA_AUTODISCOVERY_MULTICAST_ADDRESS,
		                                MIRA_AUTODISCOVERY_MULTICAST_PORT);
		announceService.send(std::string((const char*)&data, (const char*)&data+sizeof(DiscoverService::AnnounceMessage)));
	}
}

void RemoteModule::stop()
{
	if (mDiscoverService)
		mDiscoverService->stop();

	mThread.interrupt();
	mThread.join();

	if(mServer)
		mServer->stop();

	while (!mConnections.empty())
	{
		{
			boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
			mConnections.begin()->second->stop();
		}
		MIRA_SLEEP(50)
	}
}

void RemoteModule::addKnownFramework(const std::string& address)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	KnownFramework f;
	f.address = address;
	mKnownFrameworks.push_back(f);
}

bool RemoteModule::isFrameworkConnected(const std::string& frameworkID)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	typedef std::pair<const UUID, RemoteConnection*> ConnectionPair;
	foreach(ConnectionPair& connection, mConnections)
		if (connection.second->frameworkID == frameworkID)
			return true;
	return false;
}

bool RemoteModule::isConnectedTo(const std::string& address)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	typedef std::pair<const UUID, RemoteConnection*> ConnectionPair;
	foreach(ConnectionPair& connection, mConnections)
		if (connection.second->address.address == address)
			return true;
	return false;
}

void RemoteModule::disconnectFramework(const std::string& frameworkID,
                                       bool autoReconnect)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	typedef std::pair<const UUID, RemoteConnection*> ConnectionPair;
	foreach(ConnectionPair& connection, mConnections)
		if (connection.second->frameworkID == frameworkID)
		{
			connection.second->address.keep = autoReconnect;
			connection.second->stop();
			break; // make sure to break the foreach loop, since iterators became invalid
		}
}

void RemoteModule::disconnectFrom(const std::string& address,
                                  bool autoReconnect)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	typedef std::pair<const UUID, RemoteConnection*> ConnectionPair;
	foreach(ConnectionPair& connection, mConnections)
		if (connection.second->address.address == address)
		{
			connection.second->address.keep = autoReconnect;
			connection.second->stop();
			break; // make sure to break the foreach loop, since iterators became invalid
		}
}

void RemoteModule::migrateUnitToThisFramework(const std::string& id)
{
	// Unit runs already in this framework
	if (MIRA_FW.getUnitManager()->getUnit(id))
		return;

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->hasAuthority(id))
		{
			c.second->migrateUnit(id);
			break;
		}
	}
}

void RemoteModule::publishAuthority(const AuthorityDescription& authority)
{
	RemoteConnection::AuthorityDescriptions authorities;
	authorities.push_back(authority);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->publishAuthorities(authorities);
	}
}

void RemoteModule::unpublishAuthority(const AuthorityDescription& authority)
{
	RemoteConnection::AuthorityDescriptions authorities;
	authorities.push_back(authority);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->unpublishAuthorities(authorities);
	}
}

void RemoteModule::publishService(const std::string& service)
{
	// someone in our local framework has published a service
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: Publishing service to remote frameworks service='"
		<< service << "'";

	std::set<std::string> services;
	services.insert(service);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->publishServices(services);
	}
}

void RemoteModule::unpublishService(const std::string& service)
{
	// someone in our local framework has unpublished a service
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: UnPublishing service to remote frameworks service='"
		<< service << "'";

	std::set<std::string> services;
	services.insert(service);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->unpublishServices(services);
	}
}

void RemoteModule::publishChannel(const std::string& channelID, const Typename& type)
{
	// someone in our local framework has published a channel
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: Publishing channel to remote frameworks channel='"
		<< channelID << "'";

	typedef std::map<std::string,Typename> ChannelTypeMap;
	ChannelTypeMap channels;
	channels[channelID] = type;

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->publishChannels(channels);
	}
}

void RemoteModule::unpublishChannel(const std::string& channelID)
{
	// no one in our local framework has published this channel any longer
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: No more publishers for channel='"
		<< channelID << "' exist.";

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->synchronized)
			c.second->unpublishChannel(channelID);
	}
}

void RemoteModule::subscribeChannel(const std::string& channelID)
{
	// someone in our local framework has subscribed to a channel
	// search in all connected remote frameworks for a publisher of this
	// channel and subscribe there
	MIRA_LOG(DEBUG) << "Framework: Looking for remote publishers of channel '"
		<< channelID << "'";
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ServiceLevel serviceLevel = getServiceLevel(channelID);
	Buffer<uint8> slBuffer;
	BinaryBufferSerializer bs(&slBuffer);
	bs.serialize(serviceLevel);
	FrameworkMessageHeader header(slBuffer.size(), SUBSCRIBE_CHANNEL_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(slBuffer.data(), slBuffer.size())
		}};
	foreach(ConnectionMap::value_type& c, mConnections)
		if (MIRA_FW.getChannelManager().hasPublished(c.second->authority->getGlobalID(),
		                                             channelID))
			c.second->write(buffers);
}

void RemoteModule::unsubscribeChannel(const std::string& channelID)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (MIRA_FW.getChannelManager().hasPublished(c.second->authority->getGlobalID(),
		                                             channelID))
		{
			FrameworkMessageHeader header(channelID.length(), UNSUBSCRIBE_CHANNEL_MSG);
			boost::array<boost::asio::const_buffer, 2> buffers =
			{{
				header.asBuffer(),
				boost::asio::buffer(channelID)
			}};
			c.second->write(buffers);
		}
	}
}

void RemoteModule::updateIncomingStats(std::size_t size)
{
	mIncomingData += size;
}

void RemoteModule::updateOutgoingStats(std::size_t size)
{
	mOutgoingData += size;
}

std::size_t RemoteModule::getIncomingBytesPerSecond()
{
	return mIncomingStats.dataPerSecond;
}

std::size_t RemoteModule::getOutgoingBytesPerSecond()
{
	return mOutgoingStats.dataPerSecond;
}

void RemoteModule::onRemoteFrameworkDiscovered(const std::string& host, uint16 port, UUID id)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(id);
	// do we have already a connection to that framework
	if (i == mConnections.end())
	{
		MIRA_LOG(DEBUG) << "Framework: Discovered remote framework ID='"
			<< id << "' at " << host << ":" << port;
		// no we have not -> store it!
		KnownFramework f;
		f.address = MakeString() << host << ":" << port;
		f.keep = false;
		mKnownFrameworks.push_back(f);
	}
}

bool RemoteModule::onIncomingConnected(RemoteIncomingConnection* connection)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(connection->remoteID);
	// do we have already a connection to that framework
	if (i == mConnections.end())
	{
		MIRA_LOG(NOTICE) << "Framework: We have an incoming connection "
			"from remote framework at " << connection->address.address
			<< " ID='" << connection->frameworkID << "'";
		// no we have not -> store it!
		mConnections[connection->remoteID] = connection;
		connection->authority = new Authority("/", "IncomingConnection",
		                                      Authority::ANONYMOUS |
		                                      Authority::INTERNAL |
		                                      Authority::INVISIBLE_PUBLISHER_SUBSCRIBER);
		return true;
	}
	else
	{
		MIRA_LOG(DEBUG) << "Framework: We have an incoming connection "
			"from an already connected remote framework ID='"
			<< connection->frameworkID << "'";
		// yes we have -> disconnect
		return false;
	}
}

void RemoteModule::onIncomingDisconnected(RemoteIncomingConnection* connection)
{
	{
		boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
		if (!connection->remoteID.is_nil())
		{
			ConnectionMap::iterator i = mConnections.find(connection->remoteID);
			if (i == mConnections.end())
			{
				// we do not have a connection to this id
				// so put back the address to the known framework list
				if (connection->address.keep && !connection->address.address.empty())
					mKnownFrameworks.push_back(connection->address);
			}
			else
			{
				if (i->second == connection)
				{
					// we are the connection -> erase us and add address to the known framework list
					mConnections.erase(i);
					if (connection->address.keep && !connection->address.address.empty())
						mKnownFrameworks.push_back(connection->address);
					MIRA_LOG(NOTICE) << "Framework: A client from remote framework ID='"
						<< connection->frameworkID << "' has disconnected from our server";
				}
			}
		}
		else
			// put back the address to the known framework list
			if (connection->address.keep && !connection->address.address.empty())
				mKnownFrameworks.push_back(connection->address);
	}
	delete connection;

	// schedule termination if flag is set
	checkForExit();
}

bool RemoteModule::onOutgoingConnected(RemoteOutgoingConnection* connection)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(connection->remoteID);
	// do we have already a connection to that framework
	if (i == mConnections.end())
	{
		MIRA_LOG(DEBUG) << "Framework: We have connected to remote framework at "
			<< connection->address.address << " ID='" << connection->frameworkID << "'";
		// no we have not -> move from pending to connected!
		mConnections[connection->remoteID] = connection;
		connection->authority = new Authority("/", "OutgoingConnection",
		                                      Authority::ANONYMOUS |
		                                      Authority::INTERNAL |
		                                      Authority::INVISIBLE_PUBLISHER_SUBSCRIBER);
		mPendingConnections.erase(connection);
		return true;
	}
	else
	{
		MIRA_LOG(DEBUG) << "Framework: We have connected to an already "
			"connected remote framework at " << connection->address.address << " ID='"
			<< connection->frameworkID << "'";
		// yes we have -> copy the address to put it back to known framework list 
		// after disconnect
		i->second->address = connection->address;
		return false;
	}
}

void RemoteModule::onOutgoingDisconnected(RemoteOutgoingConnection* connection)
{
	{
		boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
		// remove from pending
		mPendingConnections.erase(connection);
		if (!connection->remoteID.is_nil())
		{
			ConnectionMap::iterator i = mConnections.find(connection->remoteID);
			if (i == mConnections.end())
			{
				// we do not have a connection to this id
				// so put back the address to the known framework list
				if (connection->address.keep)
					mKnownFrameworks.push_back(connection->address);
			}
			else
			{
				if (i->second == connection)
				{
					// we are the connection -> erase us and add address to the known framework list
					mConnections.erase(i);
					if (connection->address.keep)
						mKnownFrameworks.push_back(connection->address);
					MIRA_LOG(DEBUG) << "Framework: We have disconnected from a remote "
						"framework at " << connection->address.address << " ID='"
						<< connection->frameworkID << "'";
				}
			}
		}
		else
			// put back the address to the known framework list
			if (connection->address.keep)
				mKnownFrameworks.push_back(connection->address);
	}
	delete connection;

	// schedule termination if flag is set
	checkForExit();
}

void RemoteModule::process()
{
	// update statistics
	mIncomingStats.update(Time::now(), mIncomingData);
	atomic::write(&mIncomingData, 0);
	mOutgoingStats.update(Time::now(), mOutgoingData);
	atomic::write(&mOutgoingData, 0);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);

	if(!isPingTimeoutEnabled()) {
		foreach(ConnectionMap::value_type& i, mConnections)
				i.second->syncTime();
	}

	if (mKnownFrameworks.empty())
		return;

	// The framework we will try to connect to next
	KnownFramework address = mKnownFrameworks.front();
	// Remove it from list
	mKnownFrameworks.pop_front();
	// If last connection try was not at least some seconds ago add address back and
	// continue
	if ((Time::now() - address.lastConnectionTry) < Duration::seconds(2))
	{
		mKnownFrameworks.push_back(address);
		return;
	}
	// Update last connection time
	address.lastConnectionTry = Time::now();
	RemoteOutgoingConnection* connection = NULL;
	try
	{
		connection = new RemoteOutgoingConnection(address);
	}
	catch(XInvalidParameter& ex)
	{
		MIRA_LOG(ERROR) << ex.message() << " Will not try to connect again!";
		delete connection;
		// schedule termination if flag is set
		checkForExit();
		return;
	}
	catch(Exception& ex)
	{
		MIRA_LOG(WARNING) << ex.message() << " Retrying...";
		delete connection;
		// schedule termination if flag is set
		checkForExit();
		mKnownFrameworks.push_back(address);
		return;
	}
	mPendingConnections.insert(connection);
	connection->start();
}

void RemoteModule::checkForExit()
{
	if (mExitOnDisconnect)
	{
		MIRA_LOG(ERROR) << "Exiting framework after connection was closed / failed connection attempt!";
		MIRA_FW.requestTermination();
	}
}

///////////////////////////////////////////////////////////////////////////////

}
