/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <fw/Unit.h>

namespace mira { namespace fw {

///////////////////////////////////////////////////////////////////////////////

/// RPC call example unit
class ServiceCaller : public Unit
{
	/// Register at class factory
	MIRA_OBJECT(ServiceCaller)
public:

	/// Tell base class to call process in a 5100 millisecond interval
	ServiceCaller() :
		Unit(Duration::milliseconds(5100))
	{
	}

	/// Reflect method for serialization and services
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		// call base class reflect
		Unit::reflect(r);
		// publish our myMethod message
		r.method("myMethod", &ServiceCaller::myMethod, this, "",
		         "text", "input text", "hello");
	}

	/// Overwritten from MicroUnit. Gets called once on startup
	virtual void initialize()
	{
		// publish our service methods (myMethod)
		publishService(*this);
	}

	/// Overwritten from Unit. Gets called every 5100 milliseconds
	virtual void process(const Timer& timer)
	{
		std::cout << "Services that implement 'ICountSetable': " << std::endl;
		auto list = queryServicesForInterface("ICountSetable");
		foreach(const std::string& s, list)
			std::cout << s << std::endl;

		try {
			MIRA_LOG(NOTICE) << "ServiceCaller: calling";
			RPCFuture<int> result = callService<int>("IntPublisher","setCounter", 0);

			MIRA_LOG(NOTICE) << "ServiceCaller: waiting for result";
			// here we will block until the result is available:
			MIRA_LOG(NOTICE) << "Result: " << result.get();
		} catch (Exception&) {}

		try {
			MIRA_LOG(NOTICE) << "ServiceCaller: calling2";
			RPCFuture<std::string> result2 = callService<std::string,std::string>("ServiceCaller","myMethod", "Hallo");
			// here we will block until the result is available:
			MIRA_LOG(NOTICE) << "Result2: " << result2.get();
		} catch (Exception&) {}
	}

public:

	std::string myMethod(const std::string& text)
	{
		MIRA_LOG(NOTICE) << "ServiceCaller: myMethod called";
		return "<" + text + ">";
	}

};

///////////////////////////////////////////////////////////////////////////////

}}

/// Tell class factory that we are derived from Unit
MIRA_CLASS_SERIALIZATION(mira::fw::ServiceCaller, mira::Unit);
