/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ThreadMonitorTest.C
 *    Test case for thread monitor.
 *
 * @author Erik Einhorn
 * @date   2012/01/06
 */


#include <boost/test/unit_test.hpp>

#include <thread/Thread.h>
#include <thread/ThreadMonitor.h>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

void thread1()
{
	ThreadMonitor::instance().addThisThread("thread");
	MIRA_SLEEP(800);
	ThreadMonitor::instance().removeThisThread();
}

void thread2()
{
	ThreadMonitor::instance().addThisThread("thread2");
	MIRA_SLEEP(800);
	// do not call removeThisThread here for testing
}

void thread3()
{
	ThreadMonitor::instance().addThisThread("thread");
	MIRA_SLEEP(800);
	// do not call removeThisThread here for testing
}

///////////////////////////////////////////////////////////////////////////////

bool checkForThread(boost::thread::id id, const std::string& name)
{
	ThreadMonitor::ThreadInfoVector info =
			                   ThreadMonitor::instance().getThreadInformation();

	for(std::size_t i=0; i<info.size(); ++i)
	{
		if(info[i].bid == id)
			return info[i].name == name;
	}

	return false;
}

BOOST_AUTO_TEST_CASE( ThreadMonitorTest )
{
	ThreadMonitor::ThreadInfoVector info;

	info = ThreadMonitor::instance().getThreadInformation();

	BOOST_REQUIRE_EQUAL(info.size(),1);
	BOOST_CHECK_EQUAL(info[0].known,false);

	ThreadMonitor::instance().addThisThread("main");

	info = ThreadMonitor::instance().getThreadInformation();
	BOOST_REQUIRE_EQUAL(info.size(),1);
	BOOST_CHECK_EQUAL(info[0].known,true);
	BOOST_CHECK_EQUAL(info[0].name,"main");

	boost::thread t1(boost::bind(&thread1));
	MIRA_SLEEP(200);

	info = ThreadMonitor::instance().getThreadInformation();
	BOOST_REQUIRE_EQUAL(info.size(),2);
	BOOST_CHECK(checkForThread(t1.get_id(), "thread"));

	boost::thread t2(boost::bind(&thread2));
	boost::thread t3(boost::bind(&thread3));
	MIRA_SLEEP(200);

	info = ThreadMonitor::instance().getThreadInformation();
	BOOST_REQUIRE_EQUAL(info.size(),4);
	BOOST_CHECK(checkForThread(t2.get_id(), "thread2"));
	BOOST_CHECK(checkForThread(t3.get_id(), "thread"));

	// wait until the threads have stopped
	MIRA_SLEEP(800);

	info = ThreadMonitor::instance().getThreadInformation();
	BOOST_REQUIRE_EQUAL(info.size(),1);

}

