/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationPointerTest.C
 *    Test for serialization framework.
 *
 * @author Erik Einhorn
 */


#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/boost/shared_ptr.hpp>
#include <serialization/adapters/std/shared_ptr.hpp>

#include "BasicClasses.h"
#include "PolymorphicClasses.h"
#include "CommonTest.h"

using namespace std;
using namespace mira;

class ClassWithNullPointer
{
public:

	ClassWithNullPointer() : mPtr((BaseClass*)(1234)) {}
	ClassWithNullPointer(bool) : mPtr(NULL) {}

	void check() {
		BOOST_REQUIRE(mPtr==NULL);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ptr" , mPtr,  "");
	}

	BaseClass* mPtr;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithPointerToContainer
{
public:

	ClassWithPointerToContainer()
	{
		mObject = NULL;
		mPtr = NULL;
		mPtr2 = NULL;
		mPtr3 = NULL;
		mPtr4 = NULL;
	}

	ClassWithPointerToContainer(bool)
	{
		mObject = new ClassNormal(true);
		mPtr  = &mObject->vv[1];
		mPtr2 = new BaseClass(true);
		mPtr3 = new DerivedClass(true);
		mPtr4 = new BaseClass(true);
		mPolymorphics.push_back(mPtr2);
		mPolymorphics.push_back(mPtr3);
		mPolymorphics.push_back(mPtr4);
		mPolymorphicsList.push_back(mPtr2);
		mPolymorphicsList.push_back(mPtr3);
		mPolymorphicsList.push_back(mPtr4);
		mPolymorphicsSet.insert(mPtr2);
		mPolymorphicsSet.insert(mPtr3);
		mPolymorphicsSet.insert(mPtr4);
		mPolymorphicsMap[0] = mPtr2;
		mPolymorphicsMap[1] = mPtr3;
		mPolymorphicsMap[2] = mPtr4;
		mPolymorphicsMap2[mPtr2] = 0;
		mPolymorphicsMap2[mPtr3] = 1;
		mPolymorphicsMap2[mPtr4] = 2;
	}

	virtual ~ClassWithPointerToContainer()
	{
		delete mObject;
		mObject = NULL;
		mPtr = NULL;

		delete mPtr2;
		mPtr2 = NULL;
		delete mPtr3;
		mPtr3 = NULL;
		delete mPtr4;
		mPtr4 = NULL;
	}

	void check() {
		BOOST_REQUIRE(mObject!=NULL);
		mObject->check();
		BOOST_REQUIRE(mPtr!=NULL);
		BOOST_CHECK_EQUAL(mPtr, &mObject->vv[1]);

		BOOST_REQUIRE(mPtr2!=NULL);
		BOOST_CHECK_EQUAL(mPtr2->isDerived(), false);
		BOOST_REQUIRE(mPtr3!=NULL);
		BOOST_CHECK_EQUAL(mPtr3->isDerived(), true);
		BOOST_REQUIRE(mPtr4!=NULL);
		BOOST_CHECK_EQUAL(mPtr4->isDerived(), false);

		BOOST_REQUIRE(mPolymorphics.size()==3);
		auto it = mPolymorphics.begin();
		BOOST_CHECK_EQUAL(*(it++), mPtr2);
		BOOST_CHECK_EQUAL(*(it++), mPtr3);
		BOOST_CHECK_EQUAL(*it, mPtr4);

		BOOST_REQUIRE(mPolymorphicsList.size()==3);
		auto itList = mPolymorphicsList.begin();
		BOOST_CHECK_EQUAL(*(itList++), mPtr2);
		BOOST_CHECK_EQUAL(*(itList++), mPtr3);
		BOOST_CHECK_EQUAL(*itList, mPtr4);

		BOOST_CHECK_EQUAL(mPolymorphicsSet.size(), 3);
		BOOST_CHECK_EQUAL(mPolymorphicsSet.count(mPtr2), 1);
		BOOST_CHECK_EQUAL(mPolymorphicsSet.count(mPtr3), 1);
		BOOST_CHECK_EQUAL(mPolymorphicsSet.count(mPtr4), 1);

		BOOST_REQUIRE(mPolymorphicsMap.size()==3);
		auto itMap = mPolymorphicsMap.begin();
		BOOST_CHECK_EQUAL((itMap++)->second, mPtr2);
		BOOST_CHECK_EQUAL((itMap++)->second, mPtr3);
		BOOST_CHECK_EQUAL(itMap->second, mPtr4);

		BOOST_CHECK_EQUAL(mPolymorphicsMap2.size(), 3);
		BOOST_CHECK_EQUAL(mPolymorphicsMap2.count(mPtr2), 1);
		BOOST_CHECK_EQUAL(mPolymorphicsMap2.count(mPtr3), 1);
		BOOST_CHECK_EQUAL(mPolymorphicsMap2.count(mPtr4), 1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("object", mObject, "");
		r.member("ptr" , mPtr,  "");
		r.member("ptr2", mPtr2, "");
		r.member("ptr3", mPtr3, "");
		r.member("ptr4", mPtr4, "");
		r.member("polymorphics" , mPolymorphics, "");
		r.member("polymorphicslist" , mPolymorphicsList, "");
		r.member("polymorphicsset" , mPolymorphicsSet, "");
		r.member("polymorphicsmap" , mPolymorphicsMap, "");
		r.member("polymorphicsmap2" , mPolymorphicsMap2, "");
	}

private:

	vector<int>* mPtr;
	ClassNormal* mObject;

	BaseClass* mPtr2;
	BaseClass* mPtr3;
	BaseClass* mPtr4;

	vector<BaseClass*> mPolymorphics;
	list<BaseClass*> mPolymorphicsList;
	set<BaseClass*> mPolymorphicsSet;
	map<int,BaseClass*> mPolymorphicsMap;
	map<BaseClass*,int> mPolymorphicsMap2;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithDoublePointer
{
public:

	ClassWithDoublePointer()
	{
		//mPtr = NULL;
		//mPtr2 = NULL;
		mObject = NULL;
		mObject2 = NULL;
		mString = NULL;
	}

	ClassWithDoublePointer(bool)
	{
		//mPtr = new int;
		//*mPtr = 12345;
		//mPtr2 = mPtr;
		mObject = new ClassNormal(true);
		mObject2 = mObject;
		mString = &mObject->s;
	}

	virtual ~ClassWithDoublePointer()
	{
		delete mObject;
		mObject = NULL;
		mObject2 = NULL;
		mString = NULL;
	}

	void check() {
		BOOST_REQUIRE(mObject!=NULL);
		BOOST_CHECK(mObject2==mObject);
		BOOST_CHECK(mString==&mObject->s);
		mObject->check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("object", mObject, "");
		r.member("object2", mObject2, "");
		r.member("string", mString, "");
	}

private:

	ClassNormal* mObject;
	ClassNormal* mObject2;
	string* mString;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithPointerOnPointer
{
public:

	ClassWithPointerOnPointer()
	{
		mObjectPtr = NULL;
		mObjectPtrPtr = NULL;
	}

	ClassWithPointerOnPointer(bool)
	{
		mObjectPtr = new ClassNormal(true);
		mObjectPtrPtr = &mObjectPtr;
	}

	virtual ~ClassWithPointerOnPointer()
	{
		delete mObjectPtr;
		mObjectPtr = NULL;
		mObjectPtrPtr = NULL;
	}

	void check() {
		BOOST_REQUIRE(mObjectPtr!=NULL);
		BOOST_REQUIRE(mObjectPtrPtr!=NULL);
		BOOST_CHECK(mObjectPtrPtr==&mObjectPtr);
		mObjectPtr->check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ptr" ,    mObjectPtr,  "");
		r.member("ptrptr" , mObjectPtrPtr,  "");
	}

private:

	ClassNormal*  mObjectPtr;
	ClassNormal** mObjectPtrPtr;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithPointerConflict
{
public:

	ClassWithPointerConflict()
	{
		mPtr = NULL;
	}

	ClassWithPointerConflict(bool) : mObject(true)
	{
		mPtr = &mObject;
	}

	void check() {
		mObject.check();
		BOOST_CHECK_EQUAL(mPtr, &mObject);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ptr",   mPtr, "");
		r.member("value", mObject, "");
	}

private:

	ClassNormal* mPtr;
	ClassNormal  mObject;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithoutPointerConflict
{
public:

	ClassWithoutPointerConflict()
	{
		mPtr = NULL;
	}

	ClassWithoutPointerConflict(bool) : mObject(true)
	{
		mPtr = &mObject;
	}

	ClassWithoutPointerConflict(const ClassWithoutPointerConflict& other) : mObject(other.mObject)
	{
		mPtr = NULL;
		if (other.mPtr == &other.mObject)
			mPtr = &mObject;
	}

	ClassWithoutPointerConflict& operator=(const ClassWithoutPointerConflict& other)
	{
		mObject = other.mObject;
		mPtr = NULL;
		if (other.mPtr == &other.mObject)
			mPtr = &mObject;

		return *this;
	}

	void check() {
		mObject.check();
		BOOST_CHECK_EQUAL(mPtr, &mObject);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("value", mObject, "");
		r.member("ptr",   mPtr, "");
	}

private:

	ClassNormal* mPtr;
	ClassNormal  mObject;
};

//////////////////////////////////////////////////////////////////////////////

class ClassSharedPtr
{
public:

	ClassSharedPtr()
	{
	}

	ClassSharedPtr(bool)
	{
		mPtr1.reset(new ClassNormal(true));
		mPtr2 = mPtr1;

		mConstPtr.reset(new ClassNormal(true));
	}

	void check() {
		//mObject.check();
		mPtr1->check();
		BOOST_CHECK_EQUAL(mPtr1.get(), mPtr2.get());

		// +1 since one shared instance is held within the Deserializer
		BOOST_CHECK_EQUAL(mPtr1.use_count(), 2+1);

		mConstPtr->check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		//r.member("value", mObject, "");
		r.member("ptr1",   mPtr1, "");
		r.member("ptr2",   mPtr2, "");
		r.member("constPtr", mConstPtr, "");
	}

private:

	//ClassNormal  mObject;
	boost::shared_ptr<ClassNormal> mPtr1;
	boost::shared_ptr<ClassNormal> mPtr2;
	boost::shared_ptr<ClassNormal const> mConstPtr;
};

//////////////////////////////////////////////////////////////////////////////

class ClassSharedRecursion2;
typedef std::shared_ptr<ClassSharedRecursion2> ClassSharedRecursion2Ptr;

class ClassSharedRecursion1;
typedef std::shared_ptr<ClassSharedRecursion1> ClassSharedRecursion1Ptr;

class ClassSharedRecursion1
{
public:
	ClassSharedRecursion1() {

	}

	ClassSharedRecursion1( ClassSharedRecursion2Ptr ptr ) : mPtr( ptr ) {

	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ptr",   mPtr, "");
	}

	ClassSharedRecursion2Ptr mPtr;
};

class ClassSharedRecursion2 : public std::enable_shared_from_this<ClassSharedRecursion2>
{
public:

	ClassSharedRecursion2()
	{

	}

	ClassSharedRecursion2(bool)
	{
	}

	void setup() {
		ClassSharedRecursion1Ptr p1 = ClassSharedRecursion1Ptr( new ClassSharedRecursion1( thisPtr() ) );
		mMap[ p1 ] = 123;
	}

	void check() {
		BOOST_CHECK(!mMap.empty());
		BOOST_CHECK(mMap.begin()->first->mPtr.get() == this);
	}

	ClassSharedRecursion2Ptr thisPtr()
	{
		return shared_from_this();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("map1",   mMap, "");
	}

private:

	std::map<ClassSharedRecursion1Ptr, int> mMap;
};

class ClassSharedRecursionWrapper
{
public:
	ClassSharedRecursionWrapper()
	{

	}

	ClassSharedRecursionWrapper(bool)
	{
		mPtr = ClassSharedRecursion2Ptr( new ClassSharedRecursion2() );
		mPtr->setup();
	}

	void check() {
		BOOST_CHECK(mPtr.get()!=NULL);
		mPtr->check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ptr", mPtr, "");
	}

private:
	ClassSharedRecursion2Ptr mPtr;
};

//////////////////////////////////////////////////////////////////////////////

class ClassWithMemberClassWithoutPointerConflictBase
{
public:
	ClassWithMemberClassWithoutPointerConflictBase()
	{
	}

	ClassWithMemberClassWithoutPointerConflictBase(bool) : mMember(true)
	{
	}

	void check() {
		mMember.check();
	}

protected:
	ClassWithoutPointerConflict mMember;
};

template<bool GetterSetter = false>
class ClassWithMemberClassWithoutPointerConflict : public ClassWithMemberClassWithoutPointerConflictBase
{
public:
	ClassWithMemberClassWithoutPointerConflict<GetterSetter>()
	{
	}

	ClassWithMemberClassWithoutPointerConflict<GetterSetter>(bool flag)
		: ClassWithMemberClassWithoutPointerConflictBase(flag)
	{
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Member", mMember, "");
	}
};

template<>
class ClassWithMemberClassWithoutPointerConflict<true> : public ClassWithMemberClassWithoutPointerConflictBase
{
public:

	ClassWithMemberClassWithoutPointerConflict<true>()
	{
	}

	ClassWithMemberClassWithoutPointerConflict<true>(bool)
		: ClassWithMemberClassWithoutPointerConflictBase(true)
	{
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Member",
		         getter(&ClassWithMemberClassWithoutPointerConflict<true>::getMember, this),
		         setter(&ClassWithMemberClassWithoutPointerConflict<true>::setMember, this),
		         "");
	}

private:
	const ClassWithoutPointerConflict& getMember() const { return mMember; }
	void setMember(const ClassWithoutPointerConflict& member) { mMember = member; }
};

typedef ClassWithMemberClassWithoutPointerConflict<false> ClassWithTrackingWithoutGetterSetter;
typedef ClassWithMemberClassWithoutPointerConflict<true> ClassWithTrackingAndGetterSetter;

//////////////////////////////////////////////////////////////////////////////

// this does not even require pointer tracking, but ClassNormal is (involuntarily)
// tracked because some other classes serialize shared_ptr<ClassNormal>
// (behaviour depends on other classes (test 1-6) being instantiated!)
class ClassWithTrackingAndGetterSetterSimple
{
public:

	ClassWithTrackingAndGetterSetterSimple()
	{
	}

	ClassWithTrackingAndGetterSetterSimple(bool) : mMember(true)
	{
	}

	void check() {
		mMember.check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Member",
		         getter(&ClassWithTrackingAndGetterSetterSimple::getMember, this),
		         setter(&ClassWithTrackingAndGetterSetterSimple::setMember, this),
		         "");
	}

protected:
	const ClassNormal& getMember() const { return mMember; }
	void setMember(const ClassNormal& member) { mMember = member; }

private:
	ClassNormal mMember;
};

class ClassWithTwoMembersWithTrackingAndGetterSetter
{
public:

	ClassWithTwoMembersWithTrackingAndGetterSetter()
	{
	}

	ClassWithTwoMembersWithTrackingAndGetterSetter(bool)
		: mMember1(true), mMember2(true)
	{
	}

	void check() {
		mMember1.check();
		mMember2.check();
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Member1", mMember1, "");
		r.member("Member2", mMember2, "");
	}

protected:
	ClassWithTrackingAndGetterSetterSimple mMember1;
	ClassWithTrackingAndGetterSetterSimple mMember2;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( TestClassWithNullPointer )
{
	testAll<ClassWithNullPointer>("Pointer", 0);
}


BOOST_AUTO_TEST_CASE( TestClassWithPointerToContainer )
{
	testAll<ClassWithPointerToContainer>("Pointer", 1);
}

// Test some special cases (mostly pointer stuff)

BOOST_AUTO_TEST_CASE( TestClassWithDoublePointer )
{
	testAll<ClassWithDoublePointer>("Pointer", 2);
}

/*
// Will generate a compiler error, since pointers on pointers
// are not supported yet
BOOST_AUTO_TEST_CASE( XmlClassWithPointerOnPointer )
{
	testAll<ClassWithPointerOnPointer>("Pointer", 3);
}
*/

BOOST_AUTO_TEST_CASE( TestClassWithoutPointerConflict )
{
	testAll<ClassWithoutPointerConflict>("Pointer", 4);
}

BOOST_AUTO_TEST_CASE( TestClassWithPointerConflict )
{
	BOOST_CHECK_THROW(testAll<ClassWithPointerConflict>("Pointer", 5), XIO);
}

BOOST_AUTO_TEST_CASE( TestClassSharedPtr )
{
	testAll<ClassSharedPtr>("Pointer", 6);
}

BOOST_AUTO_TEST_CASE( TestSpecialSharedPtr )
{
	testAll<ClassSharedRecursionWrapper>("Pointer", 7);
}

BOOST_AUTO_TEST_CASE( TestMemberClassWithPointerTracking )
{
	testAll<ClassWithTrackingWithoutGetterSetter>("Pointer", 8);
}

BOOST_AUTO_TEST_CASE( TestMemberClassWithPointerTrackingAndGetterSetter )
{
	testAll<ClassWithTrackingAndGetterSetter>("Pointer", 9);
}

BOOST_AUTO_TEST_CASE( TestTwoMembersWithPointerTrackingAndGetterSetter )
{
	testAll<ClassWithTwoMembersWithTrackingAndGetterSetter>("Pointer", 10);
}
