/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */


#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>

#include <math/NormalRandomGenerator.h>
#include <math/UniformRandomGenerator.h>

#include <math/IncrementalMoments.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( IncrementalMomentsTest1D )
{
	NormalRandomGenerator<1> rnd(3.0f);

	IncrementalMoments<float,1,1> m1; // order 1: mean only
	IncrementalMoments<float,1,2> m2; // order 2: mean and variance

	for(int i=0; i<10000; ++i)
	{
		float x = rnd() + 3.0f;
		m1.insert(x);
		m2.insert(x);
	}

	std::cout << "m1 mean: " << m1.mean() << std::endl;
	std::cout << "m2 mean: " << m2.mean() << std::endl;
	std::cout << "m2 var:  " << m2.var() << std::endl;

	BOOST_CHECK_CLOSE(m1.mean(),3.0f,1.0f);
	BOOST_CHECK_CLOSE(m2.mean(),3.0f,1.0f);
	BOOST_CHECK_CLOSE(m2.var(),9.0f,1.0f);
}


BOOST_AUTO_TEST_CASE( IncrementalMomentsTest2D )
{
	Eigen::Matrix2f sigma;
	sigma << 2.0f,1.0f,
			 1.0f,3.0f;

	NormalRandomGenerator<2,float> rnd(sigma);

	IncrementalMoments<float,2,1> m1; // order 1: mean only
	IncrementalMoments<float,2,2> m2; // order 2: mean and variance

	for(int i=0; i<10000; ++i)
	{
		Eigen::Vector2f x = rnd() + Eigen::Vector2f(2.0f,4.0f);
		m1.insert(x);
		m2.insert(x);
	}

	std::cout << "m1 mean: " << m1.mean() << std::endl;
	std::cout << "m2 mean: " << m2.mean() << std::endl;
	std::cout << "m2 cov:  " << m2.cov() << std::endl;

	BOOST_CHECK_CLOSE(m1.mean().x(),2.0f,1.0f);
	BOOST_CHECK_CLOSE(m1.mean().y(),4.0f,1.0f);

	BOOST_CHECK_CLOSE(m2.mean().x(),2.0f,1.0f);
	BOOST_CHECK_CLOSE(m2.mean().y(),4.0f,1.0f);

	BOOST_CHECK_CLOSE(m2.cov()(0,0),2.0f,1.0f);
	BOOST_CHECK_CLOSE(m2.cov()(0,1),1.0f,1.0f);
	BOOST_CHECK_CLOSE(m2.cov()(1,1),3.0f,1.0f);
}
