/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryJSONConverter.C
 *    Implementation of BinaryJSONConverter.h.
 *
 * @author Erik Einhorn
 * @date   2012/10/29
 */

#include <serialization/BinaryJSONConverter.h>
#include <serialization/BinarySerializer.h>

namespace mira {
namespace BinaryJSONConverter {

///////////////////////////////////////////////////////////////////////////////

template <typename T, typename U>
static void atomicToJSONHelper(BinaryBufferIstream& stream, JSONValue& oObject)
{
	T v;
	stream >> v;
	oObject = JSONValue((U)v);
}

static void atomicToJSON(BinaryBufferIstream& stream, const TypeMeta& type, JSONValue& oObject)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  atomicToJSONHelper<uint8,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT8:   atomicToJSONHelper<int8,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT16: atomicToJSONHelper<uint16,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT16:  atomicToJSONHelper<int16,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT32: atomicToJSONHelper<uint32,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT32:  atomicToJSONHelper<int32,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT64: atomicToJSONHelper<uint64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT64:  atomicToJSONHelper<int64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_FLOAT:  atomicToJSONHelper<float,double>(stream,oObject); break;
		case TypeMeta::TYPE_DOUBLE: atomicToJSONHelper<double,double>(stream,oObject); break;
		case TypeMeta::TYPE_BOOL:   atomicToJSONHelper<bool,bool>(stream,oObject); break;
		case TypeMeta::TYPE_STRING: atomicToJSONHelper<std::string,std::string>(stream,oObject); break;
		case TypeMeta::TYPE_ENUMERATION:  atomicToJSONHelper<int,int>(stream,oObject); break;
		case TypeMeta::TYPE_VERSION: atomicToJSONHelper<int,int>(stream,oObject); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject);

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject);

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject);

static void binaryToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToJSON(stream,type,metadb, oObject);
			break;
		default:
			atomicToJSON(stream,type,oObject);
			break;
		}
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");

	oObject = JSONObject();

	CompoundMetaPtr compound = it->second;
	foreach(const CompoundMeta::Member& m, compound->members)
		binaryToJSON(stream,*m.type,metadb,oObject.get_obj()[m.name]);
}

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	oObject = JSONArray();
	JSONArray& a = oObject.get_array();

	uint32 count;
	stream >> count;

	a.resize(count);

	for(uint32 i=0; i<count; ++i)
		binaryToJSON(stream,nestedType,metadb,a[i]);
}

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	uint8 pointerType8U;
	stream >> pointerType8U;

	BinarySerializerMixin::PointerType pointerType = (BinarySerializerMixin::PointerType) pointerType8U;

	switch(pointerType)
	{
		case BinarySerializerMixin::NULL_POINTER:
			oObject = JSONValue();
			break;
		case BinarySerializerMixin::POINTER_REFERENCE:
		{
			int ref;
			stream >> ref; // read the ref id
			// ... and resolve it
			//pointer = this->template resolveReference<T>(ref);
			// TODO: Oh god!
			MIRA_THROW(XIO, "Not implemented yet");
			break;
		}

		case BinarySerializerMixin::NORMAL_POINTER:
			binaryToJSON(stream,unqualifiedType,metadb,oObject);
			break;

		case BinarySerializerMixin::POLYMORPHIC_POINTER: {
			// we have a "polymorphic" pointer, so get class type
			std::string typenam;
			stream >> typenam;

			// we can convert a polymorphic pointer only, if we have the
			// typename in our type database
			auto it = metadb.find(typenam);
			if(it==metadb.end())
				MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

			assert(!unqualifiedType.hasQualifier());
			TypeMeta type;
			type.setClassType(typenam);
			binaryToJSON(stream,type,metadb,oObject);
			oObject.get_obj()["@class"] = json::Value(typenam);
			break;
		}
	}
}

void binaryToJSON(const Buffer<uint8>& data, bool containsTypename,
                  const TypeMeta& type, const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	BinaryBufferIstream stream((Buffer<uint8>*)&data);
	if(containsTypename) {
		std::string typenam;
		stream >> typenam;
	}

	binaryToJSON(stream,type,metadb,oObject);
}

//////////////////////////////////////////////////////////////////////////////

static void atomicToBinary(const JSONValue& iObject, const TypeMeta& type,
                           BinaryBufferOstream& stream)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  stream << (uint8)iObject.get_int(); break;
		case TypeMeta::TYPE_INT8:   stream << (int8)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT16: stream << (uint16)iObject.get_int(); break;
		case TypeMeta::TYPE_INT16:  stream << (int16)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT32: stream << (uint32)iObject.get_int(); break;
		case TypeMeta::TYPE_INT32:  stream << (int32)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT64: stream << (uint64)iObject.get_int64(); break;
		case TypeMeta::TYPE_INT64:  stream << (int64)iObject.get_int64(); break;
		case TypeMeta::TYPE_FLOAT:  stream << (float)iObject.get_real(); break;
		case TypeMeta::TYPE_DOUBLE: stream << iObject.get_real(); break;
		case TypeMeta::TYPE_BOOL:   stream << iObject.get_bool(); break;
		case TypeMeta::TYPE_STRING: stream << iObject.get_str(); break;
		case TypeMeta::TYPE_ENUMERATION:  stream << iObject.get_int(); break;
		case TypeMeta::TYPE_VERSION: stream << iObject.get_int(); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream, int defaultVersion = -1);

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream, int defaultVersion = -1);

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream, int defaultVersion = -1);

static void JSONToBinary(const JSONValue& iObject, const TypeMeta& type,
                         const MetaTypeDatabase& metadb, BinaryBufferOstream& stream,
                         int defaultVersion = -1)
{
	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToBinary(iObject,unqualifiedType,metadb,stream,defaultVersion);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToBinary(iObject,unqualifiedType,metadb,stream,defaultVersion);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToBinary(iObject,type,metadb, stream,defaultVersion);
			break;
		default:
			atomicToBinary(iObject,type,stream);
			break;
		}
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream, int defaultVersion)
{
	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");

	CompoundMetaPtr compound = it->second;
	foreach(const CompoundMeta::Member& m, compound->members)
	{
		auto f = iObject.get_obj().find(m.name);
		if ((f == iObject.get_obj().end()) && (m.name == "@version") && (defaultVersion >= 0)) {
			MIRA_LOG(WARNING) << "JSONToBinary: '@version' member not found in JSON object. "
			                     "Setting default version " << defaultVersion << ".";
			stream << defaultVersion;
			continue;
		}
		const JSONValue& v = f->second;
		JSONToBinary(v,*m.type,metadb,stream,defaultVersion);
	}
}

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream, int defaultVersion)
{
	const JSONArray& a = iObject.get_array();

	uint32 count = a.size();
	stream << count;

	for(uint32 i=0; i<count; ++i)
		JSONToBinary(a[i],nestedType,metadb,stream,defaultVersion);
}

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream, int defaultVersion)
{
	// null pointer
	if (iObject.is_null())
	{
		stream << (uint8)BinarySerializerMixin::NULL_POINTER;
		return;
	}
	// polymorphic pointer
	if (iObject.type() == json_spirit::obj_type && iObject.get_obj().count("@class"))
	{
		stream << (uint8)BinarySerializerMixin::POLYMORPHIC_POINTER;
		std::string typenam = iObject.get_obj().find("@class")->second.get_str();
		stream << typenam;
		auto it = metadb.find(typenam);
		if(it==metadb.end())
			MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

		assert(!unqualifiedType.hasQualifier());
		TypeMeta type;
		type.setClassType(typenam);
		JSONToBinary(iObject,type,metadb,stream,defaultVersion);
		return;
	}

	// TODO reference pointers

	// if we reach here we have a normal pointer
	stream << (uint8)BinarySerializerMixin::NORMAL_POINTER;
	JSONToBinary(iObject, unqualifiedType, metadb, stream,defaultVersion);
}

void JSONToBinary(const JSONValue& iObject, const TypeMeta& type,
                  const MetaTypeDatabase& metadb, Buffer<uint8>& oData,
                  bool includeTypename, int defaultVersion)
{
	BinaryBufferOstream stream(&oData);
	if (includeTypename)
		stream << type.getTypename();

	JSONToBinary(iObject, type, metadb, stream,defaultVersion);
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace

