/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Base64.C
 *    Implementation of Base64.h.
 *
 * @author Tim Langner
 * @date   2011/12/06
 */

#include <security/Base64.h>

#include <openssl/bio.h>
#include <openssl/buffer.h>
#include <openssl/evp.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::string base64Encode(const std::string& data)
{
	Buffer<uint8> inBuf((uint8*)data.data(), data.size());
	Buffer<uint8> outBuf = base64Encode(inBuf);
	return std::string(outBuf.data(), outBuf.data()+outBuf.size());
}

Buffer<uint8> base64Encode(const Buffer<uint8>& data)
{
	BIO *bmem, *b64;
	BUF_MEM *bptr;

	b64 = BIO_new(BIO_f_base64());
	bmem = BIO_new(BIO_s_mem());
	b64 = BIO_push(b64, bmem);
	BIO_write(b64, data.data(), data.size());
	BIO_flush(b64);
	BIO_get_mem_ptr(b64, &bptr);

	Buffer<uint8> r(bptr->length);
	memcpy(r.data(), bptr->data, bptr->length);
	BIO_free_all(b64);
	return r;
}

std::string base64Decode(const std::string& base64)
{
	Buffer<uint8> inBuf((uint8*)base64.data(), base64.size());
	Buffer<uint8> outBuf = base64Decode(inBuf);
	return std::string(outBuf.data(), outBuf.data()+outBuf.size());
}

Buffer<uint8> base64Decode(const Buffer<uint8>& base64)
{
	BIO *bmem, *b64;

	std::size_t inLen = base64.size();
	std::size_t outMaxLen = (inLen * 6 + 7) / 8;
	Buffer<uint8> buf(outMaxLen);

	b64 = BIO_new(BIO_f_base64());
	bmem = BIO_new_mem_buf((char*) base64.data(), inLen);
	b64 = BIO_push(b64, bmem);
	std::size_t outLen = BIO_read(b64, buf.data(), outMaxLen);
	BIO_free_all(b64);
	buf.resize(outLen);
	return buf;
}

///////////////////////////////////////////////////////////////////////////////

}
