/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DistanceLUT.C
 *    Implementation of DistanceLUT.h.
 *
 * @author Erik Einhorn
 * @date   2009/09/28
 */

#include <geometry/DistanceLUT.h>
#include <error/Logging.h>

#include <iostream>
#include <math.h>
#include <assert.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// computes the Euclidean distance of point px,py to the cell defined by
// (minx, miny) and (maxx, maxy).
inline static void distancesToCell(float px, float py,
                                   float minx, float miny,
                                   float maxx, float maxy,
                                   float& oMinDist, float& oMaxDist)
{
	// get point on or in the rectangle that is the closest to px,py
	float nx = px;
	float ny = py;

	if(nx<minx)      nx = minx;
	else if(nx>maxx) nx = maxx;

	if(ny<miny)      ny = miny;
	else if(ny>maxy) ny = maxy;

	// get min distance
	// (no need for special handling of points inside of the rectangle)
	oMinDist = hypotf(nx-px, ny-py);

	// determine the rectangle corner with the max distance
	if(py < (miny+maxy)*0.5f) {
		if(px < (minx+maxx)*0.5f) {
			nx = maxx;
			ny = maxy;
		} else {
			nx = minx;
			ny = maxy;
		}
	} else {
		if(px < (minx+maxx)*0.5f) {
			nx = maxx;
			ny = miny;
		} else {
			nx = minx;
			ny = miny;
		}
	}

	// get max distance
	oMaxDist =  hypotf(nx-px, ny-py);
}

///////////////////////////////////////////////////////////////////////////////

DistanceLUT::DistanceLUT() : mWidth(0), mSubdivisions(0), mLayers(0), mSize(0)
{
	// create a lut with some default values
	computeLUT(1000, 5);
}

DistanceLUT::~DistanceLUT()
{
	freeLUT();
}

void DistanceLUT::createLUT(int width, int subdivisions)
{
	assert(width>0 && subdivisions>0);

	mWidth        = width;
	mSubdivisions = subdivisions;

	mSize         = mWidth*mWidth;
	mLayers       = (mSubdivisions+1)*(mSubdivisions+1);

	mLUT.resize(mLayers);

	for(int i=0; i<mLayers;++i)
		mLUT[i].reset(new CellDist[mSize]);
}

void DistanceLUT::freeLUT()
{
	mLUT.clear();
}

void DistanceLUT::computeLUT(int width, int subdivisions)
{
	boost::mutex::scoped_lock lock(mMutex);

	if(width<mWidth && subdivisions<mSubdivisions)
		return; // no need to do anything

	freeLUT();
	createLUT(width, subdivisions); // sets width and subdivisions

	MIRA_LOGTIMER(DEBUG, DistanceLUT, "Preparing...")

	// lower left point of the region
	int x0 = -mWidth/2;
	int y0 = -mWidth/2;

	// Subdivisions are done within the central 1x1 cell, which is subdivided
	// into bins. Each bin is represented by a single point that is located at
	// the lower left corner of the bin. For each cell in the LUT (that is
	// created for the current bin) the distances to that point are computed.
	// the LUT consists of cells and reaching from -width/2 ... +width/2.
	int layer=0;
	for(int suby=0; suby<=mSubdivisions; ++suby)
	for(int subx=0; subx<=mSubdivisions; ++subx, ++layer)
	{
		// the pointer to the LUT for the current bin (layer)
		CellDist* buffer = mLUT[layer].get();

		// the coordinates of the point that represents the current bin
		// (lower left corner)
		float dx = 0.0f;
		float dy = 0.0f;

		dx = subx / (float)mSubdivisions;
		dy = suby / (float)mSubdivisions;

		// now fill the layer (with idx)
		int idx=0;
		for(int y=0; y<mWidth; ++y)
		{
			// region cell y-coordinates
			float miny = (float)(y0+y);
			float maxy = miny + 1.0f;

			for(int x=0; x<mWidth; ++x, ++idx)
			{
				// region cell x-coordinates
				float minx = (float)(x0+x);
				float maxx = minx + 1.0f;
				// the distance from the point (dx,dy) to each region cell
				// (minx,miny,maxx,maxy)
				distancesToCell(dx, dy, minx, miny, maxx, maxy,
				                buffer[idx].minDist,
				                buffer[idx].maxDist);
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void DistanceLUT::ensureRegion(const Rect2i& region)
{
	// make sure that the LUT has a sufficient size
	if(region.minCorner.x() <= -mWidth/2 || region.maxCorner.x() >= mWidth/2 ||
	   region.minCorner.y() <= -mWidth/2 || region.maxCorner.y() >= mWidth/2)
	{
		int width = mWidth*2;
		while(region.minCorner.x() <= -width/2 || region.maxCorner.x() >= width/2 ||
		      region.minCorner.y() <= -width/2 || region.maxCorner.y() >= width/2)
			width*=2;

		computeLUT(width, mSubdivisions);
	}
}


DistanceLUT::Region DistanceLUT::getRegion(const Point2f& d,
                                           const Rect2i& region)
{
	assert(d.x()>=0.0f && d.x()<=1.0f);
	assert(d.y()>=0.0f && d.y()<=1.0f);

	ensureRegion(region);

	// lock after ensureRegion(), as the above call may also lock the mutex
	boost::mutex::scoped_lock lock(mMutex);

	int width_2 = mWidth/2;

	// get sub-cell index
	int subx = (int)(d.x() * (float)mSubdivisions);
	int suby = (int)(d.y() * (float)mSubdivisions);

	int layer = suby * mSubdivisions + subx;
	assert(layer>=0 && layer<mLayers);

	CellDist* data = mLUT[layer].get() + width_2*mWidth+width_2;

	return Region(mLUT[layer], data, mWidth, region);
}

///////////////////////////////////////////////////////////////////////////////

}
