/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Singleton.h
 *    A singleton class that can be freely configured using policies that
 *    control the creation, instantiation, lifetime and thread-safety.
 *
 * @author Erik Einhorn
 * @date   2011/12/23
 */

#ifndef _MIRA_SINGLETON_H_
#define _MIRA_SINGLETON_H_

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#include <boost/type_traits/remove_cv.hpp>
#include <boost/thread/mutex.hpp>
#endif

#include <error/Exceptions.h>
#include <platform/Platform.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

/**
 * @ingroup SingletonModule
 * Helper class, that provides methods for getting and setting singleton
 * pointers for a specific type into a SINGLE and UNIQUE map. This is necessary
 * for Windows, where each dynamic library gets its own instance of static
 * variables. Hence, without this helper the singleton code would produce a
 * separate singleton per library. Under Linux the two methods of this helper
 * do nothing.
 */
class MIRA_BASE_EXPORT SingletonSharedLibraryHelper
{
public:

#ifdef MIRA_WINDOWS
	static void* getSharedLibrarySingleton(const std::type_info& type);
	static void  setSharedLibrarySingleton(const std::type_info& type, 
	                                       void* singleton);
#else
	static void* getSharedLibrarySingleton(const std::type_info& type)
	{
		return NULL;
	}
	static void  setSharedLibrarySingleton(const std::type_info& type, 
	                                       void* singleton)
	{}
#endif
};

/// type used in std::atexit as function pointer
typedef void (*atexit_lifetime_fn) (void);

/**
 * @ingroup SingletonModule
 * This class contains all types that describe the singleton (the
 * singleton class and all policy). It simplifies the hand over of all these
 * types between the template classes.
 */
template <typename T,
          template <class> class I,
          template <class> class C,
          template <class> class L,
          class M>
struct SingletonTrait
{
	// type of singleton class without const and volatile qualifiers
	typedef typename boost::remove_cv<T>::type Type;
	typedef I<SingletonTrait> InstantiationPolicy;
	typedef C<Type> CreationPolicy;
	typedef L<Type> LifetimePolicy;
	typedef M LockPolicy;
};

/**
 * @ingroup SingletonModule
 * The base class of all singletons. It provides the static members that take
 * the instance pointer and a flag if the singleton was already destroyed.
 * Moreover this class "unpacks" the types from the SingletonTrait.
 */
template <typename Trait>
class SingletonBase : public SingletonSharedLibraryHelper, boost::noncopyable
{
public:
	typedef typename Trait::InstantiationPolicy InstantiationPolicy;
	typedef typename Trait::CreationPolicy CreationPolicy;
	typedef typename Trait::LifetimePolicy LifetimePolicy;
	typedef typename Trait::LockPolicy LockPolicy;
	typedef typename Trait::Type Type;
	typedef typename LockPolicy::template Volatile<Type>::type* TypePtr;

protected:

	static TypePtr sInstance;
	static bool sDestroyed;

};

// the static member sInstance
template <typename Trait>
typename SingletonBase<Trait>::TypePtr SingletonBase<Trait>::sInstance = NULL;

// the static member sDestroyed
template <typename Trait>
bool SingletonBase<Trait>::sDestroyed = false;

} // namespace Private
///@endcond INTERNAL


namespace singleton {

///////////////////////////////////////////////////////////////////////////////
// Creation policies

/**
 * @ingroup SingletonModule
 * Implementation of the CreationPolicy that is used by the Singleton template.
 * This policy uses the new/delete operators for creating/destructing the
 * singleton object.
 * @note The class you want to create as a singleton needs a public default
 *       constructor. In case of a private constructor you need to add this
 *       struct as a friend. 
 */
template <typename T>
struct CreateUsingNew {
	static T* create() { return new T; }
	static void destroy(T* p) { delete p; }
};

/**
 * @ingroup SingletonModule
 * Implementation of the CreationPolicy that is used by the Singleton template.
 * This policy creates the singleton object in static memory, i.e. without
 * allocating memory. Instead, the memory is provided by the compiler at compile
 * time.
 * @note The class you want to create as a singleton needs a public default
 *       constructor. In case of a private constructor you need to add this
 *       struct as a friend. 
 */
template <typename T>
struct CreateStatic {

#ifdef _MSC_VER
#  define MIRA_CREATESTATIC_ALIGN __declspec(align(16))
#else
#  define MIRA_CREATESTATIC_ALIGN __attribute__((aligned(16)))
#endif

	static T* create() {
		// create 16 byte aligned static memory block
		static MIRA_CREATESTATIC_ALIGN char staticMemory[sizeof(T)];
		return new(&staticMemory) T;
	}
	static void destroy(T* p) {
		p->~T();
	}
#undef MIRA_CREATESTATIC_ALIGN
};

/**
 * @ingroup SingletonModule
 * Implementation of the CreationPolicy that is used by the Singleton template.
 * This policy creates the singleton object using a custom allocator, e.g.:
 * \code
 *     Singleton<...,CreateUsing<std::allocator>::Allocator, ...>
 * \endcode
 * @note The class you want to create as a singleton needs a public default
 *       constructor. In case of a private constructor you need to add this
 *       struct as a friend. 
 */
template <template <class> class Alloc>
struct CreateUsing {
	template <typename T>
	struct Allocator {
		static Alloc<T> alloc;
		static T* create() { return new (alloc.allocate(1)) T; }
		static void destroy(T* p) {
			p->~T();
			alloc.deallocate(p,1);
		}
	};
};

///////////////////////////////////////////////////////////////////////////////
// Lifetime Policies

/**
 * @ingroup SingletonModule
 * Implementation of the LifetimePolicy that is used by the Singleton template.
 * This policy implements the normal C++ lifetime, i.e. singletons that are
 * created first, are destroyed last.
 */
template <typename T>
struct NormalLifetime
{
	static void scheduleDestruction(T*, Private::atexit_lifetime_fn fun) {
		std::atexit(fun);
	}
};

/**
 * @ingroup SingletonModule
 * Implementation of the LifetimePolicy that is used by the Singleton template.
 * This policy NEVER destroys the singleton object, i.e. the singletons
 * destructor will never be called ("leaky singleton").
 */
template <typename T>
struct NoDestroyLifetime
{
	static void scheduleDestruction(T*, Private::atexit_lifetime_fn fun) {}
};

///////////////////////////////////////////////////////////////////////////////
// Instantiation policies

/**
 * @ingroup SingletonModule
 * Implementation of the InstantiationPolicy that is used by the Singleton template.
 * This policy ensures lazy instantiation, i.e. the singleton object will be
 * instantiated at the moment when the instance() method is called for the
 * first time. If the instance() method is never called, the singleton will
 * never be instantiated.
 */
template <typename Trait>
class LazyInstantiation : public Private::SingletonBase<Trait>
{
	typedef Private::SingletonBase<Trait> Base;

public:
	typedef typename Base::Type Type;

	static Type& instance() {
		if(!Base::sInstance)
			makeInstance();
		return *((Type*)Base::sInstance);
	}

protected:

	static Type* makeInstance()
	{
		// according to the standard, the creation of static variables is thread-safe
		static typename Base::LockPolicy::Mutex sMutex;

		// protect the instantiation of a new singleton object
		typename Base::LockPolicy::Lock lock(sMutex);

		// together with the check in instance() this is the
		// double-checked locking pattern which is safe, if sInstance is
		// declared volatile
		if(!Base::sInstance) {
			if(Base::sDestroyed)
				MIRA_THROW(XLogical, "Singleton of class '" 
				           << typeName<Type>() << "' was already destroyed!");

			Type* p = static_cast<Type*>(Base::getSharedLibrarySingleton(typeid(Type)));
			if(!p) {
				p = Base::CreationPolicy::create();
				Base::LifetimePolicy::scheduleDestruction(p,&destroyInstance);
				Base::setSharedLibrarySingleton(typeid(Type),p);
			}
			Base::sInstance = p;
		}

		assert(Base::sInstance);
		return (Type*)Base::sInstance;
	}

	static void destroyInstance(void){
		if (!Base::sDestroyed) {
			Base::CreationPolicy::destroy((Type*)Base::sInstance);
			Base::sInstance=NULL;
			Base::sDestroyed=true;
		}
	}
};


/**
 * @ingroup SingletonModule
 * Implementation of the InstantiationPolicy that is used by the Singleton template.
 * This policy ensures early/eager instantiation, i.e. the singleton object
 * will be instantiated BEFORE the first line of the applications main() function
 * is executed. This is useful, if the singleton must do some initialization
 * as early as possible.
 * Please note, that the singleton will not be instantiated if the instance()
 * method is never called!
 */
template <typename Trait>
class EagerInstantiation : public LazyInstantiation<Trait>
{
	typedef LazyInstantiation<Trait> Base;

public:
	typedef typename Base::Type Type;

	static Type& instance() {
		if(!sInstantiationHelper)
			sInstantiationHelper = Base::makeInstance();

		assert(sInstantiationHelper);
		return *sInstantiationHelper;
	}
private:
	// include this to provoke instantiation at pre-execution time
	static void use(const Type*) {}
	static Type* forceInstantiation() {
		Type* inst = Base::makeInstance();
		// refer to instance, causing it to be instantiated (and
		// initialized at startup on working compilers)
		use(sInstantiationHelper);
		return inst;
	}
	
	static Type* sInstantiationHelper;
};
template <typename Trait>
typename EagerInstantiation<Trait>::Type* EagerInstantiation<Trait>::sInstantiationHelper = EagerInstantiation<Trait>::forceInstantiation();


/**
 * @ingroup SingletonModule
 * Implementation of the InstantiationPolicy that is used by the Singleton template.
 * This policy allows full control of the construction and destruction time
 * of the singleton. The singleton must be instantiated (and hence destroyed)
 * explicitly by the user. It will throw XLogical, if someone tries to create
 * a second instance of the singleton.
 * Due to the explicit instantiation, this singleton type is the only singleton
 * that allows to pass parameters to the constructor.
 *
 * Please note, that Singletons with this InstantiationPolicy can not take
 * the CreationPolicy and LifetimePolicy into account. Those policies will be
 * ignored.
 */
template <typename Trait>
class ExplicitInstantiation : public Private::SingletonBase<Trait>
{
	typedef Private::SingletonBase<Trait> Base;

public:
	typedef typename Base::Type Type;

	ExplicitInstantiation() {
		if(Base::sInstance!=NULL || Base::getSharedLibrarySingleton(typeid(Type))!=NULL)
			MIRA_THROW(XLogical, "There should be just one object of '"
			           << typeName<Type>() << "', you tried to initialize another one!");
		Base::sInstance = static_cast<Type*>(this);
		Base::setSharedLibrarySingleton(typeid(Type), this);
		Base::sDestroyed=false;
	}

	/// Destroys and removes the shared singleton instance. 
	/// Subsequent calls to instance() will fail.
	~ExplicitInstantiation()
	{
		// remove singleton instance
		Base::sInstance = NULL;
		Base::setSharedLibrarySingleton(typeid(Type), NULL);
		Base::sDestroyed=true;
	}

	static Type& instance()
	{
		if(Base::sInstance==NULL) {
			Type* p = static_cast<Type*>(Base::getSharedLibrarySingleton(typeid(Type)));

			if(p==NULL)
				MIRA_THROW(XLogical, "No instance of the explicit singleton '" 
				           << typeName<Type>() << "' created, "
				           "or instance was already destroyed. "
				           "You must create one object of this class before "
				           "you can start using it!");

			Base::sInstance = p;
		}
		return *((Type*)Base::sInstance);
	}
};


///////////////////////////////////////////////////////////////////////////////
// Lock Policies

/**
 * @ingroup SingletonModule
 * Implementation of the LockPolicy that is used by the Singleton template.
 * This policy performs no locking at all and has no performance penalties.
 */
struct NoLock
{
	typedef int Mutex; // use an int as dummy mutex instance
	struct Lock {
		Lock(int) {}
	};
	template <typename T>
	struct Volatile {
		typedef T type;
	};
};

/**
 * @ingroup SingletonModule
 * Implementation of the LockPolicy that is used by the Singleton template.
 * This policy locks the creation of the singleton, making it thread-safe.
 * Internally, this policy causes the declaration of the internal instance
 * pointer to be "volatile" which may result in minor performance penalties
 * when accessing the singleton instance.
 */
struct MutexLock
{
	typedef boost::mutex Mutex;
	typedef boost::mutex::scoped_lock Lock;

	// adds volatile to T to make the double-checked locking pattern safe
	template <typename T>
	struct Volatile {
		typedef volatile T type;
	};
};

} // namespace singleton

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SingletonModule
 * A singleton template class that can be freely configured using policies that
 * control the instantiation, creation, lifetime and thread-safety.
 *
 * Using this template class you can create singletons that meet all your needs
 * by specifying different combination of policies as template parameters.
 *
 * <b>Instantiation Policies:</b>
 *  - \ref mira::singleton::LazyInstantiation "LazyInstantiation"
 *  - \ref mira::singleton::EagerInstantiation "EagerInstantiation"
 *  - \ref mira::singleton::ExplicitInstantiation "ExplicitInstantiation"
 *
 * <b>Creation Policies:</b>
 *  - \ref mira::singleton::CreateUsingNew "CreateUsingNew"
 *  - \ref mira::singleton::CreateStatic "CreateStatic"
 *  - \ref mira::singleton::CreateUsing "CreateUsing...Allocator"
 *
 * <b>Lifetime Policies:</b>
 *  - \ref mira::singleton::NormalLifetime "NormalLifetime"
 *  - \ref mira::singleton::NoDestroyLifetime "NoDestroyLifetime"
 *  - \ref mira::singleton::LongevityLifetime "LongevityLifetime" (not yet implemented)
 *
 * <b>Lock Policies:</b>
 *  - \ref mira::singleton::MutexLock "MutexLock"
 *  - \ref mira::singleton::NoLock "NoLock"
 *
 *  Example Usage:
 *  \code
 *
 *  class MySingleton : public Singleton<MySingleton,
 *                                   singleton::LazyInstantiation,
 *                                   singleton::CreateUsingNew,
 *                                   singleton::NormalLifetime,
 *                                   singleton::NoLock>
 *  {
 *  public:
 *     ...
 *     void foo();
 *  };
 *
 *  ...
 *
 *  MySingleton::instance().foo();
 *  \endcode
 *
 *  For convenience some of commonly used policy combinations are provided by
 *  - \ref mira::LazySingleton "LazySingleton"
 *  - \ref mira::EagerSingleton "EagerSingleton"
 *  - \ref mira::ExplicitSingleton "ExplicitSingleton"
 *  - \ref mira::LazySingletonNoLock "LazySingletonNoLock"
 *  - \ref mira::LazySingletonNoLock "LazySingletonNoLock"
 */
template <typename T,
          template <class> class TInstantiationPolicy = singleton::LazyInstantiation,
          template <class> class TCreationPolicy = singleton::CreateUsingNew,
          template <class> class TLifetimePolicy = singleton::NormalLifetime,
          class TLockPolicy = singleton::MutexLock>
class Singleton : public TInstantiationPolicy<Private::SingletonTrait<T,TInstantiationPolicy,TCreationPolicy,TLifetimePolicy,TLockPolicy>>
{
	typedef TInstantiationPolicy<Private::SingletonTrait<T,TInstantiationPolicy,TCreationPolicy,TLifetimePolicy,TLockPolicy>> Base;
public:

	// type of singleton class T without const and volatile qualifiers
	typedef typename Base::Type Type;

	/**
	 * Returns a reference to the singleton instance.
	 * @throws XLogical If the singleton was already destroyed (dead reference) 
	 * or not yet created (when using ExplicitInstantiation).
	 */
	static Type& instance()	{
		return Base::instance();
	}

	/**
	 * Returns true, if the singleton was already destroyed. Trying to access
	 * it will result in an exception.
	 */
	static bool isDestroyed() {
		return Base::sDestroyed;
	}

};

/**
 * @ingroup SingletonModule
 * Provided for convenience.
 * This is equivalent to Singleton<T, singleton::LazyInstantiation, singleton::CreateUsingNew>.
 */
template <typename T>
class LazySingleton : public Singleton<T, singleton::LazyInstantiation, singleton::CreateUsingNew> {};

/**
 * @ingroup SingletonModule
 * Provided for convenience.
 * This is equivalent to Singleton<T, singleton::EagerInstantiation, singleton::CreateUsingNew>.
 */
template <typename T>
class EagerSingleton : public Singleton<T, singleton::EagerInstantiation, singleton::CreateUsingNew> {};

/**
 * @ingroup SingletonModule
 * Provided for convenience.
 * This is equivalent to Singleton<T, singleton::ExplicitInstantiation>.
 */
template <typename T>
class ExplicitSingleton : public Singleton<T, singleton::ExplicitInstantiation> {};

/**
 * @ingroup SingletonModule
 * Provided for convenience. Same as LazySingleton but without thread safe instantiation.
 * This is equivalent to Singleton<T, singleton::LazyInstantiation, ..., singleton::NoLock>.
 */
template <typename T>
class LazySingletonNoLock : public Singleton<T, singleton::LazyInstantiation,
                           singleton::CreateUsingNew, singleton::NormalLifetime,
                           singleton::NoLock> {};

/**
 * @ingroup SingletonModule
 * Provided for convenience. Same as EagerSingleton but without thread safe instantiation.
 * This is equivalent to Singleton<T, singleton::EagerInstantiation, ..., singleton::NoLock>.
 */
template <typename T>
class EagerSingletonNoLock : public Singleton<T, singleton::EagerInstantiation,
                           singleton::CreateUsingNew, singleton::NormalLifetime,
                           singleton::NoLock> {};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
