/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Serialization.h
 *    Provides MIRA_CLASS_SERIALIZATION macro and includes the major
 *    headers of the serialization framework.
 *
 * @author Erik Einhorn
 * @date   2010/07/07
 */

#ifndef _MIRA_SERIALIZATION_H_
#define _MIRA_SERIALIZATION_H_

#include <serialization/XMLSerializer.h>
#include <serialization/BinarySerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/PropertySerializer.h>
#include <serialization/MetaSerializer.h>
#include <serialization/Print.h>

#include <factory/Factory.h>

#include <utils/Registrar.h>
#include <utils/PParam.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
/**
 * Registerer that is used with the registrar macro below.
 */
template<typename Serializer, typename T>
class SerializerClassTypeRegisterer
{
public:
	SerializerClassTypeRegisterer()
	{
		// register the class T in the Serializer
		Serializer::template registerClass<T>();
	}

	~SerializerClassTypeRegisterer()
	{
		// register the class T in the Serializer
		Serializer::template unregisterClass<T>();
	}
};

#define MIRA_SERIALIZER_SEQUENCE     \
	(mira::XMLSerializer)            \
	(mira::XMLDeserializer)          \
	(mira::JSONSerializer)           \
	(mira::JSONDeserializer)         \
	(mira::BinaryBufferSerializer)   \
	(mira::BinaryBufferDeserializer) \
	(mira::BinaryStreamSerializer)   \
	(mira::BinaryStreamDeserializer) \
	(mira::PropertySerializer)       \
	(mira::MetaSerializer)


#define MIRA_CLASS_SERIALIZATION_IMPL(r, Class, Serializer) \
	MIRA_REGISTRAR(MIRA_PPARAM(mira::SerializerClassTypeRegisterer<Serializer, Class>)) \

///@endcond

/**
 * @ingroup SerializationModule
 *
 * Registers the specified class in the class factory and additionally
 * registers polymorphic serialization/deserialization with all known
 * serializers.
 * The first parameter must be the type of the class to register, while
 * the second parameter must be the type of the base class.
 *
 * If you derive a class from mira::Object, you must use this macro to register
 * your class at the class factory and to register it at all known
 * Serializers and Deserializers.
 *
 * If you implement a new RecursiveMemberReflector you should add it below to
 * the macro.
 */
#define MIRA_CLASS_SERIALIZATION( pCLASS, ... ) \
	MIRA_CLASS_REGISTER( pCLASS, __VA_ARGS__ )  \
	BOOST_PP_SEQ_FOR_EACH( MIRA_CLASS_SERIALIZATION_IMPL, pCLASS, MIRA_SERIALIZER_SEQUENCE )

///@cond INTERNAL
#define MIRA_INTERNAL_TEMPLATE_CLASS_SERIALIZATION_IMPL(r, Args, i, Serializer) \
	/* Args(0) - current template specialization */                            \
	/* Args(1) - number of template parameters */                              \
	/* Args(2) - The class */                                                  \
	/* Args(3-n) - The namespace */                                            \
	MIRA_REGISTRAR(MIRA_PPARAM(mira::SerializerClassTypeRegisterer<Serializer,    \
		MIRA_INTERNAL_CONC_NAMESPACE(Args,3)BOOST_PP_SEQ_ELEM(2,Args)<BOOST_PP_TUPLE_REM_CTOR( BOOST_PP_SEQ_ELEM(1,Args), BOOST_PP_SEQ_ELEM(0,Args))> >))\


#define MIRA_INTERNAL_TEMPLATE_CLASS_SERIALIZATION(r, Args, TmplSpec)        \
	/* Args(0) - number of template parameters */                           \
	/* Args(1) - The class */                                               \
	/* Args(2-n) - The namespace */                                         \
	BOOST_PP_SEQ_FOR_EACH_I(MIRA_INTERNAL_TEMPLATE_CLASS_SERIALIZATION_IMPL, \
	                        (TmplSpec)Args, MIRA_SERIALIZER_SEQUENCE)

///@endcond

/**
 * @ingroup SerializationModule
 *
 * Registers the specified template class in the class factory and additionally
 * registers polymorphic serialization/deserialization with all known
 * serializers.
 * See MIRA_CLASS_SERIALIZATION for more details.
 */
#define MIRA_TEMPLATE_CLASS_SERIALIZATION(Namespace, Class, TmplSpec, ... ) \
	MIRA_TEMPLATE_CLASS_REGISTER(Namespace, Class, TmplSpec, __VA_ARGS__ )  \
	BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_TEMPLATE_CLASS_SERIALIZATION, (1)(Class)Namespace,MIRA_MAKE_SEQ(1,TmplSpec))

/**
 * @ingroup SerializationModule
 *
 * Registers the specified template class in the class factory and additionally
 * registers polymorphic serialization/deserialization with all known
 * serializers.
 * See MIRA_CLASS_SERIALIZATION for more details.
 */
#define MIRA_VARTEMPLATE_CLASS_SERIALIZATION(Namespace, Class, NumTmplPrm, TmplSpec, ... ) \
	MIRA_VARTEMPLATE_CLASS_REGISTER(Namespace, Class, NumTmplPrm,               \
	                               TmplSpec, __VA_ARGS__ )                     \
	BOOST_PP_SEQ_FOR_EACH(MIRA_INTERNAL_TEMPLATE_CLASS_SERIALIZATION,           \
                          (NumTmplPrm)(Class)Namespace,                        \
                          MIRA_MAKE_SEQ(NumTmplPrm,TmplSpec))

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
