/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyNode.h
 *    Declaration and implementation of the property node hierarchy.
 *
 * @author Erik Einhorn
 * @date   2011/04/13
 */

#ifndef _MIRA_PROPERTYNODE_H_
#define _MIRA_PROPERTYNODE_H_

#include <serialization/PropertyHint.h>
#include <serialization/adapters/std/pair>
#include <serialization/SplitReflect.h>
#include <serialization/JSONSerializer.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declarations
class PropertySerializer;

template <typename T>
class TypedPropertyNode;

class PropertyNodeInfo
{
public:
	PropertyNodeInfo() {}
	PropertyNodeInfo(const std::string& id, const std::string& name,
	                 const std::string& comment, const Typename& type,
	                 bool isReadOnly) :
	                	 mID(id), mName(name), mComment(comment),
	                	 mType(type), mIsReadOnly(isReadOnly) {}

	// special copy constructor, that clones the property hint
	PropertyNodeInfo(const PropertyNodeInfo& info) :
		mID(info.mID), mName(info.mName), mComment(info.mComment),
		mHint(info.mHint.clone()), mType(info.mType),
		mIsReadOnly(info.mIsReadOnly) {}

public:
	PropertyNodeInfo& operator=(const PropertyNodeInfo& info)
	{
		mID         = info.mID;
		mName       = info.mName;
		mComment    = info.mComment;
		mHint       = info.mHint.clone();
		mType       = info.mType;
		mIsReadOnly = info.mIsReadOnly;

		return *this;
	}

public:

	template<typename Reflector>
	void reflectCommon(Reflector& r)
	{
		r.member("ID", mID, "");
		r.member("Name", mName, "");
		r.member("Comment", mComment, "");
		r.member("Type", mType, "The typename");
		r.member("IsReadOnly", mIsReadOnly, "");
	}

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		reflectCommon(r);
		PropertyHint::AttributeValueList hints = mHint.toList();
		r.member("Hint", hints, "The property hints");
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		reflectCommon(r);
		PropertyHint::AttributeValueList hints;
		r.member("Hint", hints, "The property hints");
		mHint.fromList(hints);
	}

	MIRA_SPLIT_REFLECT_MEMBER

public:

	/// Returns the unique id of this property.
	const std::string& id() const { return mID; }

	/// Returns the name of this property as specified in the reflect() method.
	const std::string& name() const { return mName; }

	/// Returns the comment that is associated with this property.
	const std::string& comment() const { return mComment; }

	/// Returns the type of this property as Typename.
	const Typename& type() const { return mType; }

	/**
	 * Returns the specified value for the given property hint attribute.
	 * If no such attribute is set in this hint, then the defaultValue is
	 * returned.
	 */
	template <typename T>
	T getHint(const std::string& attribute, const T& defaultValue = T()) const {
		return mHint.get(attribute, defaultValue);
	}

	/// Returns true if a hint with the specified attribute exists
	bool hasHint(const std::string& attribute) const {
		return mHint.has(attribute);
	}

	/// Returns true, if this property is read-only and hence, can not be modified.
	bool isReadOnly() const {return mIsReadOnly;}

	/// For internal use by PropertySerializer only: Overrides the name of the property
	void setName(const std::string& name) {
		mName = name;
	}

protected:
	std::string  mID;      ///< the unique id of the property
	std::string  mName;    ///< the name of the property (mostly equal to mID)
	std::string  mComment; ///< the comment that is associated to the property
	PropertyHint mHint;    ///< the specified property hints
	Typename     mType;    ///< the type of the property
	bool         mIsReadOnly; ///< indicates whether the property is read-only
};

/**
 * @ingroup SerializationModule
 * Abstract base class for all derived property node classes.
 * A property node represents a property, which is a special member of an
 * object that is reflected via the "property" method to indicate that the
 * member can be changed dynamically at runtime.
 * A property node usually is part of a hierarchy of property nodes. If a
 * property has a member that itself is property, its corresponding
 * property node is added as children to the parent node.
 *
 * The property nodes are created by the PropertySerializer that is used to
 * visit all properties that are specified in the reflect() methods of the
 * objects. The PropertySerializer will create TypedPropertyNode instances
 * that implement the abstract PropertyNode class for each certain type.
 */
class MIRA_BASE_EXPORT PropertyNode : public PropertyNodeInfo
{
	friend class PropertySerializer; // needs to call addChild, etc

public:
	PropertyNode(const PropertyNodeInfo& info) :
	    PropertyNodeInfo(info), mParent(NULL) {}

	PropertyNode(const std::string& id, const std::string& name,
	             const std::string& comment, const Typename& type,
	             bool isReadOnly) :
		PropertyNodeInfo(id, name, comment, type, isReadOnly), mParent(NULL) {}

	virtual ~PropertyNode();

public:

	/// Returns the parent property node (or NULL, if this is a root node)
	virtual PropertyNode* parent() { return mParent; }

	/// Returns the parent property node (or NULL, if this is a root node)
	virtual const PropertyNode* parent() const { return mParent; }

	/// Returns a vector with all child property nodes.
	virtual const std::vector<PropertyNode*>& children() const { return mChildren; }

	/**
	 * Searches for a child property node (which may also be a child of a
	 * child of a child, etc).
	 * The "path" to the child node is specified via the provided vector,
	 * where the first item corresponds to the id of the child property, the
	 * second item corresponds to the child property of the child property,
	 * etc. The parameter 'level' specifies the item where to start within
	 * the specified vector, this parameter usually will be 0 in the first
	 * call.
	 */
	const PropertyNode* findChildNode(const std::vector<std::string>& ids,
	                                  std::size_t level = 0) const;

	/**
	 * Searches for a child property node (which may also be a child of a
	 * child of a child, etc).
	 * The "path" to the property is addressed via its full name. The following
	 * example
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * addresses the child property "value" of the property "myMember"
	 * of the child property "myObject".
	 */
	const PropertyNode* findChildNode(const std::string& id) const;

	/**
	 * @copydoc findChildNode(const std::string& id) const
	 */
	PropertyNode* findChildNode(const std::string& id);

public:

	/**
	 * Returns the full qualified ID of this property, including the names
	 * of the parent properties separated by '.'
	 *
	 * If no optional parameter is specified, the name is generated up to
	 * the root of the property hierarchy. If another parent node is
	 * specified as the name is generated relative to that node.
	 */
	std::string fullID(PropertyNode* p = NULL) const
	{
		std::string s = id();
		for(const PropertyNode* n = parent(); n!=NULL && n!=p; n=n->parent())
			s = n->id() + "." + s;
		return s;
	}

public:

	/**
	 * Sets the value of the property, where the value is described as
	 * JSON value.
	 */
	virtual void setFromJSON(const json::Value& node) = 0;

	/**
	 * Returns the value of the property as JSON value.
	 */
	virtual json::Value getAsJSON() const = 0;

public:

	/**
	 * Sets the value of the property, where the value is given as string.
	 */
	virtual void setFromString(const std::string& value);

	/**
	 * Returns the value of the property as string.
	 */
	virtual std::string getAsString() const;

public:

	/**
	 * Casts this property node to a typed property node, or returns NULL
	 * if the types do not match
	 */
	template <typename T>
	TypedPropertyNode<T>* toTyped();

	/**
	 * Casts this property node to a typed property node, or returns NULL
	 * if the types do not match
	 */
	template <typename T>
	const TypedPropertyNode<T>* toTyped() const;

protected:

	/// Adds the specified property node as child node
	void addChild(PropertyNode* child);

private:

	/// Pointer to the parent property node
	PropertyNode* mParent;

	/// Pointers to all child property nodes
	std::vector<PropertyNode*> mChildren;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Abstract base class for all typed property nodes.
 * A typed property node allows to get and set the value of the property
 * directly using the correct type of the property.
 * An untyped PropertyNode can be cast easily into a TypedPropertyNode
 * using the PropertyNode::toTyped() method.
 *
 * \copydetails mira::PropertyNode
 *
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNode : public PropertyNode
{
public:

	/// The type of the property
	typedef T value_type; // STL conform typedef

public:

	TypedPropertyNode(const PropertyNodeInfo& info) :
		PropertyNode(info) {
		assert(info.type()==typeName<value_type>());
	}

	TypedPropertyNode(const std::string& id, const std::string& name,
	                  const std::string& comment, bool isReadOnly) :
		PropertyNode(id, name, comment, typeName<value_type>(), isReadOnly) {}

public:

	/**
	 * Sets the property to the specified value.
	 */
	virtual void set(const value_type& value) = 0;

	/**
	 * Returns the value of the property.
	 */
	virtual value_type get() const = 0;

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Partial Implementation of the get/set of TypedPropertyNode specialized
 * for normal classes and classes that are not copyable.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T, bool /* = false */>
class TypedPropertyNodeImplGetSetMixin : public TypedPropertyNode<T>
{
	typedef TypedPropertyNode<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImplGetSetMixin(const std::string& id, const std::string& name,
	                                 const std::string& comment, value_type& value,
	                                 bool isReadOnly) :
		Base(id, name, comment, isReadOnly), mValue(&value) {}

protected:

	void setValue(const value_type& value) {
		/**
		 * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		 * IF YOU GET A COMPILER ERROR HERE, your class with type ValueType
		 * does not support the assignment operator. In this case, you
		 * must inherit your class from boost::non_copyable to indicate
		 * that. This will resolve this issue.
		 */
		*mValue = value;
	}

public:

	virtual value_type get() const {
		return *mValue;
	}

protected:
	value_type* mValue;
};

template <typename T>
class TypedPropertyNodeImplGetSetMixin<T,true> : public TypedPropertyNode<T>
{
	typedef TypedPropertyNode<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImplGetSetMixin(const std::string& id, const std::string& name,
	                                 const std::string& comment, value_type& value,
	                                 bool isReadOnly) :
		Base(id, name, comment, isReadOnly), mValue(&value) {}

protected:

	void setValue(const value_type& value) {
		MIRA_THROW(XLogical, "Cannot set value of property, since the underlying class does not provide an assignment operator");
	}

public:

	virtual value_type get() const {
		MIRA_THROW(XLogical, "Cannot get value of property, since the underlying class does not provide an assignment operator");
	}

protected:
	value_type* mValue;
};



/**
 * Implementation of TypedPropertyNode.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNodeImpl : public TypedPropertyNodeImplGetSetMixin<T,boost::is_base_of<boost::noncopyable, T>::value>
{
	typedef TypedPropertyNodeImplGetSetMixin<T,boost::is_base_of<boost::noncopyable, T>::value> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, value_type& value,
	                      bool isReadOnly) :
		Base(id, name, comment, value, isReadOnly) {}
public:

	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(*this->mValue);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}

	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(*this->mValue);
	}

public:

	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		Base::setValue(value);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}

	// implemented in GetSetMixin
	//virtual value_type get() const;

protected:
	friend class PropertySerializer;

	/**
	 * Is called by PropertySerializer to update the internal representation
	 * of the value of the property.
	 */
	void update(T& value) {
		this->mValue = &value;
	}

private:

	/**
	 * Synchronizes this node and its children with the
	 * the actual underlying data representation.
	 */
	void synchronize();


};

///////////////////////////////////////////////////////////////////////////////

/**
 * Implementation of TypedPropertyNode for pointers.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNodeImpl<T*> : public TypedPropertyNode<T*>
{
	typedef TypedPropertyNode<T*> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, const value_type& value,
	                      bool isReadOnly) :
		Base(id, name, comment, isReadOnly), mPointer(value) {}
public:
	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(mPointer);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}
	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(mPointer);
	}
public:
	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		mPointer = value;
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}
	virtual value_type get() const {
		return mPointer;
	}
protected:
	friend class PropertySerializer;
	void update(T* value) { // is called from PropertySerializer
		mPointer = value;
	}
private:
	void synchronize();
private:
	value_type mPointer;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Implementation of TypedPropertyNode for Accessors, i.e. getters and setters.
 * @note This class is for internal use by the property serialization framework
 */
template <typename Getter, typename Setter>
class TypedPropertyNodeImpl<Accessor<Getter,Setter>> :
		public TypedPropertyNode<typename Accessor<Getter,Setter>::value_type>
{
	typedef typename Accessor<Getter,Setter>::value_type T;
	typedef Accessor<Getter,Setter> AccessorType;
	typedef TypedPropertyNode<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, const AccessorType& value,
	                      bool isReadOnly) :
		Base(id, name, comment, isReadOnly), mAccessor(value) {}
public:

	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(mAccessor);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}

	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(mAccessor);
	}

public:

	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		mAccessor.set(value);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}

	virtual value_type get() const {
		return mAccessor.get();
	}

protected:
	friend class PropertySerializer;

	void update(AccessorType& value) { // is called from PropertySerializer
		mAccessor = value;
	}

private:
	void synchronize();
private:
	AccessorType mAccessor;
};

///////////////////////////////////////////////////////////////////////////////

// Support for remote property management

/**
 * Special derived class of PropertyNode, that allows to handle "remote
 * properties" transparently. Remote properties, usually are not located
 * in the local process and hence not in the local process memory. Instead,
 * the calls to "get" and "set" need to be delegated to the other process,
 * where the properties are located. Therefore, this class acts as a proxy
 * that mirrors those remote properties in the local process.
 *
 * Derived classes must implement the setFromJSON() and getAsJSON() methods
 * and delegate the calls to the remote side.
 *
 * This class provides a special toTyped() method. It creates a special
 * TypedRemotePropertyNode of the correct type. For details see
 * TypedRemotePropertyNode. The AbstractRemotePropertyNode holds this
 * typed property node and will destroy it automatically.
 *
 * @note This class is for internal use by the property serialization framework
 * @ingroup SerializationModule
 */
class AbstractRemotePropertyNode : public PropertyNode
{
public:
	AbstractRemotePropertyNode(const PropertyNodeInfo& info) :
		PropertyNode(info), mTypedNode(NULL) {}

	virtual ~AbstractRemotePropertyNode() {
		delete mTypedNode;
	}

public:

	/// Adds the specified property node as child node
	void addChild(PropertyNode* child) {
		PropertyNode::addChild(child);
	}

public:

	/// Specialized method that creates a TypedRemotePropertyNode
	template <typename T>
	TypedPropertyNode<T>* toTyped();

private:

	/// Pointer to the TypedRemotePropertyNode, or NULL, if we are yet untyped.
	PropertyNode* mTypedNode;
};

/**
 * Special TypedPropertyNode for remote properties.
 * Instances of this class are created by the toTyped() method of
 * the AbstractRemotePropertyNode class. The instances are also owned by
 * their corresponding AbstractRemotePropertyNode objects.
 *
 * This class stores the PropertyNodeInfo only AND is of the correct type
 * of the underlying property. Instances of this class have no further
 * information concerning the property hierarchy. Calls to the parent() and
 * children() methods are delegated to the associated
 * AbstractRemotePropertyNode class.
 *
 * Besides the sole PropertyNodeInfo this class only provides the get() and
 * set(). These methods depend on the data type of the property. They serialize
 * the data into JSON format and delegate the data via setFromJSON() and
 * getAsJSON() to the associated AbstractRemotePropertyNode class
 * (which must handle the remote transport).
 *
 * @note This class is for internal use by the property serialization framework
 * @ingroup SerializationModule
 */
template <typename T>
class TypedRemotePropertyNode : public TypedPropertyNode<T>
{
public:
	typedef TypedPropertyNode<T> Base;
	typedef typename Base::value_type value_type;

public:

	TypedRemotePropertyNode(AbstractRemotePropertyNode* node) :
		Base(*node), mNode(node) {}

public:
	// implementation of PropertyNode that delegates to the "real"
	// RemotePropertyNode
	virtual void setFromJSON(const json::Value& node) { mNode->setFromJSON(node); }
	virtual json::Value getAsJSON() const { return mNode->getAsJSON(); }

public:
	// the following methods of PropertyNode must be delegated to the "real"
	// RemotePropertyNode, that has the parent and the children.
	virtual PropertyNode* parent() { return mNode->parent();}
	virtual const PropertyNode* parent() const { return mNode->parent(); }
	virtual const std::vector<PropertyNode*>& children() const { return mNode->children(); }

public:
	// implementation of TypedPropertyNode

	virtual void set(const value_type& value)
	{
		try {
			// serialize the value to json and set it remotely via setFromJSON()
			JSONSerializer s;
			json::Value jsonval = s.serialize(value);
			setFromJSON(jsonval);
		} catch(std::exception& ex) {
			MIRA_THROW(XIO, "Failed to convert value to json: " << ex.what())
		}
	}

	virtual value_type get() const
	{
		try {
			// get the value remotely via getAsJSON
			json::Value jsonval = getAsJSON();
			// and deserialize it into the value
			value_type value;
			JSONDeserializer ds(jsonval);
			ds.deserialize(value);
			return value;
		} catch(std::exception& ex) {
			MIRA_THROW(XIO, "Failed to obtain remote value: " << ex.what())
		}
	}

private:
	AbstractRemotePropertyNode* mNode;
};

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

// implementations of the PropertyNode toTyped methods
template <typename T>
inline TypedPropertyNode<T>* PropertyNode::toTyped() {

	// try a direct cast to the desired type
	TypedPropertyNode<T>* node = dynamic_cast<TypedPropertyNode<T>*>(this);
	if(node!=NULL)
		return node; // success

	// if the above cast failed, the property could still be a remote property
	AbstractRemotePropertyNode* remoteNode = dynamic_cast<AbstractRemotePropertyNode*>(this);
	if(remoteNode!=NULL)
		return remoteNode->toTyped<T>();

	// otherwise we cannot cast
	MIRA_THROW(XBadCast, "Cannot cast PropertyNode to type '" << typeName<T>() << "'");
}

template <typename T>
inline const TypedPropertyNode<T>* PropertyNode::toTyped() const {
	PropertyNode* This = const_cast<PropertyNode*>(this);
	return This->toTyped<T>();
}

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline TypedPropertyNode<T>* AbstractRemotePropertyNode::toTyped()
{
	if(mTypedNode) {
		TypedPropertyNode<T>* node = dynamic_cast<TypedPropertyNode<T>*>(mTypedNode);
		if(node==NULL)
			MIRA_THROW(XBadCast, "Cannot cast PropertyNode which is of "
			                     "type '" << mType << "' to requested type '"
			                     << typeName<T>() << "'");
		return node;
	} else {
		if(mType != typeName<T>())
			MIRA_THROW(XBadCast, "Cannot cast remote PropertyNode which is of "
			                     "type '" << mType << "' to requested type '"
			                     << typeName<T>() << "'");

		TypedRemotePropertyNode<T>* node = new TypedRemotePropertyNode<T>(this);
		mTypedNode = node;

		return node;
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
