/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Accessor.h
 *    Contains internal accessor class that abstracts from the underlying
 *    getter and setter classes or direct access to provide a transparent
 *    and uniform interface (The classes in this header are part of the
 *    internals)
 *
 * @author Erik Einhorn
 * @date   2011/04/12
 */

#ifndef _MIRA_ACCESSOR_H_
#define _MIRA_ACCESSOR_H_

#include <serialization/GetterSetter.h>
#include <serialization/SplitReflect.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Implements the "getter part" of an Accessor. This part provides the get()
 * method and a reflectRead() method.
 * This generic AccessorGetterPart allows direct access on the member.
 * There is also a specialization for getters.
 */
template<typename T>
class AccessorGetterPart
{
public:
	typedef T value_type;

	static const bool isDirectGetter = true;

public:
	AccessorGetterPart(const T& value) : getter((T*)&value) {}

public:
	const value_type& get() const {
		return *getter;
	}

public:
	template<typename Reflector>
	void reflectRead(Reflector& r) {
		r.delegate(*getter);
	}

private:
	T*        getter; // for reading
};

//specialization for getters
template <typename T>
struct AccessorGetterPart<Getter<T>>
{
public:
	typedef T value_type;

	static const bool isDirectGetter = false;

public:
	AccessorGetterPart(const Getter<T>& iGetter) : getter(iGetter) {}

public:
	value_type get() const {
		return getter.get();
	}

public:
	template<typename Reflector>
	void reflectRead(Reflector& r) {
		r.delegate(getter);
	}

private:
	Getter<T> getter; // for reading
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Implements the "setter part" of an Accessor. This part provides the set()
 * method and a reflectWrite() method.
 * This part is implemented for setters only.
 */
template<typename T>
class AccessorSetterPart {
	static_assert(sizeof(T)==0, "Not implemented for direct access. "
			                    "There was no need yet");
};

// specialization for setters
template<typename T>
class AccessorSetterPart<Setter<T>>
{
public:
	typedef T value_type;

public:
	AccessorSetterPart(const Setter<T>& iSetter) : setter(iSetter) {}

public:
	void set(const value_type& value) {
		setter.set(value);
	}

public:
	template<typename Reflector>
	void reflectWrite(Reflector& r)	{
		r.delegate(setter);
	}

private:
	Setter<T> setter; // for writing
};

/// "Null-Setter" tag-class where the AccessorSetterPart does nothing.
template <typename T>
class NullSetter {};

// specialization for "Null"-Setter, that does nothing
template<typename T>
class AccessorSetterPart<NullSetter<T>>
{
public:
	typedef T value_type;

public:
	AccessorSetterPart() {}
	AccessorSetterPart(const NullSetter<T>&) {}

public:
	void set(const value_type& value) {}

public:
	template<typename Reflector>
	void reflectWrite(Reflector& r)	{
	}
};


///////////////////////////////////////////////////////////////////////////////

/**
 * The Accessor class is used as an adapter to reduce the code bloat within
 * the reflection and serialization implementations. An accessor allows reading
 * and writing access to data/member variable. It abstracts from the actual
 * underlying access to that data, which may be a direct access or an access
 * using getters and setters. In total there are 4 cases how to access data:
 * -# read directly, write directly
 * -# read directly, write via a setter
 * -# read via a getter, write directly
 * -# read via a getter, write via a setter
 *
 * All these cases can be reduced to a single case, if this Accessor class
 * is used as an abstraction layer.
 *
 * It provides an assignment operator to mimic direct writing access and it
 * provides a cast operator to mimic direct reading access (even if getters
 * and setters are used).
 *
 * This class puts together the AccessorGetterPart and AccessorSetterPart.
 */
template <typename Getter, typename Setter>
class Accessor : public AccessorGetterPart<Getter>,
                 public AccessorSetterPart<Setter>
{
	typedef AccessorGetterPart<Getter> GetterPart;
	typedef AccessorSetterPart<Setter> SetterPart;

public:
	Accessor(const Getter& getter, const Setter& setter) :
		AccessorGetterPart<Getter>(getter), AccessorSetterPart<Setter>(setter) {}

	Accessor(const Getter& getter) :
		AccessorGetterPart<Getter>(getter) {} // for null setter

public:

	static_assert(std::is_same<typename GetterPart::value_type,
			                   typename SetterPart::value_type>::value,
			"Type of value in Getter and Setter must be the same");

	/// the underlying type (as STL conform typedef)
	typedef typename GetterPart::value_type value_type;

	/// the type that is returned by the get() method of the GetterPart.
	typedef decltype(((GetterPart*)(NULL))->get()) GetReturnValueType;

	/// mimic the assignment behavior of underlying type, e.g. accessor = 1234;
	Accessor& operator=(const value_type& value) {
		this->set(value);
		return *this;
	}

	/**
	 * cast operator to mimic cast behavior of underlying value type;
	 * applies the getter, e.g. int val = mygetter;
	 */
	operator GetReturnValueType() const { return this->get(); }

public:

	MIRA_SPLIT_REFLECT_MEMBER
	// reflectRead and reflectWrite are implemented in AccessorGetterPart and
	// AccessorSetterPart.
};

// mark the Accessor as transparent serializable (internal detail)
template <typename Getter, typename Setter, typename SerializerTag>
class IsTransparentSerializable<Accessor<Getter,Setter>,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

/**
 * Helper method that creates an accessor from a different combination of
 * either direct access to a variable (i.e. the reference of the variable
 * itself) and getters and setters that may be used to access the variable.
 */
template <typename Getter, typename Setter>
Accessor<Getter,Setter> makeAccessor(const Getter& getter, const Setter& setter) {
	return Accessor<Getter,Setter>(getter,setter);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
