/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Process.h
 *    Platform independent system calls.
 *
 * @author Erik Einhorn
 * @date   2011/02/10
 */

#ifndef _MIRA_EXECUTEPROCESS_H_
#define _MIRA_EXECUTEPROCESS_H_

#include <string>
#include <vector>

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#include <boost/optional.hpp>
#include <boost/iostreams/device/file_descriptor.hpp>
#include <boost/iostreams/stream.hpp>
#endif

#include <platform/Platform.h>

#include <utils/EnumToFlags.h>
#include <utils/Time.h>

#ifdef MIRA_LINUX
#  include <unistd.h>
#endif


namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Executes a given command/process.
 * @param[in] cmd The command to execute
 * @param[out] oStdOut Optional pointer to string where console output is stored
 * @param[out] oErrOut Optional pointer to string where error output is stored
 * @return -1 on error.
 */
MIRA_BASE_EXPORT int executeProcess(const std::string& cmd, 
                                    std::string* oStdOut=NULL,
                                    std::string* oErrOut=NULL);

///////////////////////////////////////////////////////////////////////////////

/**
 * Encapsulates a process, that was launched from the current process.
 * A new process can be spawned using the createProcess() method, which
 * returns an object to the created process. The process object can be
 * used to terminate the process and to check its current status.
 *
 * Note, that Process objects can not be copied. Use the move semantic
 * and std::move() to pass instances of this class.
 */
class MIRA_BASE_EXPORT Process : boost::noncopyable
{
public:
	enum ExitStatus
	{
		NORMALEXIT,
		CRASHED,
		KILLED,
	};



public:

	/**
	 * Holds the environment variables that can be passed to a process.
	 *
	 * The environment of the calling process can be obtained using
	 * systemEnvironment().
	 */
	class MIRA_BASE_EXPORT Environment
	{
	public:
		Environment() {};
		Environment(const Environment& other);
		Environment(Environment&& other);

		Environment& operator=(const Environment& other);
		Environment& operator=(Environment&& other);

	public:
		/// Removes all environment variables
		void clear();

		bool empty() const;

		/// inserts or replaces the environment variable var with the given value.
		void insert(const std::string& var, const std::string& value);

		/**
		 * Inserts or replaces the environment variable.
		 * The variable and the value are given in the format:
		 * <pre>
		 * variable=value
		 * </pre>
		 */
		void insert(const std::string& string);

		/**
		 * Inserts all variables from the given string which has the format
		 * <pre>
		 * variable1=value1;variable2=value2;variable3=value3
		 * </pre>
		 */
		void insertList(const std::string& string);

		/// inserts all variables from the given environment, while replacing existing variables.
		void insert(const Environment& env);

		/// removes the environment variable
		void remove(const std::string& var);

		/// returns the value of the environment variable var, or the default value, if it does not exist.
		const std::string& value(const std::string& var, const std::string& defaultValue = std::string()) const;

		/**
		 * Generates and returns an vector of strings, conventionally of the
		 * form key=value, which can be used to call low-level system functions
		 * such as execXe().
		 */
		std::vector<std::string> envp();


	public:

		const std::map<std::string,std::string>& variables() const { return mVariables; }

	public:

		// Returns the environment of the calling process.
		static Environment systemEnvironment();

	private:
		std::map<std::string,std::string> mVariables;

	};



public:


	/// Creates an invalid process object.
	Process();

	/// Destructor, which will terminate the process, if it is still running.
	~Process();

	// move constructor and move assignment op to support move semantics.
	Process(Process&& other);
	Process& operator=(Process&& other);

public:

	/// Swaps the ownership of the process context within this and other.
	void swap(Process& other);

public:

	enum CreationFlags {
		noflags = 0x00,
		interruptChildOnParentDeath  = 0x01 ///< Will send the SIGINT signal to the child when the parent gets killed
	};
	MIRA_ENUM_TO_FLAGS_INCLASS(CreationFlags)

	enum RedirectionFlags {
		none  = 0x00,
		in    = 0x01,
		out   = 0x02,
		err   = 0x04
	};
	MIRA_ENUM_TO_FLAGS_INCLASS(RedirectionFlags)


	/**
	 * Spawns a new process and executes the given command line.
	 * The command line may contain parameters that are separated by white
	 * characters.
	 *
	 * On Windows the application is searched in the following paths:
	 * -# The directory from which the application loaded.
	 * -# The current directory for the parent process.
	 * -# The Windows system directory.
	 * -# The directories that are listed in the PATH environment variable.
	 *
	 * On Linux the application is searched within:
	 * -# The directories that are listed in the PATH environment variable.
	 *
	 * @param flags Allows to specify some flags that control the creation and
	 *              termination of the created process. @see CreationFlags.
	 *
	 * @param streamRedirection Allows to specify which streams (stdin, stdout, etc)
	 *                          should be redirected in order to access them
	 *                          from the parent process via the cin(), cout(),
	 *                          cerr() stream method (see below).
	 *
	 * @return Returns the Process object that can be used to wait for the
	 *         process to terminate, etc.
	 *
	 * @throws XRuntime If the process can not be forked
	 * @throws XFileNotFound If the given executable can not be found
	 * @throws XInvalidParameter If the execution of the given application fails
	 */
	static Process createProcess(const std::string& commandLine,
	                             CreationFlags flags,
	                             RedirectionFlags streamRedirection = none);

	/// Same as above but the flags can be omitted
	static Process createProcess(const std::string& commandLine,
	                             RedirectionFlags streamRedirection = none);

	/**
	 * Same as above, but a more advanced interface to create a new process.
	 * Here, the arguments can be specified separately as vector of strings.
	 * Moreover, the environment of the new process (containing the environment
	 * variables) can be specified.
	 */
	static Process createProcess(const std::string& applicationName,
	                             const std::vector<std::string>& args,
	                             CreationFlags flags = noflags,
	                             RedirectionFlags streamRedirection = none,
	                             boost::optional<Environment> env = boost::optional<Environment>());




public:

	/**
	 * Blocks the current thread of the calling process, until the process that
	 * is represented by this Process object has terminated or maxWait is reached.
	 * @return true if process has terminated, false if not or maxWait has been reached
	 */
	bool wait(Duration maxWait = Duration::infinity());

	/**
	 * Return the exit code of the process. Only valid if not running and 
	 * exit status is NORMALEXIT.
	 */
	int getExitCode();

	/**
	 * Returns the exit status of the process only valid if not running.
	 */
	ExitStatus getExitStatus();
	/**
	 * Returns if the process that is represented by this Process object is still
	 * running
	 */
	bool isRunning();

	/**
	 * Stops (e.g. using SIGINT) the process that is represented by this Process object.
	 */
	void interrupt();
	/**
	 * Terminates (e.g. using SIGTERM) the process that is represented by this Process object.
	 */
	void terminate();

	/**
	 * Kills (e.g. using SIGKILL) the process that is represented by this Process object.
	 */
	void kill();

	/**
	 * Performs an orderly shutdown by executing the following steps.
	 *  # looks if process has already ended - returns
	 *  # sends interrupt request and waits until timeout. if process has ended - returns
	 *  # sends terminate request and waits until timeout. if process has ended - returns
	 *  # sends kill request and waits until process has ended
	 * This method blocks until the process that is represented by this 
	 * Process object has terminated.
	 * @param timeout Timeout used for waiting for process to terminate in the above steps
	 */
	void shutdown(Duration timeout = Duration::seconds(5));

	/**
	 * Returns the process id (pid) of the process. Or 0, if no process was
	 * started.
	 */
	uint32 getPID() const;

public:

	typedef boost::iostreams::stream<boost::iostreams::file_descriptor_sink> ostream;
	typedef boost::iostreams::stream<boost::iostreams::file_descriptor_source> istream;

	/**
	 * Output stream for writing to the stdin of the child process.
	 * This stream is available only, if the appropriate redirection flag
	 * was set in createProcess().
	 */
	ostream& cin();

	/**
	 * Input stream for reading from the stdout of the child process.
	 * This stream is available only, if the appropriate redirection flag
	 * was set in createProcess().
	 */
	istream& cout();

	/**
	 * Input stream for reading from the stderr of the child process.
	 * This stream is available only, if the appropriate redirection flag
	 * was set in createProcess().
	 */
	istream& cerr();

private:

#ifdef MIRA_WINDOWS
	//PROCESS_INFORMATION mProcess; //  process information (handle, etc)
#endif

#ifdef MIRA_LINUX
	pid_t mProcess; // handle of the process
#endif

	int mExitCode;
	ExitStatus mExitStatus;

	boost::shared_ptr<ostream> mCinFd;
	boost::shared_ptr<istream> mCoutFd;
	boost::shared_ptr<istream> mCerrFd;

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
