/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RasterRect.h
 *    Rasterizes an 2D Rectangle scanline per scanline.
 *
 * @author Jens Kessler
 * @date   2011/01/17
 */

#ifndef _MIRA_RASTERRECT_H_
#define _MIRA_RASTERRECT_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Rasterizes an Rectangle (2D only).
 * Instead of rastering the line pixel by pixel the rectangle is rastered
 * scanline by scanline from top to bottom. For each segment the visitor's
 * operator()(int x, int y, int length) is called with the parameters
 * x, y, and length that specify the coordinate starting position of the
 * segment and its length.
 *
 * Therefore, the visitor must implement the following concept:
 * \code
 * concept RectVisitor
 * {
 *     void operator()(int x, int y, int length);
 * };
 * \endcode
 *
 * @see @ref GeometryPage
 *
 * @param[in] xl : the x-coordinate of the lower left corner
 * @param[in] yl : the y-coordinate of the lower left corner
 * @param[in] xr : the x-coordinate of the upper right corner
 * @param[in] yr : the y-coordinate of the upper right corner
 * @param[in] visitor : the visitor to execute scanline computations
 *
 */

template<typename Visitor>
inline void rasterRect(int xl, int yl, int xr, int yr, Visitor&& visitor)
{
	if (xl > xr) std::swap(xl,xr);
	if (yl > yr) std::swap(yl,yr);

	int width = xr - xl;
	for(int y = yl; yl < yr; yl++)
	{
		visitor(xl, yl, width);
	}
}

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Rasters a rect (see description  above). Changed interface to be more easy.
 */
template<typename Visitor>
inline void rasterRect(Rect2i rect, Visitor&& visitor)
{
	rasterRect(rect.minCorner[0], rect.minCorner[1],
	           rect.maxCorner[0], rect.maxCorner[1], visitor);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
