/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAPackageGUI.C
 *    Implementation of MIRAPackageGUI.h
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2011/09/20
 */

#include "app/MIRAPackageGUI.h"

#include <QMessageBox>
#include <QCloseEvent>
#include <QFileDialog>

#include <boost/bimap.hpp>

#include <utils/PathFinder.h>

#include "core/Package.h"
#include "core/Tools.h"
#include "gui/PackageTreeModel.h"
#include "gui/TagTableModel.h"
#include "gui/dialogs/DependencyDialog.h"
#include "gui/dialogs/InstallOverviewDialog.h"
#include "gui/dialogs/PathSelectDialog.h"
#include "gui/dialogs/RepositoryListDialog.h"
#include "gui/dialogs/UpdateDialog.h"
#include "gui/dialogs/ExportDialog.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MIRAPackageGUI::MIRAPackageGUI() : QMainWindow(), MIRAPackage()
{
	Ui::MIRAPackageGUI::setupUi(this);

	mListWidget = new PackageListWidget( this );

	setCentralWidget( mListWidget );

	connect( actionReindex, SIGNAL( triggered() ), this, SLOT( reindexDatabase() ) );
	connect( actionThoroughReindex, SIGNAL( triggered() ),
			this, SLOT( deepReindexDatabase() ) );

	connect( actionStore, SIGNAL( triggered() ), this, SLOT( storeDatabase() ) );
	connect( actionExport, SIGNAL( triggered() ), this, SLOT( exportDB() ) );
	connect( actionImport, SIGNAL( triggered() ), this, SLOT( importDB() ) );

	connect( actionModify, SIGNAL( triggered() ), this, SLOT( showRepositoryList() ) );
	connect( mListWidget, SIGNAL( packageInstallRequested( Package* ) ),
			this, SLOT( addPackageForInstall( Package* ) ) );
	connect( mListWidget, SIGNAL( packageUninstallRequested( Package* ) ),
			this, SLOT( addPackageForUninstall( Package* ) ) );
	connect( mListWidget, SIGNAL( applyChangesRequested() ), this, SLOT( doIt() ) );
	connect( mListWidget, SIGNAL( packageUpdateRequested() ), this, SLOT( update() ) );
	connect( actionRight_now, SIGNAL( triggered() ), this, SLOT( doIt() ) );
	connect( actionSelect_MIRA_Path, SIGNAL( triggered() ), this, SLOT( resolveMIRAPathSlot() ) );

	mProgressDialog = NULL;

	resolveMIRAPath();
}

MIRAPackageGUI::~MIRAPackageGUI()
{
}

void MIRAPackageGUI::resolveMIRAPath( bool forceSelection )
{
	// catch environment resolving error
	try {
		PathProvider::instance();
	}
	catch ( Exception& ex ) {
		QMessageBox::critical(this, "Environment resolving error",
			QString::fromStdString( ex.message() ) );
		MIRA_RETHROW( ex, "During startup of MIRAPackageGUI" );
	}
	MIRAPackage::resolveMIRAPath( forceSelection );
}

void MIRAPackageGUI::reindexWebRepository(RepositoryPtr repo, bool thorough)
{
	mListWidget->packageView->clearSelection();
	mListWidget->mPackageModel->beginReset();
	mListWidget->mTagModel->beginReset();
	MIRAPackage::reindexWebRepository( repo, thorough );
	mListWidget->mPackageModel->endReset();
	mListWidget->mTagModel->endReset();
}

void MIRAPackageGUI::checkForInstalled()
{
	mListWidget->mPackageModel->beginReset();
	mListWidget->mTagModel->beginReset();
	mDB.resetPackageFlags();
	MIRAPackage::checkForInstalled();
	mListWidget->mPackageModel->endReset();
	mListWidget->mTagModel->endReset();
}

void MIRAPackageGUI::statusMessage( std::string const& message )
{
	statusBar()->showMessage( QString::fromStdString( message ) );
}

int MIRAPackageGUI::errorMessage( string const& message, string const& actionText,
                                  vector<string>* actions )
{
	if ( actions->size() == 0 ) {
		QMessageBox::critical(this,"Error",QString::fromStdString(message)+"\n"+
			QString::fromStdString(actionText), "OK" );
		return -1;
	}
	if ( actions->size() == 1 ) {
		return QMessageBox::critical(this,"Error",QString::fromStdString(message)+"\n"+
				QString::fromStdString(actionText),
				QString::fromStdString((*actions)[0]));
	}
	if ( actions->size() == 2 ) {
		return QMessageBox::critical(this,"Error",QString::fromStdString(message)+"\n"+
				QString::fromStdString(actionText),
				QString::fromStdString((*actions)[0]),
				QString::fromStdString((*actions)[1]));
	}
	if ( actions->size() == 3 ) {
		return QMessageBox::critical(this,"Error",QString::fromStdString(message)+"\n"+
				QString::fromStdString(actionText),
				QString::fromStdString((*actions)[0]),
				QString::fromStdString((*actions)[1]),
				QString::fromStdString((*actions)[2]));
	}
	return -1;
}

void MIRAPackageGUI::statusProgress( std::string const& message,
                                     std::string const& title,
                                     int value, int maximum )
{
	if (mProgressDialog == NULL) {
		mProgressDialog = new QProgressDialog(this);
		mProgressDialog->setMinimum(0);
		mProgressDialog->setMinimumDuration(0);
		mProgressDialog->setCancelButton(NULL);
		mProgressDialog->setWindowModality(Qt::WindowModal);
	}
	mProgressDialog->setMaximum(maximum);
	mProgressDialog->setValue(value);
	mProgressDialog->setLabelText( QString::fromStdString( message ) );
	mProgressDialog->setWindowTitle( QString::fromStdString( title ) );
	mProgressDialog->show();

	statusBar()->showMessage( QString::fromStdString( message ) );

	if(value == maximum)
		mProgressDialog->hide();

	// TODO: HACK
	// make sure to give Qt some time to show and update the dialog
	// Why isn't it sufficient to call flush() and processEvents() only once??
	for(int i=0; i<100; ++i) {
		QApplication::instance()->flush();
		QApplication::instance()->processEvents();
	}
}

void MIRAPackageGUI::exportDB()
{
	QString tFile = QFileDialog::getSaveFileName(this, "Choose filename","","*.mpe");
	if ( !tFile.isEmpty() ) {
		// append the file extension (if it is not already present)
		if (!tFile.endsWith(".mpe")) {
			tFile.append(".mpe");

			// we have to check the existence of the file a second time
			if ( QFile::exists( tFile ) ) {
				if ( QMessageBox::question( this, "Overwrite file",
						"Do you want to overwrite the file \"" + tFile + "\"?",
						QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
					return;
			}
		}
		MIRAPackage::exportDB( tFile.toStdString() );
	}
}

void MIRAPackageGUI::importDB()
{
	QString tFile = QFileDialog::getOpenFileName(this, "Choose file","","*.mpe");
	if ( !tFile.isEmpty() )
		MIRAPackage::importDB( tFile.toStdString() );
}

void MIRAPackageGUI::reindexDatabase()
{
	reindexWebRepositories( false );
	checkForInstalled();
}

void MIRAPackageGUI::deepReindexDatabase()
{
	reindexWebRepositories( true );
	checkForInstalled();
}

void MIRAPackageGUI::storeDatabase()
{
	save();
}

void MIRAPackageGUI::showRepositoryList()
{
	RepositoryListDialog repositoryList( this, this );
	repositoryList.setModal( true );
	if ( repositoryList.exec() == QDialog::Accepted ) {
		if ( QMessageBox::question(this,"reindex now?","Repository list has been modified.\n"
				"Do you want to reindex the repositories now?",QMessageBox::Yes, QMessageBox::No)
			== QMessageBox::Yes) {
			reindexDatabase();
		}
	}
}

void MIRAPackageGUI::addPackageForInstall( Package* package )
{
	addPackageToCheckoutPlan( package );
	mListWidget->update();
}

void MIRAPackageGUI::addPackageForUninstall( Package* package )
{
	addPackageForRemoval( package );
	mListWidget->update();
}

void MIRAPackageGUI::resolveMIRAPathSlot()
{
	resolveMIRAPath( true );
}

void MIRAPackageGUI::doIt()
{
	try {
		MIRAPackage::doIt();
	}
	catch ( Exception& ex ) {
		QMessageBox::critical( this, "Error",
					QString("Error during checkout: ")
					+ QString::fromUtf8( ex.message().c_str() ),
					QMessageBox::Ok );
			statusMessage( "Error during checkout." );
	}
}

void MIRAPackageGUI::update()
{
	if ( QMessageBox::question(this,"reindex?",
			"Do you want to reindex the Repositories first of all?\n"
			,QMessageBox::Yes, QMessageBox::No)
		== QMessageBox::Yes) {
		reindexDatabase();
	}

	try {
		MIRAPackage::update();
	}
	catch ( Exception& ex ) {
		QMessageBox::critical( this, "Error",
					QString("Error during checkout: ")
					+ QString::fromUtf8( ex.message().c_str() ),
					QMessageBox::Ok );
			statusMessage( "Error during checkout." );
	}
}

void MIRAPackageGUI::closeEvent(QCloseEvent *event)
{
	typedef Database::ActionPlan::value_type mapType;
	foreach( mapType const& action, mDB.mActionPlan ) {
		if ( (action.second & Database::INSTALL) ||
				(action.second & Database::UNINSTALL) ) {
			if ( QMessageBox::warning( this, "changed not applied yet",
					"You have scheduled changed which are not applied yet.\n"
					"Do you really want to leave?",
					QMessageBox::No, QMessageBox::Yes) == QMessageBox::Yes )
			{
				event->accept();
			}
			else {
				event->ignore();
			}
			return;
		}
	}
	QMainWindow::closeEvent( event );
}

string MIRAPackageGUI::selectMIRAPath()
{
	PathSelectDialog* dialog = new PathSelectDialog(
		PathProvider::miraPaths(), this );
	dialog->select( mInstallRoot );
	if (!dialog->exec())
		return mInstallRoot;
	return dialog->mSelectedPath;
}

bool MIRAPackageGUI::confirmCheckoutPlan()
{
	// show the install overview dialog to enable the user to change the
	// installation path
	vector<Package*> uninstallSequence = getPackageSequence( Database::UNINSTALL );
	vector<Package*> installSequence = getPackageSequence( Database::INSTALL );
	InstallOverviewDialog* tDialog = new InstallOverviewDialog(
		uninstallSequence, installSequence, &mDB, this );
	return (tDialog->exec() == QDialog::Accepted);
}

bool MIRAPackageGUI::confirmExportPackages( Database& ioDB, map<Path,string>& oPathMap )
{
	// show the export dialog to enable the user to remove packages from the
	// export action
	ExportDialog* tDialog = new ExportDialog( ioDB, oPathMap, this );
	return (tDialog->exec() == QDialog::Accepted);
}

bool MIRAPackageGUI::confirmUpdatePlan( vector<pair<Package*,Package*> >& updatePlan )
{
	UpdateDialog* tDialog = new UpdateDialog( &updatePlan, this );
	return (tDialog->exec() == QDialog::Accepted);
}

bool MIRAPackageGUI::confirmDependencies( PackageGroup* rootPackage, Database* database )
{
	QDialog* dialog = new DependencyDialog( rootPackage, database, this );
	return (dialog->exec() == QDialog::Accepted );
}

// An internal struct, which is used for reflection of the GUI state.
struct MIRAPackageGUIState
{
	std::string        geometry;
	PackageListWidget* widget;

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Geometry", geometry, "");
		r.member("MainWidget", *widget, "");
	}
};

void MIRAPackageGUI::loadContent(const XMLDom& content)
{
	MIRAPackage::loadContent(content);

	// restore our windows geometry
	try {
		MIRAPackageGUIState guiState;
		guiState.widget = mListWidget;

		XMLDeserializer ds(content);
		ds.deserialize("gui", guiState);

		QByteArray geometry = QByteArray::fromBase64(QByteArray(guiState.geometry.c_str()));
		restoreGeometry(geometry);
	} catch(...) {}
}

void MIRAPackageGUI::saveContent(XMLDom& content)
{
	MIRAPackage::saveContent(content);

	// Ensure, that we don't have an old "gui" node in the XML document
	XMLDom::iterator iter = content.root().find("gui");
	if (iter != content.root().end())
		iter.remove();

	QByteArray geometry = saveGeometry();
	std::string geometryStr = QString(geometry.toBase64()).toStdString();

	MIRAPackageGUIState guiState;
	guiState.geometry = geometryStr;
	guiState.widget = mListWidget;

	XMLSerializer s(content);
	s.serialize("gui", guiState);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

