/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkGraphTextView.C
 *    Shows all informations of the computation framework in textual form.
 *
 * @author Erik Einhorn
 * @date   2012/08/27
 */

#include <serialization/Serialization.h>

// include the view base class
#include <views/FrameworkGraphTextView.h>

// include Qt specific stuff
#include <QTreeWidget>
#include <QHeaderView>
#include <QTimerEvent>

using namespace mira;

#define UPDATE_INTERVAL 2000

namespace mira { 

///////////////////////////////////////////////////////////////////////////////

class FrameworkGraphTextView::UI : public QTreeWidget
{
public:
	UI(QWidget* parent) : QTreeWidget(parent)
	{
	}
};

///////////////////////////////////////////////////////////////////////////////

FrameworkGraphTextView::FrameworkGraphTextView() : ui(NULL)
{
	mUpdateTimer = 0;
}

QWidget* FrameworkGraphTextView::createPartControl()
{
	ui = new UI(this);
	ui->setColumnCount(2);
	ui->setAlternatingRowColors(true);
	ui->header()->setResizeMode(QHeaderView::ResizeToContents);
	ui->header()->hide();


	mUpdateTimer = startTimer(UPDATE_INTERVAL);

	updateGraph();
	return ui;
}

void FrameworkGraphTextView::timerEvent(QTimerEvent *event)
{
	if(event->timerId()==mUpdateTimer)
		updateGraph();
}

QTreeWidgetItem* FrameworkGraphTextView::createChannelItem(Graph::ChannelNodePtr cn, bool subscribe)
{
	QString info = "";
	QColor color;
	if(MIRA_FW.getChannelManager().hasChannel(cn->id)) {
		uint64 updates = MIRA_FW.getChannelManager().getNrOfDataChanges(cn->id);
		uint64 lastupdate = mChannels[cn->id].lastUpdate;
		bool hasPublisher = MIRA_FW.getChannelManager().hasPublisher(cn->id, true);

		if(!mChannels[cn->id].reset) {
			mChannels[cn->id].updates = updates;
			mChannels[cn->id].reset = true;
		}

		float rate = 0.0f;
		if(lastupdate>0)
			rate = (updates-lastupdate) * 1000.0f / UPDATE_INTERVAL;

		if(subscribe && !hasPublisher)
			color = QColor(255,192,192); // subscriber but no publisher -> red
		else if(subscribe && (rate==0 || updates == 0))
			color = QColor(255,255,192); // no (recent) update -> yellow
		else
			color = QColor(192,255,192); // okay -> green
		info = QString("Updates: %1 (%2 Hz)").arg(updates).arg(rate);
	} else {
		color = QColor(192,192,192); // no info -> gray
	}

	QTreeWidgetItem* item = new QTreeWidgetItem((QStringList() << cn->id.c_str() << info));
	item->setBackgroundColor(0,color);
	item->setBackgroundColor(1,color);
	return item;
}
void FrameworkGraphTextView::updateGraph()
{
	ui->clear();

	try {
		mGraph.discover();

		QFont normal = ui->font();
		QFont bold = normal;
		bold.setBold(true);
		QFont italic = normal;
		italic.setItalic(true);

		foreach(Graph::AuthorityNodePtr n, mGraph.getAuthorities())
		{
			if(n->desc.id.find("#")!=std::string::npos)
				continue;

			StatusManager::StatusMap status = MIRA_FW.getAuthorityManager().getStatusMap(n->desc.getGlobalID());

			Status::StatusMode mostSevereMode = Status::OK;
			QString info = "OK";
			foreach(const auto& p, status)
			{
				if(p.second.mode>mostSevereMode) {
					mostSevereMode = p.second.mode;
					if (p.second.trText.empty())
						info = QString::fromUtf8(p.second.message.c_str());
					else
						info = QString::fromUtf8((p.second.trText+": "+p.second.message).c_str());
				}
			}
			QColor color = QColor(192,255,192);
			if(mostSevereMode==Status::ERROR)
				color = QColor(255,192,192);
			else if(mostSevereMode==Status::WARNING)
				color = QColor(255,255,192);

			QTreeWidgetItem* aitem = new QTreeWidgetItem((QStringList() << n->desc.id.c_str() << info));
			aitem->setFont(0,bold);
			aitem->setBackgroundColor(0,color);
			aitem->setBackgroundColor(1,color);
			ui->addTopLevelItem(aitem);


			// subscribed
			QTreeWidgetItem* pitem = new QTreeWidgetItem((QStringList() << "Subscribed" << ""));
			pitem->setFont(0,italic);
			aitem->addChild(pitem);
			foreach(Graph::ChannelNodePtr cn, n->getSubscribedChannels())
			{
				QTreeWidgetItem* item = createChannelItem(cn,true);
				pitem->addChild(item);
			}

			// published
			pitem = new QTreeWidgetItem((QStringList() << "Published" << ""));
			pitem->setFont(0,italic);
			aitem->addChild(pitem);
			foreach(Graph::ChannelNodePtr cn, n->getPublishedChannels())
			{
				QTreeWidgetItem* item = createChannelItem(cn,false);
				pitem->addChild(item);
			}
		}
	}
	catch(Exception& ex) {
		MIRA_LOG(ERROR) << "Exception while updating framework status: " << ex.what();
		// Ignore errors
	}

	foreach(auto& p, mChannels)
	{
		p.second.lastUpdate = p.second.updates;
		p.second.reset = false;
	}


	ui->expandAll();
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::FrameworkGraphTextView, mira::EditorPart );
