/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DefinitionsListView.C
 *    Implementation of DefinitionsListView.h
 *
 * @author Christian Reuther
 * @date   2017/04/25
 */

#include <views/DefinitionsListView.h>

#include <fw/Framework.h>
#include <fw/NameRegistry.h>

#include <QHeaderView>

#include <widgets/TreeViewFilter.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

DefinitionsListView::DefinitionsListView() : mira::EditorPart()
{
	mTabWidget = NULL;
	mVariablesView = NULL;
	mAliasView = NULL;
	mUpdateTimer = 0;
}

QWidget* DefinitionsListView::createPartControl()
{
	mTabWidget = new QTabWidget(this);

	QWidget* aliasWidget = new QWidget(mTabWidget);
	QVBoxLayout* aliasLayout = new QVBoxLayout;
	mAliasView = new QTreeWidget(aliasWidget);
	mAliasView->setColumnCount(3);
	mAliasView->setRootIsDecorated(false);
	mAliasView->setAlternatingRowColors(false);

	QStringList headers;
	headers << trUtf8("Alias reference") << trUtf8("Alias name") << trUtf8("Source");
	mAliasView->setHeaderLabels(headers);

	mAliasView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mAliasView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mAliasView->header()->setResizeMode(2, QHeaderView::Stretch);

	TreeViewFilter* aliasFilter = new TreeViewFilter(mAliasView, aliasWidget);
	aliasFilter->setExpandAll(true);
	aliasFilter->setFilterColumns(0, 1);
	aliasFilter->setFocus();

	aliasLayout->addWidget(aliasFilter);
	aliasLayout->addWidget(mAliasView);

	aliasWidget->setLayout(aliasLayout);

	QWidget* variablesWidget = new  QWidget(mTabWidget);
	QVBoxLayout* varLayout = new QVBoxLayout;
	mVariablesView = new QTreeWidget(variablesWidget);
	mVariablesView->setColumnCount(3);
	mVariablesView->setRootIsDecorated(false);
	mVariablesView->setAlternatingRowColors(true);

	headers.clear();
	headers << trUtf8("Variable") << trUtf8("Value") << trUtf8("Source");
	mVariablesView->setHeaderLabels(headers);

	mVariablesView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(2, QHeaderView::Stretch);

	TreeViewFilter* varFilter = new TreeViewFilter(mVariablesView, variablesWidget);
	varFilter->setExpandAll(true);
	varFilter->setFilterColumns(0, 1);
	varFilter->setFocus();

	varLayout->addWidget(varFilter);
	varLayout->addWidget(mVariablesView);

	variablesWidget->setLayout(varLayout);

	updateAliasList();
	updateVariablesList();

	mTabWidget->addTab(aliasWidget, "Aliases");
	mTabWidget->addTab(variablesWidget, "Variables");

	mUpdateTimer = startTimer(1000);

	return mTabWidget;
}

void DefinitionsListView::timerEvent(QTimerEvent* event)
{
	if(event->timerId() == mUpdateTimer) {
		updateVariablesList();
		updateAliasList();
	}
}

void DefinitionsListView::updateVariablesList()
{
	const XMLVariablesMap& variables = MIRA_FW.getVariables();
	for (int i = mVariablesView->topLevelItemCount(); i < (int)variables.size(); ++i)
		mVariablesView->addTopLevelItem(new QTreeWidgetItem());

	while (mVariablesView->topLevelItemCount() > (int)variables.size())
		delete mVariablesView->topLevelItem(0);

	int row = 0;
	foreach(const auto& v, variables)
	{
		QTreeWidgetItem* item = mVariablesView->topLevelItem(row);
		item->setText(0, QString::fromStdString(v.first));
		item->setText(1, QString::fromStdString(v.second));
		QString src = QString::fromStdString(v.second.annotation);
		item->setText(2, src);
		item->setToolTip(0, src);
		item->setToolTip(1, src);
		item->setToolTip(2, src);

		++row;
	}
}

void DefinitionsListView::updateAliasList()
{
	const mira::NameRegistry::AliasMap& aliases = MIRA_FW.getNameRegistry().mAliases;
	for (int i = mAliasView->topLevelItemCount(); i < (int)aliases.size(); ++i)
		mAliasView->addTopLevelItem(new QTreeWidgetItem());

	while (mAliasView->topLevelItemCount() > (int)aliases.size())
		delete mAliasView->topLevelItem(0);

	// Create mapping "to --> [from]", i.e. for each "real" resource, which "virtual"
	// resources point to it?
	std::map<std::string, std::vector<std::pair<std::string, std::string>>> mapping;
	foreach(const auto& a, aliases)
		mapping[a.second.first.str()].push_back(std::make_pair(a.first.str(), a.second.second));

	// Insert rows for all mappings as created above
	int row = 0;
	int basePalette = 0;
	foreach(const auto& a, mapping)
	{
		QTreeWidgetItem* fromItem = mAliasView->topLevelItem(row);
		fromItem->setText(0, QString::fromStdString(a.first));

		const QBrush& brush = (basePalette ? palette().base() : palette().alternateBase());
		basePalette = 1 - basePalette;

		// Do all the "to" bits
		foreach(const auto& to, a.second)
		{
			QTreeWidgetItem* toItem = mAliasView->topLevelItem(row);
			toItem->setText(1, QString::fromStdString(to.first));
			QString src = QString::fromStdString(to.second);
			toItem->setText(2, src);

			toItem->setToolTip(1, src);
			toItem->setToolTip(2, src);

			toItem->setBackground(0, brush);
			toItem->setBackground(1, brush);
			toItem->setBackground(2, brush);

			++row;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::DefinitionsListView, mira::EditorPart);
