/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CaptureView.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   12.01.2011
 */

#include <QMessageBox>
#include <QFormLayout>

#include <serialization/Serialization.h>
#include <fw/Framework.h>

#include <views/CaptureView.h>
#include <views/Visualization3DView.h>
#include <widgets/QtUtils.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::CaptureView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

CaptureView::CaptureView() : ui(NULL)
{
}

CaptureView::~CaptureView()
{
	if(getSite()!=NULL)
		getSite()->removePartListener(this);
}


QWidget* CaptureView::createPartControl()
{
	getSite()->addPartListener(this);
	ui = new UI(this);
	return ui;
}

void CaptureView::editorClosed(IWorkbenchPart* part)
{
	// disable capturing if editor has changed
	if(ui!=NULL)
		ui->btCapture->setChecked(false);
}
void CaptureView::editorActivated(IWorkbenchPart* part)
{
	editorClosed(part);
}

///////////////////////////////////////////////////////////////////////////////

CaptureView::UI::UI(CaptureView* iView) :
	QWidget(iView),
	view(iView),
	authority("/", "CaptureView", Authority::ANONYMOUS | Authority::HIDDEN)
{
	editorToCapture = NULL;
	assert(view!=NULL);
	imageChannel = authority.publish<Img<>>("CaptureImage");
	setupUi();
}

void CaptureView::UI::setupUi()
{

	QHBoxLayout* hLayout = new QHBoxLayout(this);

	splitter = new QSplitter(this);
	hLayout->addWidget(splitter);

	QWidget* widget = new QWidget;
	splitter->addWidget(widget);

	mainLayout = new QGridLayout(widget);
	mainLayout->setContentsMargins(0, 0, 0, 0);

	// 1st row
	mainLayout->addWidget(new QLabel(tr("Source:"),widget), 0, 0);

	cbSource = new QComboBox(widget);
	cbSource->addItem(tr("Current Editor"));
	cbSource->addItem(tr("Workspace"));
	//cbSource->addItem(tr("Window")); // not yet implemented
	mainLayout->addWidget(cbSource, 0, 1);

	// 2nd row
	mainLayout->addWidget(new QLabel(tr("Output:"),widget), 1, 0);

	hLayout = new QHBoxLayout();
	hLayout->setContentsMargins(0, 0, 0, 0);

	cbOutput = new QComboBox(widget);
	cbOutput->addItem("Images");
	cbOutput->addItem("Video");
	cbOutput->addItem("Channel");
	hLayout->addWidget(cbOutput);

	sbFramerate = new QSpinBox(widget);
	sbFramerate->setRange(1,50);
	sbFramerate->setValue(12);
	hLayout->addWidget(sbFramerate);
	hLayout->addWidget(new QLabel(tr("fps"),widget));
	mainLayout->addLayout(hLayout, 1, 1);


	// 3rd row
	mainLayout->addWidget(new QLabel(tr("Path:"),widget), 2, 0);

	hLayout = new QHBoxLayout();
	hLayout->setContentsMargins(0, 0, 0, 0);
	leOutputPath = new QLineEdit(widget);

	btOutputSelect = new QToolButton(widget);
	btOutputSelect->setText("...");
	hLayout->addWidget(leOutputPath);
	hLayout->addWidget(btOutputSelect);
	mainLayout->addLayout(hLayout, 2, 1);

	// 4th row
	hLayout = new QHBoxLayout();
	hLayout->setContentsMargins(0, 0, 0, 0);
	btCapture = new QPushButton(widget);
	btCapture->setText(tr("Capture"));
	btCapture->setCheckable(true);

	sbOutputFrame = new QSpinBox(widget);
	sbOutputFrame->setMaximum(std::numeric_limits<int>::max());

	hLayout->addWidget(btCapture);
	hLayout->addWidget(sbOutputFrame);

	mainLayout->addLayout(hLayout, 3, 1);



	// tab widget

	settingsTabWidget = new QTabWidget;
	settingsTabWidget->setTabPosition(QTabWidget::East);
	splitter->addWidget(settingsTabWidget);

	// Image Tab
	imageSettingsTab = new QWidget;
	//settingsTabWidget->addTab(imageSettingsTab, "Images");

	QFormLayout *formLayout = new QFormLayout(imageSettingsTab);

	cbFormat = new QComboBox(imageSettingsTab);
	cbFormat->addItem("JPG");
	cbFormat->addItem("BMP");
	cbFormat->addItem("PNG");
	formLayout->addRow("Format:", cbFormat);

	sbImageQuality = new QSpinBox(widget);
	sbImageQuality->setRange(0,100);
	sbImageQuality->setValue(100);
	formLayout->addRow("Quality:", sbImageQuality);


	// Video Tab
	videoSettingsTab = new QWidget;
	//settingsTabWidget->addTab(videoSettingsTab, "Video");

	QVBoxLayout* vLayout = new QVBoxLayout(videoSettingsTab);
	vLayout->setContentsMargins(0, 0, 0, 0);
	hLayout = new QHBoxLayout();
	hLayout->setContentsMargins(0, 0, 0, 0);
	hLayout->addWidget(new QLabel(tr("Format:"),videoSettingsTab));

	cbVideoFormat = new QComboBox(videoSettingsTab);

	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap codecClasses = video::AbstractVideoOutput::CLASS().getDerivedClasses();
	foreach(ClassMap::value_type i, codecClasses)
	{
		if(i.second.isAbstract())
			continue;
		if (i.second.getMetaInfo("Codec").empty())
			continue;
		cbVideoFormat->addItem(i.second.getMetaInfo("Codec").c_str(),i.first.c_str());
	}
	hLayout->addWidget(cbVideoFormat);
	vLayout->addLayout(hLayout);


	mPropertyEditor = new PropertyEditor(this);
	mPropertyEditor->setHideSingleRootNode(true);
	mPropertyEditor->hideFilter();

	vLayout->addWidget(mPropertyEditor);


	connect(&captureTimer, SIGNAL(timeout()), this, SLOT(captureNext()));
	connect(btCapture, SIGNAL(toggled(bool)), this, SLOT(capture(bool)) );
	connect(cbOutput, SIGNAL(currentIndexChanged(int)), this, SLOT(outputChanged(int)));
	connect(cbVideoFormat, SIGNAL(currentIndexChanged(int)), this, SLOT(videoFormatChanged(int)));

	outputChanged(0);
	videoFormatChanged(cbVideoFormat->currentIndex());
}

void CaptureView::UI::choosePath()
{

}

void CaptureView::UI::outputChanged(int idx)
{
	settingsTabWidget->removeTab(0);
	videoSettingsTab->hide();
	imageSettingsTab->hide();

	switch(idx) {
	case 0:
		settingsTabWidget->insertTab(0, imageSettingsTab, "Images");
		imageSettingsTab->show();
		leOutputPath->setText("image");
		break;

	case 1:
		settingsTabWidget->insertTab(0, videoSettingsTab, "Videos");
		videoSettingsTab->show();
		leOutputPath->setText("video.avi");
		if(cbVideoFormat->count()==0)
			QMessageBox::warning(this, "Missing Codecs", "No video codecs available. Please install some codecs first, e.g. by installing the FFmpeg package");
		break;
	}

}


void CaptureView::UI::videoFormatChanged(int idx)
{
	if(idx<0)
		return;

	QString codec = cbVideoFormat->itemData(idx).toString();
	auto videoClass = ClassFactory::instance().getClassByIdentifier(codec.toStdString());
	videoOut.reset(dynamic_cast<video::AbstractVideoOutput*>(videoClass.newInstance()));
	if(!videoOut)
		MIRA_THROW(XInvalidParameter, "Could not create codec '" << codec.toStdString() << "'");

	mPropertyEditor->clear();
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties("Codec", videoOut);
	mPropertyEditor->addProperty(p);
}

void CaptureView::UI::captureNext()
{
	QString format = cbFormat->currentText();
	QByteArray t = format.toUtf8();
	const char* formatc = t.constData();

	QString extension = format.toLower();

	assert(editorToCapture!=NULL);
	QImage img;
	if(cbSource->currentIndex()==0)
		img = editorToCapture->captureContent();
	else {
		IWorkbenchPartSite* site = editorToCapture->getSite();
		std::list<IWorkbenchPart*> l = site->getEditors();
		std::list<EditorPart*> editors;
		foreach(IWorkbenchPart* e, l)
		{
			EditorPart* editor = dynamic_cast<EditorPart*>(e);
			if(editor)
				editors.push_back(editor);
		}

		QRect rect;
		foreach(EditorPart* editor, editors)
		{
			QRect r(editor->pos(), editor->size());
			rect |= r;
		}

		rect.setHeight(rect.height()-40);
		img = QImage(rect.size(),QImage::Format_RGB32);
		img.fill(Qt::black);
		QPainter painter(&img);
		foreach(EditorPart* editor, editors)
			painter.drawImage(editor->pos(), editor->captureContent());
	}

	if (cbOutput->currentIndex() == 0)
	{
		QString filename = leOutputPath->text()+QString("_%1.%2").arg(sbOutputFrame->value(),6,10,QChar('0')).arg(extension);
		img.save(filename, formatc, sbImageQuality->value());
	} else if (cbOutput->currentIndex() == 2)
	{
		auto w = imageChannel.write();
		QtUtils::fromQImage(img, w->value());
	} else if (videoOut && cbOutput->currentIndex() == 1)
	{
		Img<> outImg;
		QtUtils::fromQImage(img, outImg);
		videoOut->encode(outImg, Time::now());
	}
	sbOutputFrame->setValue(sbOutputFrame->value()+1);
}

void CaptureView::UI::capture(bool on)
{
	if(on) {
		assert(view->getSite()!=NULL);

		editorToCapture = dynamic_cast<EditorPart*>(view->getSite()->getActiveEditor());
		bool ok = editorToCapture != NULL;
		if (cbOutput->currentIndex() == 1)
			videoOut->open(leOutputPath->text().toStdString());

		if(ok)
			captureTimer.start(1000/sbFramerate->value());
		else
			btCapture->setChecked(false);
	} else {
		captureTimer.stop();
		editorToCapture = NULL;
		if (videoOut)
			videoOut->close();
	}
}


///////////////////////////////////////////////////////////////////////////////

}

