/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GoalTool2D.C
 *    Tool to specify poses for a INavigation service in the 2D graphics view.
 *
 * @author Christian Reuther
 * @date   2014/08/29
 */

#include <visualization/2d/PoseTool2D.h>

#include <QMouseEvent>

#include <fw/ServiceProperty.h>

#include <navigation/tasks/OrientationTask.h>
#include <navigation/tasks/PositionTask.h>
#include <navigation/tasks/PreferredDirectionTask.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

class GoalTool2D : public PoseTool2D
{
	MIRA_META_OBJECT(GoalTool2D,
	                 ("Name", "Goal")
	                 ("Description", "Specifies poses for a INavigation service")
	                 ("Category", "Navigation")
	                 ("Order", "200"))

public:
	GoalTool2D() : mNavigationService("INavigation") {
		mBaseTask = "{\"SubTasks\" : [ { \"@class\" : \"mira::navigation::PreferredDirectionTask\", \"Direction\" : 1, \"WrongDirectionCost\" : 0.5 } ] }";
	}

	template<typename Reflector> void reflect(Reflector& r) {
		PoseTool2D::reflect(r);

		r.property("NavigationService", mNavigationService, "Service used for sending task to", "");
		r.property("BaseTask", mBaseTask, "The task where only the PositionTask and OrientationTask is added as subtask");
	}

	virtual QIcon getIcon();

protected:
	virtual void onNewPose(const PoseCov2& pose);
	virtual void onMousePressed(QMouseEvent* e);
	std::string getService();

	std::string mBaseTask;
	ServiceProperty mNavigationService;
};

///////////////////////////////////////////////////////////////////////////////

QIcon GoalTool2D::getIcon() {
	return QIcon(QPixmap(":/icons/ToolGoal.png"));
}

void GoalTool2D::onNewPose(const PoseCov2& pose) {
	const std::string service = getService();
	if(service.empty())
		return;

	json::Value v;
	json::read(mBaseTask, v);
	JSONDeserializer js(v);

	TaskPtr task;
	js.deserialize(task);
	task->addSubTask(SubTaskPtr(new PositionTask(pose.t, pose.cov(0,0), pose.cov(0,0)+0.10f, getSite()->getFixedFrame())));
	task->addSubTask(SubTaskPtr(new OrientationTask(pose.phi(), pose.cov(2,2), getSite()->getFixedFrame())));

	auto result = callService<void>(service, "setTask", task);
	result.timedWait(Duration::seconds(10));
	result.get();
}

void GoalTool2D::onMousePressed(QMouseEvent* e) {
	PoseTool2D::onMousePressed(e);

	if(e->button() != Qt::RightButton)
		return;

	const std::string service = getService();
	if(service.empty())
		return;

	// Cancel current task
	auto result = callService<void>(service, "setTask", TaskPtr(new Task));
	result.timedWait(Duration::seconds(10));
	result.get();
}

std::string GoalTool2D::getService() {
	std::string s = mNavigationService;
	// user did not provide a service -> search for interface
	if(s.empty()) {
		auto services = queryServicesForInterface("INavigation");
		s = services.empty() ? "" : services.front();
	}
	return s;
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::navigation::GoalTool2D, mira::VisualizationTool2D)
