/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VectorVisualization.C
 *    Visualization of Eigen::Vector3f, Eigen::Vector3d, Eigen::VectorX
 *
 * @author Erik Einhorn
 * @date   2014/03/16
 */

#include <visualization/Visualization3DBasic.h>

#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/SetterNotify.h>

#include <visualization/3d/ArrowObject.h>

using namespace mira;

namespace mira { namespace gui { 

///////////////////////////////////////////////////////////////////////////////

/**
 * Visualization of Eigen::Vector3f, Eigen::Vector3d, Eigen::VectorX
 */
template <typename TVector>
class VectorVisualizationBase : public Visualization3DBasic<TVector>
{
	typedef TVector Type;
	typedef Visualization3DBasic<Type> Base;

public:

	VectorVisualizationBase();
	virtual ~VectorVisualizationBase();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);

		r.property("Color", mColor, setterNotify(mColor, &VectorVisualizationBase::updateProp, this),
		           "The color.", Ogre::ColourValue::Blue);
	}


	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node);

protected:

	virtual void dataChanged(ChannelRead<Type> data);
	void updateProp();

private:

	ArrowObject* mArrow;
	Ogre::ColourValue mColor;
};

///////////////////////////////////////////////////////////////////////////////

template <typename TVector>
VectorVisualizationBase<TVector>::VectorVisualizationBase() : Base("Vector"), mArrow(NULL)
{
	mColor = Ogre::ColourValue::Blue;
}

template <typename TVector>
VectorVisualizationBase<TVector>::~VectorVisualizationBase()
{
	delete mArrow;
}


template <typename TVector>
void VectorVisualizationBase<TVector>::setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
{
	mArrow = new ArrowObject(mgr, node);
}

////////////////////////////////////////////////////////////////////////////////
// Helpers
template <typename T>
inline Eigen::Vector3f getVector(const Eigen::Matrix<T,2,1>& v) {
	return Eigen::Vector3f(v.x(),v.y(),0.0f);
}

template <typename T>
inline Eigen::Vector3f getVector(const Eigen::Matrix<T,3,1>& v) {
	return v.template cast<float>();
}

template <typename T>
inline Eigen::Vector3f getVector(const Eigen::Matrix<T,Eigen::Dynamic, 1>& v) {

	if(v.rows()==2)
		return Eigen::Vector3f(v.x(),v.y(),0.0f);
	else if(v.rows()>=3)
		return Eigen::Vector3f(v.x(),v.y(),v.z());
	else
		MIRA_THROW(XInvalidParameter, "Eigen::VectorX with " << v.rows() << " rows is not supported");
}

////////////////////////////////////////////////////////////////////////////////

template <typename TVector>
void VectorVisualizationBase<TVector>::dataChanged(ChannelRead<Type> data)
{
	// get the vector
	Eigen::Vector3f v = getVector(data->value());
	float length = v.norm();

	// the ArrowObject aligns the arrow with the y-axis, so get a transform
	// between our vector and the y-axis of the arrow, in order to orient
	// the arrow according to our vector
	Eigen::Quaternionf eq = Eigen::Quaternionf::FromTwoVectors(Eigen::Vector3f::UnitY(),v);
	Ogre::Quaternion q(eq.w(), eq.x(), eq.y(), eq.z());

	mArrow->setOrientation(q);
	mArrow->setLength(length);
	mArrow->setRadius(length*0.03f);
	mArrow->setColor(mColor);
}

template <typename TVector>
void VectorVisualizationBase<TVector>::updateProp()
{
	if(mArrow==NULL)
		return;
	mArrow->setColor(mColor);
}
///////////////////////////////////////////////////////////////////////////////

class VectorXfVisualization : public VectorVisualizationBase<Eigen::VectorXf>
{
MIRA_META_OBJECT(VectorXfVisualization,
			("Name", "VectorXf Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

class VectorXdVisualization : public VectorVisualizationBase<Eigen::VectorXd>
{
MIRA_META_OBJECT(VectorXdVisualization,
			("Name", "VectorXd Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

class Vector3fVisualization : public VectorVisualizationBase<Eigen::Vector3f>
{
MIRA_META_OBJECT(Vector3fVisualization,
			("Name", "Vector3f Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

class Vector3dVisualization : public VectorVisualizationBase<Eigen::Vector3d>
{
MIRA_META_OBJECT(Vector3dVisualization,
			("Name", "Vector3d Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

class Vector2fVisualization : public VectorVisualizationBase<Eigen::Vector2f>
{
MIRA_META_OBJECT(Vector2fVisualization,
			("Name", "Vector2f Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

class Vector2dVisualization : public VectorVisualizationBase<Eigen::Vector2d>
{
MIRA_META_OBJECT(Vector2dVisualization,
			("Name", "Vector2d Visualization")
			("Category", "Vectors")
			("Description", "Visualizes a single vector"))
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::VectorXfVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::gui::VectorXdVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::gui::Vector3fVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::gui::Vector3dVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::gui::Vector2fVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::gui::Vector2dVisualization, mira::Visualization3D );
