/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVectorAnnotationsVisualization.C
 *    Visualization of numbers (in a vector) to Poses (in a vector)
 *
 * @author Christoph Weinrich
 * @date   2011/09/16
 */

#include <OGRE/OgreFrustum.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreHardwarePixelBuffer.h>
#include <image/Img.h>

#include <visualization/3d/TextObject.h>
#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/VisualizationObject.h>

#include <transform/RigidTransform.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class AnnotationVisualization: public TextObject {
public:
	AnnotationVisualization(
			Ogre::SceneManager* iSceneManager,
			Ogre::SceneNode* iSceneNode,
			const Ogre::String &name,
			const Ogre::String &caption,
			const Ogre::String &fontName,
			Ogre::Real charHeight,
			const Ogre::ColourValue &color) :
		TextObject(caption, fontName, iSceneManager, iSceneNode)
	{
		setCharacterHeight(charHeight);
		setColor(color);;
	}

	AnnotationVisualization(
			Ogre::SceneManager* iSceneManager,
			Ogre::SceneNode* iSceneNode,
			const Ogre::String &name,
			const Ogre::String &caption,
			Ogre::Real charHeight,
			const Ogre::ColourValue &color) :
		TextObject(caption, iSceneManager, iSceneNode)
	{
		setCharacterHeight(charHeight);
		setColor(color);;
	}
};

///////////////////////////////////////////////////////////////////////////////

template <typename Annotation, typename PoseType>
class PoseVectorAnnotationsVisualization:
	public Visualization3DBasic<std::vector<Annotation> >
{
	typedef Visualization3DBasic<std::vector<Annotation> > Base;
	typedef std::vector<PoseType, Eigen::aligned_allocator<PoseType>> PoseVectorType;

public:

	PoseVectorAnnotationsVisualization() :
		Base("Text") {
		mPoseCov3PropertyChannel.setDataChangedCallback(boost::bind(
		    &PoseVectorAnnotationsVisualization::onPose3CovVectorChanged, this, _1));
		mNumberOfVisibleTexts = 0;
	}

	virtual ~PoseVectorAnnotationsVisualization() {
	}

public:

	template<typename Reflector>
	void reflect(Reflector& r) {
		Base::reflect(r);
		this->channelProperty(r, "Pose3CovVector", mPoseCov3PropertyChannel,
		                "Optional channel that contains the 3d positions of "
		                "the annotations informations, if not specified here, "
		                "the parameters must be included in the ... info",
		                Base::REQUIRED);
		r.property("Prefix", mPrefix, "prefix", "f=");
	}

public:

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) {
	}

	virtual std::string castToString(Annotation annotation){
		return toString(annotation, 2);
	}

	void onDataChanged(ChannelRead<std::vector<Annotation> > data) {
		if (!mPoseCov3Channel.isValid()) {
			MIRA_THROW(XLogical, "!mPoseCov3Channel.isValid()");
			return;
		}

		auto poseCov3Channel = mPoseCov3Channel.read(data->timestamp);
		if (poseCov3Channel->size() != data->size()) {
			MIRA_THROW(XLogical, "poseCov3Channel->size() != data->size()");
			return;
		}

		while (mAnnotationVisualizations.size() < data->size()) {
			Ogre::SceneManager* sceneManager =
					this->getSite()->getSceneManager();

			boost::shared_ptr<AnnotationVisualization> vis;
			vis.reset(new AnnotationVisualization(sceneManager, this->mNode,
				"FloatVisualization" + toString(this) + ":Text", "Unknown",
				0.1, Ogre::ColourValue::Black));

			mAnnotationVisualizations.push_back(vis);

			vis->setTextAlignment(TextObject::H_LEFT, TextObject::V_BELOW);
		}

		auto frameTransformation = this->template getTransform<PoseType> (
				poseCov3Channel->frameID, data->frameID,
				poseCov3Channel->timestamp);

		for (uint32 i = 0; i < mAnnotationVisualizations.size(); ++i) {
			if (i < data->size()) {
				mAnnotationVisualizations[i]->setVisible(Base::isEnabled());

				mAnnotationVisualizations[i]->setCaption(mPrefix + castToString(data->at(i)));

				mAnnotationVisualizations[i]->setTransform(
						frameTransformation * (*poseCov3Channel)[i]);

				mNumberOfVisibleTexts = i + 1;
			}
			else
				mAnnotationVisualizations[i]->setVisible(false);
		}
	}

	void onPose3CovVectorChanged(ChannelRead<PoseVectorType> read) {
		if (!mPoseCov3Channel.isValid()) {
			mPoseCov3Channel = this->template subscribe<PoseVectorType> (read.getChannelID());
		};
	}

	virtual void setEnabled(bool enabled) {
		Base::setEnabled(enabled);
		for (uint32 i = 0; i < mAnnotationVisualizations.size(); ++i) {
			if (i < mNumberOfVisibleTexts) {
				mAnnotationVisualizations[i]->setVisible(Base::isEnabled());
			}
			else
				mAnnotationVisualizations[i]->setVisible(false);
		}
	}

protected:
	ChannelProperty<PoseVectorType> mPoseCov3PropertyChannel;
	Channel< PoseVectorType> mPoseCov3Channel;

	std::vector<boost::shared_ptr<AnnotationVisualization>> mAnnotationVisualizations;

	uint mNumberOfVisibleTexts;

	std::string mPrefix;
};

///////////////////////////////////////////////////////////////////////////////

#define CREATE_ANNOTATION_VISUALIZATION(posetype,annotationtype,name)         \
class AnnotationVisualization_##annotationtype##At##posetype :                \
    public PoseVectorAnnotationsVisualization<annotationtype, posetype>       \
{                                                                             \
    MIRA_META_OBJECT(AnnotationVisualization_##annotationtype##At##posetype,  \
        ("Name", name)                                                        \
        ("Category", "Text")                                                  \
        ("Description", "Visualizes a vector of numbers positioned at a "     \
                        "vector of poses"))                                   \
};

CREATE_ANNOTATION_VISUALIZATION(PoseCov3, float,
                                "Float Annotations Vector to PoseCov3Vector")
CREATE_ANNOTATION_VISUALIZATION(PoseCov2, float,
                                "Float Annotations Vector to PoseCov2Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose3, float,
                                "Float Annotations Vector to Pose3Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose2, float,
                                "Float Annotations Vector to Pose2Vector")

CREATE_ANNOTATION_VISUALIZATION(PoseCov3, uint32,
                                "Unsigned Integer Annotations Vector to PoseCov3Vector")
CREATE_ANNOTATION_VISUALIZATION(PoseCov2, uint32,
                                "Unsigned Integer Annotations Vector to PoseCov2Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose3, uint32,
                                "Unsigned Integer Annotations Vector to Pose3Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose2, uint32,
                                "Unsigned Integer Annotations Vector to Pose2Vector")

CREATE_ANNOTATION_VISUALIZATION(PoseCov3, int,
                                "Integer Annotations Vector to PoseCov3Vector")
CREATE_ANNOTATION_VISUALIZATION(PoseCov2, int,
                                "Integer Annotations Vector to PoseCov2Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose3, int,
                                "Integer Annotations Vector to Pose3Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose2, int,
                                "Integer Annotations Vector to Pose2Vector")

using namespace std;

CREATE_ANNOTATION_VISUALIZATION(PoseCov3, string,
                                "String Annotations Vector to PoseCov3Vector")
CREATE_ANNOTATION_VISUALIZATION(PoseCov2, string,
                                "String Annotations Vector to PoseCov2Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose3, string,
                                "String Annotations Vector to Pose3Vector")
CREATE_ANNOTATION_VISUALIZATION(Pose2, string,
                                "String Annotations Vector to Pose2Vector")

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_floatAtPoseCov3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_floatAtPoseCov2,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_floatAtPose3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_floatAtPose2,
                         mira::Visualization3D)

MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_uint32AtPoseCov3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_uint32AtPoseCov2,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_uint32AtPose3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_uint32AtPose2,
                         mira::Visualization3D)

MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_intAtPoseCov3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_intAtPoseCov2,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_intAtPose3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_intAtPose2,
                         mira::Visualization3D)

MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_stringAtPoseCov3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_stringAtPoseCov2,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_stringAtPose3,
                         mira::Visualization3D)
MIRA_CLASS_SERIALIZATION(mira::gui::AnnotationVisualization_stringAtPose2,
                         mira::Visualization3D)
