/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LineVectorVisualization.C
 *    Visualization for vectors of lines.
 *
 * @author Christian Reuther
 * @date   2014/06/10
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/LineListObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

#include <geometry/Line.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/**
 * A vector of lines visualization.
 */
template <typename T>
class LineVectorVisualization3D : public Visualization3DBasic<std::vector<T>>
{
	typedef std::vector<T> VectorType;

public:
	LineVectorVisualization3D() : mLineList(NULL) {
		mColor = Ogre::ColourValue::Black;
		mWidth = 0.1f;
	}

	virtual ~LineVectorVisualization3D() {
		delete mLineList;
	}

	template <typename Reflector> void reflect(Reflector& r) {
		Visualization3DBasic<VectorType>::reflect(r);
		r.property("Color", mColor, setter<Ogre::ColourValue>(&LineVectorVisualization3D::setColor, this), "The color of the lines", Ogre::ColourValue::Black);
		r.property("Width", mWidth, setter<float>(&LineVectorVisualization3D::setWidth, this), "The width of the lines", 0.1f);
	}

public:
	void setColor(const Ogre::ColourValue& c) {
		mColor = c;

		if(!mLineList)
			return;

		mLineList->setColor(mColor);
	}

	void setWidth(float width) {
		mWidth = std::max<float>(std::numeric_limits<float>::epsilon(), width);

		if(!mLineList)
			return;

		mLineList->setLineWidth(width);
	}

public:
	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) {
		mLineList = new LineListObject(mgr, node);
		mLineList->setColor(mColor);
		mLineList->setLineWidth(mWidth);
		mLineList->setVisible(this->isEnabled());
	}

	virtual void setEnabled(bool enabled) {
		Visualization3DBasic<VectorType>::setEnabled(enabled);

		if(!mLineList)
			return;

		mLineList->setVisible(enabled);
	}

	void dataChanged(ChannelRead<VectorType> lines) {
		if(!mLineList)
			return;

		mLineList->clear();
		foreach(const T& line, lines->value()) {
			mLineList->add(Ogre::Vector3(line.front()(0), line.front()(1), 0.0),
				Ogre::Vector3(line.back()(0), line.back()(1), 0.0));
		}

		mLineList->setVisible(this->isEnabled());
	}

protected:
	LineListObject* mLineList;
	Ogre::ColourValue mColor;
	float mWidth;
};

///////////////////////////////////////////////////////////////////////////////

class Line2iVectorVisualization3D : public LineVectorVisualization3D<Line<int, 2>> {
	MIRA_META_OBJECT(Line2iVectorVisualization3D,
		("Name", "Vector of Line2i")
		("Description", "Visualizes a 2D int line vector in the xy-plane")
		("Category", "Geometry"))
};

class Line2fVectorVisualization3D : public LineVectorVisualization3D<Line<float, 2>> {
	MIRA_META_OBJECT(Line2fVectorVisualization3D,
		("Name", "Vector of Line2f")
		("Description", "Visualizes a 2D float line vector in the xy-plane")
		("Category", "Geometry"))
};

class Line2dVectorVisualization3D : public LineVectorVisualization3D<Line<double, 2>> {
	MIRA_META_OBJECT(Line2dVectorVisualization3D,
		("Name", "Vector of Line2d")
		("Description", "Visualizes a 2D double line vector in the xy-plane")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Line2iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Line2fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Line2dVectorVisualization3D, mira::Visualization3D);
