/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file InteractionTool.C
 *    Implementation of InteractionTool.
 *
 * @author Erik Einhorn
 * @date   2012/11/15
 */

#include <QMouseEvent>

#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>

#include <visualization/VisualizationTool3D.h>
#include <visualization/InteractionListener3D.h>

#include <widgets/OgreUtils.h>

#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class InteractionTool : public VisualizationTool3D
{
	MIRA_META_OBJECT(InteractionTool,
			("Name", "Interact")
			("Category", "Interact")
			("Description", "Allows to interact with the scene")
			("Default", "true")
			("Order", "3"))

public:

	InteractionTool() :
		mCurrentObject(NULL)
	{
	}

	virtual QIcon getIcon()
	{
		QPixmap pixmap(":/icons/ToolInteraction.png");
		return QIcon(pixmap);
	}

	/// Called if the tool is active and a mouse button was pressed.
	virtual void onMousePressed(QMouseEvent* e, float devicePixelRatio)
	{
		auto view = getSite()->getSceneManager()->getCurrentViewport();
		auto camera = view->getCamera();
		Ogre::Ray mouseRay = camera->getCameraToViewportRay(e->posF().x()*devicePixelRatio/view->getActualWidth(),
		                                                    e->posF().y()*devicePixelRatio/view->getActualHeight());

		Ogre::RaySceneQueryResult resultCoarse;
		OgreUtils::rayQuery(getSite()->getSceneManager()->getRootSceneNode(),mouseRay,resultCoarse);

		// result now contains all objects whose bounding boxes are hit
		// now perform a fine grained analysis down to polygon level
		Ogre::RaySceneQueryResult resultFine;
		foreach(auto v, resultCoarse)
		{
			if(v.movable && v.movable->isVisible()) // visible objects only
			{
				auto collisionResult = OgreUtils::rayObjectCollision(mouseRay,v.movable);
				if(collisionResult.first) {
					v.distance = collisionResult.second;
					resultFine.push_back(v);
				}
				else
					resultFine.push_back(v);
			}
		}

		// sort from near to far
		std::sort(resultFine.begin(), resultFine.end());

		mCurrentObject = NULL;

		// go through each hit object and notify the listeners, the
		// first object that is handled by a listener is chosen as current
		// object
		bool handled = false;
		auto listeners = getSite()->getInteractionListeners();
		foreach(const auto& i, resultFine) // from near to far
		{
			assert(i.movable);

			foreach(InteractionListener3D* l, listeners) {
				if (l->onMousePressed(e, i.movable, i.distance, &mouseRay))
					handled = true;
			}

			if(handled) {
				mCurrentObject = i.movable;
				break; // skip the rest, since we have a forground object selected
			}
		}

		// if no listener has handled the object, then perform some default
		// action and toggle the bounding box of the nearest object
		if (!resultFine.empty() && !handled && e->button() == Qt::LeftButton)
		{
			// chose nearest object by default
			mCurrentObject = resultFine.front().movable;
			bool show = mCurrentObject->getParentSceneNode()->getShowBoundingBox();
			mCurrentObject->getParentSceneNode()->showBoundingBox(!show);
		}
	}

	virtual void onMouseReleased(QMouseEvent* e)
	{
		if (mCurrentObject)
		{
			auto listeners = getSite()->getInteractionListeners();
			foreach(InteractionListener3D* l, listeners)
				l->onMouseReleased(e, mCurrentObject);
		}
		mCurrentObject = NULL;
	}

	virtual void onMouseMoved(QMouseEvent* e)
	{
		if (mCurrentObject)
		{
			auto listeners = getSite()->getInteractionListeners();
			foreach(InteractionListener3D* l, listeners)
				l->onMouseMoved(e, mCurrentObject);
		}
	}

	virtual int getDisplaceShortCut() const
	{
		return Qt::Key_Control;
	}

	void activate()
	{
		mCurrentObject = NULL;
	}

	void deactivate()
	{
	}

	void setupScene(IVisualization3DSite* site)
	{
	}

	virtual void update(Duration dt) {
	}

	Ogre::MovableObject* mCurrentObject;

};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::InteractionTool, mira::VisualizationTool3D)
