/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CameraOrbitTool.C
 *    Implementation of CameraOrbitTool.
 *
 * @author Erik Einhorn
 * @date   2011/07/01
 */

#include <visualization/3d/CameraOrbitTool.h>

#include <math/Saturate.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

inline float mouseAcceleration(float v, float accel)
{
	if(v>=0.0f)
		return pow(v,accel);
	else
		return -pow(-v,accel);
}

///////////////////////////////////////////////////////////////////////////////


CameraOrbitTool3D::CameraOrbitTool3D()
{
	setInteractionSpeed();
	mPivotIndicatorState = NONE;
	mPivotIndicatorAlpha = 0.0f;
}

QIcon CameraOrbitTool3D::getIcon()
{
	QPixmap pixmap(":/icons/ToolCameraOrbit.png");
	return QIcon(pixmap);
}

void CameraOrbitTool3D::onMousePressed(QMouseEvent* e)
{
	mOldMousePos = e->globalPos();
	updateOwnRepresentation();
	if(mPivotIndicator && e->button()) {
		updatePivotIndicator();
		mPivotIndicatorState = FADE_IN;
		mPivotIndicator->setVisible(true);
	}
}

void CameraOrbitTool3D::onMouseReleased(QMouseEvent* e) {
	if(mPivotIndicator
			&& !(e->buttons() & Qt::LeftButton)
			&& !(e->buttons() & Qt::RightButton))
		mPivotIndicatorState = FADE_OUT;
}

void CameraOrbitTool3D::onMouseMoved(QMouseEvent* e) {
	// obtain the mouse movement since the last event
	QPoint delta = e->globalPos() - mOldMousePos;

	float dx, dy, ds;
	switch (e->buttons()) {
	case Qt::LeftButton:
		dx = -mouseAcceleration(delta.x(), mMouseAcceleration) * mSpeedRot;
		dy = mouseAcceleration(delta.y(), mMouseAcceleration) * mSpeedRot;

		mPitch -= dy;

		if(mPitch > cMaxPitch)
			mPitch = cMaxPitch;
		if(mPitch < -cMaxPitch)
			mPitch = -cMaxPitch;

		mYaw += dx;

		updateCamera();
		break;

	case Qt::RightButton: {
		dx = - mouseAcceleration(delta.x(), mMouseAcceleration) * mSpeedTrans;
		dy = - mouseAcceleration(delta.y(), mMouseAcceleration) * mSpeedTrans;

		dy *= mCamera.distance;
		dx *= mCamera.distance;

		// translate the camera respective to the current orientation
		if(e->modifiers() & Qt::ShiftModifier) {
			mFocal.x() +=  -sinf(mYaw) * dx;
			mFocal.y() +=  cosf(mYaw) * dx;
			mFocal.z() +=  -dy;

		} else {
			mFocal.x() +=  cosf(mYaw) * dy -
						   sinf(mYaw) * dx;
			mFocal.y() +=  sinf(mYaw) * dy +
						   cosf(mYaw) * dx;
		}
		updatePivotIndicator();
		updateCamera();
		break;
	}

	case (Qt::LeftButton | Qt::RightButton):
	case Qt::MidButton:
		ds = mouseAcceleration(delta.y(), mMouseAcceleration) * mSpeedScale;
		mCamera.distance += ds * mCamera.distance;

		updatePivotIndicator();
		updateCamera();
		break;
	}

	mOldMousePos = e->globalPos();
}

void CameraOrbitTool3D::onMouseWheel(QWheelEvent * e) {
	updateOwnRepresentation();

	// In Qt5 we have an option to discern whether this is a high resolution scroll
	// (e.g. two fingers on a trackpad), or a "regular" angle-based mouse wheel scroll.
	// Alas we cannot use it because Qt4 doesn't offer this.
	// The angle-based scrolling seems to give errors in determining the delta at
	// least on Ubuntu 16.04 using Qt5 (hence using the deprecated delta() method).
	// Therefore we limit the scroll speed to 120, which is the typical resolution
	// for an entire 15° step (supposedly the standard for most mice). As this event
	// should be called at every turn, this will work as expected in 99% of all cases.

	const float delta = saturate<float>(e->delta(), -120.0f, 120.0f);
	const float ds = mouseAcceleration(delta, mMouseAcceleration) * mSpeedWheelScale;
	mCamera.distance -= ds * mCamera.distance;
	updatePivotIndicator();
	updateCamera();
}

void CameraOrbitTool3D::setInteractionSpeed(float trans, float rot,
                                            float scale, float wheelScale,
                                            float acceleration)
{
	mSpeedTrans = trans;
	mSpeedRot   = rot;
	mSpeedScale = scale;
	mSpeedWheelScale = wheelScale;
	mMouseAcceleration = acceleration;
}

int CameraOrbitTool3D::getDisplaceShortCut() const
{
	return Qt::Key_Control;
}

void CameraOrbitTool3D::activate()
{
	mPivotIndicatorAlpha = 0.0f;
	updateOwnRepresentation();
}

void CameraOrbitTool3D::deactivate()
{
	if(mPivotIndicator)
		mPivotIndicator->setVisible(false);
	mPivotIndicatorState=NONE;
}

void CameraOrbitTool3D::setupScene(IVisualization3DSite* site)
{
	VisualizationTool3D::setupScene(site);
	mPivotIndicator.reset(new MeshObject("Sphere.mesh", site->getSceneManager()));
	mPivotIndicator->setColor(Ogre::ColourValue(1,1,0, 0.8f));
	mPivotIndicator->getMaterial()->setDepthWriteEnabled(false);
	mPivotIndicator->setVisible(false);
}


void CameraOrbitTool3D::updatePivotIndicator()
{
	if(!mPivotIndicator)
		return;

	if(mPivotIndicatorAlpha>cFadeAlpha) {
		mPivotIndicatorAlpha=cFadeAlpha;
		mPivotIndicatorState=NONE;
	}

	if(mPivotIndicatorAlpha<0.0f) {
		mPivotIndicatorAlpha=0.0f;
		mPivotIndicatorState=NONE;
	}

	if(mPivotIndicatorAlpha==0.0f) {
		mPivotIndicator->setVisible(false);
		return;
	}

	mPivotIndicator->setPosition(Eigen::Vector3f(mFocal.x(),
	                                             mFocal.y(),
	                                             mFocal.z()));
	float s = 0.03f * mCamera.distance;
	mPivotIndicator->setScale(Eigen::Vector3f(s,s,s*0.3));
	mPivotIndicator->setColor(Ogre::ColourValue(1,1,0, mPivotIndicatorAlpha));
}

void CameraOrbitTool3D::update(Duration dt) {
	float dAlpha= cFadeAlpha * dt.totalMilliseconds() / cFadeDuration;
	switch(mPivotIndicatorState) {
	case NONE:
		break;
	case FADE_IN:
		mPivotIndicatorAlpha+=dAlpha;
		updatePivotIndicator();
		break;
	case FADE_OUT:
		mPivotIndicatorAlpha-=dAlpha;
		updatePivotIndicator();
		break;
	}
}

void CameraOrbitTool3D::updateOwnRepresentation()
{
	if(getSite()==NULL)
		return;

	mCamera = getSite()->getCamera();
	Eigen::Quaternionf q = quaternionFromYawPitchRoll(mCamera.yaw,
	                                                  mCamera.pitch,
	                                                  mCamera.roll);
	Eigen::Vector3f dir = q * Eigen::Vector3f::UnitX();
	// should be normalized already but somehow we run into rounding problems
	// where dir.z() is outside of the interval [-1,1] and asin below returns NaN
	// resulting in a segfault later on
	dir.normalize();

	mFocal = mCamera.position + dir*mCamera.distance;

	Eigen::Vector3f d = -dir;

	mPitch = -asin( d.z() );
	mYaw   = atan2( d.y(), d.x());
}

void CameraOrbitTool3D::updateCamera()
{
	Eigen::Quaternionf q = quaternionFromYawPitchRoll(mYaw,mPitch, 0.0f);
	Eigen::Vector3f dir = q * Eigen::Vector3f::UnitX();
	Eigen::Vector3f p = mFocal + dir*mCamera.distance;

	mCamera.lookAt(p, mFocal);
	getSite()->setCamera(mCamera);
}


const float CameraOrbitTool3D::cFadeDuration = 300.0f; // in [ms]
const float CameraOrbitTool3D::cFadeAlpha = 0.6f;
const float CameraOrbitTool3D::cMaxPitch = 0.4998f * pi<float>();

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::CameraOrbitTool3D, mira::VisualizationTool3D)
