/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationView.C
 *    Implementation of VisualizationView.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/29
 */

#include <views/VisualizationView.h>

#include <iterator>
#include <map>

#include <QTimer>
#include <QMimeData>
#include <QMessageBox>
#include <QMenu>
#include <QDragEnterEvent>
#include <QToolBar>
#include <QToolButton>
#include <QBoxLayout>
#include <QPushButton>
#include <QButtonGroup>
#include <QAction>
#include <QShowEvent>
#include <QFileDialog>

#include <serialization/Serialization.h>
#include <xml/XMLDomReflect.h>

#include <visualization/Visualization.h>
#include <visualization/ChannelDragnDropUtils.h>

#include <views/VisualizationControlPage.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::VisualizationView, mira::EditorPart);
MIRA_CLASS_SERIALIZATION(mira::VisualizationViewTransformable, mira::VisualizationView);

namespace mira {

///////////////////////////////////////////////////////////////////////////////

VisualizationView::VisualizationView() : mControl(NULL), DEFAULT_CATEGORY("default")
{
	mLayout = NULL;
	mToolBar = NULL;
	mButtonGroup = NULL;
	mVisualizationPart = NULL;

	mActiveTool = NULL;
	mDisplacedTool = NULL;

	mHideToolBar = new QAction(this);
	mHideToolBar->setCheckable(true);
	mHideToolBar->setChecked(false);
	mHideToolBar->setText("Hide Tools");
}

VisualizationView::~VisualizationView()
{
	delete mControl;
	destroyVisualizations();
	destroyTools();
}

void VisualizationView::destroyVisualizations()
{
	// delete our visualizations
	foreach(Visualization* v, mVisualizations)
		delete v;
	mVisualizations.clear();
}

void VisualizationView::destroyTools()
{
	// delete our visualizations
	foreach(VisualizationTool* t, mTools) {
		delete t;
	}
	mTools.clear();
	mCategoryToToolButton.clear();
	mIdToTool.clear();
	mCurrentTool.clear();
}

void VisualizationView::reflect(XMLDeserializer& r)
{
	EditorPart::reflect(r);
	r.member("Visualizations", mVisualizations, "The assigned visualizations");

	r.property("Update",
	           mUpdateInterval,
	           setter(&VisualizationView::setUpdateInterval,this),
	           "Update interval in ms",
	           40);

	r.property("WindowTitle",
			   getter(&VisualizationView::getWindowTitle,this),
			   setter(&VisualizationView::setWindowTitle,this),
			   "Title of window",
			   getClass().getMetaInfo("Name"));

	r.member("HideTools",
	         getter<bool>(boost::bind(actionGetter, mHideToolBar)),
	         setter<bool>(boost::bind(actionSetter, mHideToolBar, _1)),
	         "Hide tool bar with tools", false);

	r.member("ToolsState", mToolsState, "State of the last used tools");
}

void VisualizationView::reflect(XMLSerializer& r)
{
	EditorPart::reflect(r);
	r.member("Visualizations", mVisualizations, "The assigned visualizations");

	r.property("Update",
	           mUpdateInterval,
	           setter(&VisualizationView::setUpdateInterval,this),
	           "Update interval in ms",
	           40);

	r.property("WindowTitle",
			   getter(&VisualizationView::getWindowTitle,this),
			   setter(&VisualizationView::setWindowTitle,this),
			   "Title of window",
			   getClass().getMetaInfo("Name"));

	r.member("HideTools",
	         getter<bool>(boost::bind(actionGetter, mHideToolBar)),
	         setter<bool>(boost::bind(actionSetter, mHideToolBar, _1)),
	         "Hide tool bar with tools", false);

	XMLDom xml;
	XMLDom::sibling_iterator node = xml.root();
	foreach(VisualizationTool* tool, mTools)
	{
		XMLDom tmpXml;
		XMLSerializer xs(tmpXml);
		xs.serialize(tool->getClass().getMetaInfo("Name").c_str(), tool, "");
		node.add_child(tmpXml.root().begin());
	}

	// disable the serializer's pointer tracking for temp object!
	r.member("ToolsState", xml, "State of the last used tools", REFLECT_CTRLFLAG_TEMP_TRACKING);
}

void VisualizationView::addVisualization(Visualization* vis)
{
	mVisualizations.push_back(vis);
}

void VisualizationView::removeVisualization(Visualization* vis)
{
	mVisualizations.remove(vis);
}

void VisualizationView::moveUpVisualization(Visualization* vis)
{
	if ((mVisualizations.size() < 2) || (mVisualizations.front() == vis))
		return;

	auto it = std::find(++mVisualizations.begin(), mVisualizations.end(), vis);

	if (it == mVisualizations.end())
		return;

	auto pred = it;
	--pred;
	std::swap(*it, *pred);
}

void VisualizationView::moveDownVisualization(Visualization* vis)
{
	if ((mVisualizations.size() < 2) || (mVisualizations.back() == vis))
		return;

	auto it = std::find(mVisualizations.begin(), --mVisualizations.end(), vis);

	if (it == --mVisualizations.end())
		return;

	auto succ = it;
	++succ;
	std::swap(*it, *succ);
}

const std::list<Visualization*>& VisualizationView::getVisualizations() const
{
	return mVisualizations;
}

void VisualizationView::addTool(VisualizationTool* tool)
{
	assert(tool!=NULL);

	// create and add toolbar on demand only (if no tools are available
	// it will not be shown at all)
	if(!mToolBar) {
		mToolBar = new QToolBar("Tools", this);
		mToolBar->setIconSize(QSize(20,20));
		mToolBar->installEventFilter(this);
		mToolBar->setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
		mToolBar->setFocusPolicy(Qt::NoFocus);
		mLayout->addWidget(mToolBar);
	}

	// Get meta information of that tool
	const Class& c = tool->getClass();
	const QString name = QString::fromLocal8Bit(c.getMetaInfo("Name").c_str());
	const QString descr = QString::fromLocal8Bit(c.getMetaInfo("Description").c_str());
	const QString category = getCategory(tool);

	// Add it to our vector
	mTools.push_back(tool);

	// Do we already have a mapping for it?
	if(mCurrentTool.find(category) == mCurrentTool.end()) {
		mCurrentTool.insert(std::make_pair(category, tool));
	}

	// Add it to the QToolBar
	addToolButton(tool, name, category, descr);
}

const std::vector<VisualizationTool*>& VisualizationView::getTools() const
{
	return mTools;
}

VisualizationControlPage* VisualizationView::getControl()
{
	return mControl;
}

Object* VisualizationView::getAdapter(const Class& adapter)
{
	if(adapter == VisualizationControlPage::CLASS())
	{
		if(mControl==NULL)
			mControl = new VisualizationControlPage(this);
		return mControl;
	}

	return EditorPart::getAdapter(adapter);
}

///////////////////////////////////////////////////////////////////////////////

QWidget* VisualizationView::createPartControl()
{
	QWidget* w = new QWidget(this);
	mVisualizationPart = createVisualizationPart();
	mLayout = new QBoxLayout(QBoxLayout::BottomToTop);
	mCurrentToolboxArea = Qt::BottomToolBarArea;
	mButtonGroup = new QButtonGroup(this);
	mButtonGroup->setExclusive(true);

	mLayout->setSpacing(0);
	mLayout->setMargin(0);

	std::multimap<int,VisualizationTool*> orderedTools;

	// collect and instantiate all tools that are supported by our derived
	// VisualiztationView class, for each tool a tool button is added.
	// If there are no tools at all, no toolbar is created.
	const Class& toolClass = this->supportedVisualizationToolClass();
	if(toolClass!=NullClass::null()) {

		auto toolClasses = toolClass.getDerivedClasses();
		foreach(auto p, toolClasses)
		{
			ClassProxy c = p.second;
			if(c.isAbstract()) // skip abstract classes
				continue;

			VisualizationTool* tool;
			// try to restore from serialized state
			try
			{
				XMLDeserializer s(mToolsState);
				s.deserialize(c.getMetaInfo("Name").c_str(), tool);
			}
			catch(...)
			{
				// deserialization failed...let factory create the tool
				tool = c.newInstance<VisualizationTool>();
			}
			assert(tool!=NULL);

			// get order and add to ordered map
			std::string orderStr = c.getMetaInfo("Order");
			int order = std::numeric_limits<int>::max();
			try {
				order = fromString<int>(orderStr);
			} catch(...) {}
			orderedTools.insert(std::make_pair(order,tool));
		}
	}

	// add tools in ordered order
	foreach(auto p,orderedTools)
		addTool(p.second); // will initialize the tool in derived classes and add a button

	// Set the default tools in each category
	for(auto m = mCategoryToToolButton.begin(); m != mCategoryToToolButton.end(); ++m) {
		VisualizationTool* defaultTool = NULL;

		// Get tools of this category so that they can be searched
		for(auto t = orderedTools.begin(); t != orderedTools.end(); ++t) {
			const QString cat = getCategory(t->second);

			if(m->first != cat) {
				continue;
			}

			// Do we have a default tool? This would be the best case
			if(t->second->getClass().getMetaInfo("Default") == "true") {
				defaultTool = t->second;
				break;
			}
		}

		// We have sorted the tools. Now lets see whether we have a default one.
		// If so, we select it, otherwise we take the first one to be added to this
		// category.
		switchToTool(defaultTool ? defaultTool : mCurrentTool[m->first], false);
	}

	// Initially activate the first one of the order
	if(!orderedTools.empty()) {
		activateTool(orderedTools.begin()->second);
	}

	mLayout->addWidget(mVisualizationPart);
	w->setLayout(mLayout);

	// handle hide tools menu
	QMenu* menu = getViewMenu();
	menu->addAction(mHideToolBar);
	connect(mHideToolBar, SIGNAL(toggled(bool)), this, SLOT(onHideToolbar(bool)));
	onHideToolbar(mHideToolBar->isChecked());


	return w;
}

void VisualizationView::resetView()
{
	// Overwrite this method to reset the view, e.g. the camera's perspective and such
}

void VisualizationView::saveContentToFile()
{
	// Overwrite this method to capture the rendered scene and save it to the provided file
}

///////////////////////////////////////////////////////////////////////////////

void VisualizationView::setUpdateInterval(int interval)
{
	mUpdateInterval = interval;
	if(mUpdateTimer.isActive())
	{
		mUpdateTimer.stop();
		startUpdateTimer();
	}
}

int  VisualizationView::getUpdateInterval()
{
	return mUpdateInterval;
}

void VisualizationView::startUpdateTimer()
{
	mLastUpdateTime = Time::now();
	mUpdateTimer.start(mUpdateInterval, this);
}

void VisualizationView::timerEvent(QTimerEvent* e)
{
	if(e->timerId() != mUpdateTimer.timerId())
		return;

	Time time = Time::now();
	Duration dt = time - mLastUpdateTime;
	mLastUpdateTime = time;

	// update the active tool
	if(getActiveTool()!=NULL)
		getActiveTool()->processUpdate(dt);

	// update all of our visualizations
	foreach(Visualization* vis, mVisualizations)
	{
		assert(vis!=NULL);
		vis->processUpdate(dt);
	}

	// update the view itself (call our derived view class)
	update(dt);
}

// event filter in order to listen for mouse events of toolbar
// implements movement of the toolbar and docking at the four corners of the
// window
bool VisualizationView::eventFilter(QObject *obj, QEvent *event)
{
	// border for snapping
	const int border = 35;

	if(obj!=mToolBar)
		return false;

	if(event->type()==QEvent::MouseMove) {
		QMouseEvent* e = static_cast<QMouseEvent*>(event);
		if((e->buttons() & Qt::LeftButton) == 0)
			return false; // dragging with left button only

		setCursor(QCursor(Qt::SizeAllCursor));
		QPoint p = this->mapFromGlobal(QPoint(e->globalPos()));
		if(p.y()<border)
			setToolbarArea(Qt::TopToolBarArea);
		else if(p.y()>(height()-border))
			setToolbarArea(Qt::BottomToolBarArea);
		else if(p.x()<border)
			setToolbarArea(Qt::LeftToolBarArea);
		else if(p.x()>(width()-border))
			setToolbarArea(Qt::RightToolBarArea);
	} else if (event->type()==QEvent::MouseButtonRelease) {
		// make sure to reset the mouse cursor, after moving is finished
		setCursor(QCursor(Qt::ArrowCursor));
	}

	return false;
}

static int keyEventToKeycode(QKeyEvent* event)
{
	int keycode = event->key();

	// if special keys pressed only, return their code
	if(keycode == Qt::Key_Control ||
	   keycode == Qt::Key_Shift   ||
	   keycode == Qt::Key_Alt     ||
	   keycode == Qt::Key_Meta)
		keycode = 0;

	// otherwise add possible modifier codes
	Qt::KeyboardModifiers modifiers = event->modifiers();
	if(modifiers & Qt::ShiftModifier)
		keycode |= Qt::SHIFT;
	if(modifiers & Qt::ControlModifier)
		keycode |= Qt::CTRL;
	if(modifiers & Qt::AltModifier)
		keycode |= Qt::ALT;
	if(modifiers & Qt::MetaModifier)
		keycode |= Qt::META;

	return keycode;
}

static int normalizeKeycode(int code)
{
	int key = code & (~Qt::KeyboardModifierMask);

	// convert keycodes of special keys to modifier flags
	int keycode;
	if(key == Qt::Key_Control)
		keycode = Qt::CTRL;
	else if(key == Qt::Key_Shift)
		keycode = Qt::SHIFT;
	else if(key == Qt::Key_Alt)
		keycode = Qt::ALT;
	else if(key == Qt::Key_Meta)
		keycode = Qt::META;
	else
		keycode = key;

	// add remaining modifier flags
	keycode |= code & Qt::KeyboardModifierMask;

	return keycode;
}

void VisualizationView::keyPressEvent(QKeyEvent* event)
{
	EditorPart::keyPressEvent(event);
	const int keycode = keyEventToKeycode(event);

	// check if a tool is activated to displace another one:
	foreach(VisualizationTool* tool, mTools)
	{
		if(tool == mActiveTool)
			continue; // skip the currently active tool

		const int shortcut = normalizeKeycode(tool->getDisplaceShortCut());
		if(shortcut == keycode)
		{
			mDisplacedTool = mActiveTool;
			activateTool(tool);

			// Set the displaced tool's QToolButton as "active" while it is pressed
			mCategoryToToolButton[getCategory(mDisplacedTool)]->setDown(true);
			break;
		}
	}

	/*
	 * Custom keyboard shortcuts be here.
	 */
	if(keycode == (Qt::Key_R | Qt::CTRL))
		resetView();
	if(keycode == (Qt::Key_P | Qt::CTRL))
		saveContentToFile();
}

void VisualizationView::keyReleaseEvent(QKeyEvent* event)
{
	EditorPart::keyReleaseEvent(event);

	// Separate keycode and modifiers
	int keycode = keyEventToKeycode(event);
	const int modifiers = keycode & Qt::KeyboardModifierMask;
	keycode = keycode & (~Qt::KeyboardModifierMask);

	// Do we need to restore a displaced tool?
	if(mDisplacedTool && mDisplacedTool != mActiveTool) {
		// invert modifier flags, since they are set, if their keys are still
		// pressed. They turn false, if the corresponding key is released, hence
		// we need to invert the flags
		const int inverted = keycode |= (~modifiers) & Qt::KeyboardModifierMask;
		const int shortcut = normalizeKeycode(mActiveTool->getDisplaceShortCut());

		if(inverted & shortcut)
		{
			mCategoryToToolButton[getCategory(mDisplacedTool)]->setDown(false);

			// the shortcut was released, so restore the displaced tool
			activateTool(mDisplacedTool);
			mDisplacedTool = NULL;
		}
	}
}

// We need to check whether the previous tool is still displaced, because e.g.
// pressing CTRL+D first triggers the displacement, and then shows the "Select View"
// window of miracenter. This window will then receive the KeyReleaseEvent, and
// we would be stuck in the displaced state.
void VisualizationView::focusOutEvent(QFocusEvent* event) {
	if(mDisplacedTool == NULL) {
		return;
	}

	mCategoryToToolButton[getCategory(mDisplacedTool)]->setDown(false);

	// the shortcut was released, so restore the displaced tool
	activateTool(mDisplacedTool);
	mDisplacedTool = NULL;
}

void VisualizationView::setToolbarArea(Qt::ToolBarArea area)
{
	if(mCurrentToolboxArea == area)
		return; // nothing to do

	switch(area) {
		case  Qt::LeftToolBarArea:
			mToolBar->setOrientation(Qt::Vertical);
			mLayout->setDirection(QBoxLayout::LeftToRight);
			break;
		case  Qt::RightToolBarArea:
			mToolBar->setOrientation(Qt::Vertical);
			mLayout->setDirection(QBoxLayout::RightToLeft);
			break;
		case  Qt::TopToolBarArea:
			mToolBar->setOrientation(Qt::Horizontal);
			mLayout->setDirection(QBoxLayout::TopToBottom);
			break;
		case  Qt::BottomToolBarArea:
			mToolBar->setOrientation(Qt::Horizontal);
			mLayout->setDirection(QBoxLayout::BottomToTop);
			break;
		default:
			return;
	}
	mCurrentToolboxArea = area;
}

void VisualizationView::addToolButton(VisualizationTool* tool, const QString& name, const QString& category, const QString& description)
{
	assert(mToolBar);
	QToolButton* button = NULL;

	// Check whether we already know a tool button that is for this tools category
	if(mCategoryToToolButton.find(category) == mCategoryToToolButton.end()) {
		button = new QToolButton(mToolBar);
		button->setCheckable(true);
		button->setFocusPolicy(Qt::NoFocus);
		button->setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
		button->setPopupMode(QToolButton::MenuButtonPopup);
		button->setObjectName(category);
		button->setMenu(new QMenu(category));
		button->menu()->setToolTip(trUtf8("\"%1\" category").arg(category));

		// Lets just take the icon of the first tool as an example
		button->setIcon(tool->getIcon());
		button->setText(category);
		mToolBar->addWidget(button);
		connect(button, SIGNAL(clicked()), this, SLOT(onToolButton()));

		// Add it
		mCategoryToToolButton.insert(std::make_pair(category, button));
		mButtonGroup->addButton(button);
	} else {
		button = mCategoryToToolButton[category];
	}

	QAction* toolAction = new QAction(this);
	// Set hopefully unique identifier so we can distinguish it later on.
	const QString id = QString("%1_%2").arg(category).arg(name);
	toolAction->setObjectName(id);
	toolAction->setIcon(tool->getIcon());
	toolAction->setToolTip(description);
	toolAction->setText(name);
	toolAction->setIconVisibleInMenu(true);
	connect(toolAction, SIGNAL(triggered()), this, SLOT(onToolAction()));

	// Add it to the mapping
	mIdToTool.insert(std::make_pair(id, tool));

	// Finalize this by adding it
	button->menu()->addAction(toolAction);
}

void VisualizationView::activateTool(VisualizationTool* tool)
{
	if(tool == NULL || tool == mActiveTool)
		return;

	if(mActiveTool!=NULL) { // deactivate last tool
		mActiveTool->deactivate();
	}

	mActiveTool = tool; // activate new one
	mActiveTool->activate();

	// Activate the tool button of this tool
	switchToTool(tool, true);
}

void VisualizationView::switchToTool(VisualizationTool* tool, bool setChecked) {
	const QString category = getCategory(tool);

	for(auto m = mCategoryToToolButton.begin(); m != mCategoryToToolButton.end(); ++m) {
		if(m->first != category) {
			m->second->setChecked(false);
			continue;
		}

		m->second->setText(QString::fromLocal8Bit(tool->getClass().getMetaInfo("Name").c_str()));
		m->second->setToolTip(QString::fromLocal8Bit(tool->getClass().getMetaInfo("Description").c_str()));
		m->second->setChecked(setChecked);
	}

	// Set this as the current one for that category
	mCurrentTool[category] = tool;
}

void VisualizationView::setWindowTitle( std::string const& title )
{
	EditorPart::setWindowTitle( QString::fromStdString( title ) );
	emit windowTitleChanged(static_cast<EditorPartWindow*>(this));
}

std::string VisualizationView::getWindowTitle() const
{
	return windowTitle().toStdString();
}

void VisualizationView::onToolAction() {
	QAction* action = dynamic_cast<QAction*>(sender());
	if(!action) {
		return;
	}

	// Determine what tool has caused this
	if(mIdToTool.find(action->objectName()) == mIdToTool.end()) {
		return;
	}

	// Activate the sender
	activateTool(mIdToTool.at(action->objectName()));
	mDisplacedTool = mIdToTool.at(action->objectName());
}

void VisualizationView::onToolButton() {
	QToolButton* bt = dynamic_cast<QToolButton*>(sender());
	if(!bt) {
		return;
	}

	for(auto m = mCategoryToToolButton.begin(); m != mCategoryToToolButton.end(); ++m) {
		if(m->second != bt) {
			continue;
		}

		// Activate that tool, if it is not activated right now
		mDisplacedTool = NULL;
		activateTool(mCurrentTool.at(m->first));
	}
}

void VisualizationView::onHideToolbar(bool hide)
{
	if(mToolBar)
		mToolBar->setVisible(!hide);
}

///////////////////////////////////////////////////////////////////////////////

std::string VisualizationView::getChannelIDFromMimeData(const QMimeData* mimeData)
{
	return ChannelDragnDropUtils::getChannelIDFromMimeData(mimeData);
}

std::list<ClassProxy> VisualizationView::getVisualizationsFor(const std::string& channelID)
{
	return ChannelDragnDropUtils::getVisualizationsFor(channelID, supportedVisualizationClass());
}

void VisualizationView::dragEnterEventImpl(QDragEnterEvent *event)
{
	const QMimeData* mimeData = event->mimeData();
	const std::string channelID = getChannelIDFromMimeData(event->mimeData());
	if(channelID.empty()) // wrong MIME type
		return;

	// if a default visualization is provided, we accept all data
	if(defaultVisualizationClass()!=NullClass::null()) {
		event->acceptProposedAction();
		return;
	}

	// otherwise check if we have a visualization for the channel
	auto classes = getVisualizationsFor(channelID);
	if(classes.empty())
		return;
	event->acceptProposedAction();
}

void VisualizationView::dropEventImpl(QDropEvent* event, QWidget* widget)
{
	const QMimeData* mimeData = event->mimeData();
	const std::string channelID = getChannelIDFromMimeData(event->mimeData());
	if(channelID.empty()) // wrong MIME type
		return;

	Visualization* vis = NULL;

	auto classes = getVisualizationsFor(channelID);
	if(classes.empty()) {
		const Class& defaultClass = defaultVisualizationClass();
		if(defaultClass == NullClass::null())
			return;

		// otherwise use the default visualization
		MIRA_LOG(DEBUG) << "Using default-visualization for dropped channel '" << channelID << "'";

		// instantiate the visualization
		vis = defaultClass.newInstance<Visualization>();
	} else {

		MIRA_LOG(DEBUG) << "Found " << classes.size() << " visualization for dropped channel '" << channelID << "'";


		ClassProxy selectedClass;
		if(classes.size()==0)
			return; // no visualization found
		else if(classes.size()==1)
			selectedClass = classes.front();
		else {

			// ask user to select a visualization

			QMenu* menu = new QMenu(widget);
			int i=0;
			foreach(ClassProxy c, classes)
			{
				std::string name = c.getMetaInfo("Name");
				if(name.empty())
					name = c.getName(); // use class name, if no meta name was specified
				QAction* a = menu->addAction(name.c_str());
				a->setData(i);
				++i;
			}

			QAction* chosenAction = menu->exec(widget->mapToGlobal(event->pos()));
			if(chosenAction==NULL)
				return; // user has canceled

			// go to the i-th class that was chosen
			auto l = getVisualizationsFor(channelID);
			auto it = l.begin();
			std::advance(it, chosenAction->data().toInt());
			selectedClass = (*it);
		}

		// instantiate the visualization
		vis = selectedClass.newInstance<Visualization>();
	}

	assert(vis!=NULL);

	// setup the data connection channel using the given channelID
	vis->setupDataConnectionChannel(channelID);
	vis->setName(channelID);

	// add the initialized visualization to us and our property editor
	addVisualization(vis);

	if(getControl())
		getControl()->addVisualizationToEditor(vis);

	event->acceptProposedAction();
}

///////////////////////////////////////////////////////////////////////////////

VisualizationViewTransformable::VisualizationViewTransformable()
{
	mLastAskedFixedFrame = "///some invalid frame id///";
	mLastAskedCameraFrame = "///some invalid frame id///";

	mFrameCheckTimer = new QTimer(this);
	connect(mFrameCheckTimer, SIGNAL(timeout()), this, SLOT(checkFrames()));
	mFrameCheckTimer->start(2500);
}

VisualizationViewTransformable::~VisualizationViewTransformable()
{
}

// checks if fixed frame and camera frame are valid, otherwise tries to find a proper root node
// and asks the user.
void VisualizationViewTransformable::checkFrames()
{

	std::string suggestedFixedFrame;
	if(mFixedFrame.getID() != mLastAskedFixedFrame) {
		if(mFixedFrame.isValid()) {
			 // we have a valid fixed frame, so don't do anything
		} else {
			auto roots = MIRA_FW.getTransformer()->getRootNodes();
			if(!roots.empty()) {
				// find the best root node (the one, with the most decendants)
				uint32 bestCount = 0;
				TransformerBase::AbstractNodePtr bestRoot;
				foreach(auto root, roots)
				{
					uint32 count = root->getDescendantCount();
					if(count > bestCount) {
						bestCount = count;
						bestRoot = root;
					}
				}

				if(bestCount!=0) // if we have a root node with children
					suggestedFixedFrame=bestRoot->getID();
			}  else  {
				// we have no root node (actually no node at all), so we cannot
				// do anything -> abort
				return;
			}
		}
	}

	std::string suggestedCameraFrame;
	if(mCameraFrame.getID() != mLastAskedCameraFrame) {
		if(mCameraFrame.isValid()) {
			// we have a valid camera frame, so don't do anything
			// TODO: check if we have a connection between camera frame and fixed frame
		} else {
			if(!suggestedFixedFrame.empty())
				suggestedCameraFrame = suggestedFixedFrame;
			else
				suggestedCameraFrame = mFixedFrame.getID();
		}
	}

	// set frames without asking, if current set values are empty
	if( (mFixedFrame.empty() || suggestedFixedFrame.empty()) &&
		(mCameraFrame.empty() || suggestedCameraFrame.empty()) )
	{
		if(mFixedFrame.empty())
			mFixedFrame=suggestedFixedFrame;
		if(mCameraFrame.empty())
			mCameraFrame=suggestedCameraFrame;
		return;
	}

	// otherwise, ask the user
	QString text1 = "The current ";

	if(!suggestedFixedFrame.empty())
		text1+= "fixed frame '" + QString(mFixedFrame.getID().c_str()) + "' ";

	QString seem = "seems";
	if(!suggestedFixedFrame.empty() && !suggestedCameraFrame.empty()) {
		text1+= "and ";
		seem="seem";
	}

	if(!suggestedCameraFrame.empty())
		text1+= "camera frame '" + QString(mCameraFrame.getID().c_str()) + "' ";

	text1 += seem + " to be wrong. Shall I perform the following corrections?\n";

	if(!suggestedFixedFrame.empty())
		text1+="    Fixed frame: '" + QString(suggestedFixedFrame.c_str()) + "'\n";
	if(!suggestedCameraFrame.empty())
		text1+="    Camera frame: '" + QString(suggestedCameraFrame.c_str()) + "'\n";
	text1+="\nYou can change the fixed frame and camera frame in the 'Visualization Control' view under the 'General' properties.";

	QMessageBox::StandardButton button =
		QMessageBox::information(this, "Adjust fixed frame / camera frame ?",
		    text1, QMessageBox::Yes, QMessageBox::No);

	if(button == QMessageBox::Yes) {
		if(!suggestedFixedFrame.empty())
			mFixedFrame = suggestedFixedFrame;
		if(!suggestedCameraFrame.empty())
			mCameraFrame = suggestedCameraFrame;
	}

	mLastAskedFixedFrame = mFixedFrame.getID();
	mLastAskedCameraFrame = mCameraFrame.getID();
}

///////////////////////////////////////////////////////////////////////////////

}
