/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ParameterPreparer.C
 *    Prepare plugin used to resolve <parameter> tags in XML documents.
 *
 * @author Tim Langner
 * @date   2011/06/15
 */

#include <error/Logging.h>
#include <utils/StringAlgorithms.h>
#include <xml/XMLDomModifier.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Prepare plugin used to resolve <parameter> tags in XML documents.
 * Parameter preparer has order 100.
 */
class ParameterPreparer : public ConfigurationPreparePlugin
{
	MIRA_OBJECT(ParameterPreparer)
public:
	ParameterPreparer();
	virtual void prepareDocument(XMLDom& xml);

	virtual int getOrder() const { return 100; }

protected:
	XMLDom::iterator findUnit(const std::string& name);
	XMLDom::iterator parseParam(XMLDom::iterator& iNode);
	void processParam(XMLDom::iterator& iNode, const std::string& name,
	                  XMLDom::iterator& destNode, std::vector<std::string>& params);
	void processRemoveParam(XMLDom::iterator& iNode, const std::string& name,
	                  XMLDom::iterator& destNode, std::vector<std::string>& params);
	void preprocessParam(XMLDom::iterator& iNode);

	std::string mNamespace;
	XMLDom::iterator mRoot;
	NameRegistry mNameRegistry;
};

///////////////////////////////////////////////////////////////////////////////

ParameterPreparer::ParameterPreparer() :
	mNamespace("/")
{}

XMLDom::iterator findNSUnit(const std::vector<std::string>& ns, std::size_t index,
                            const std::string& unitName, XMLDom::iterator& iNode)
{
	if (index >= ns.size()-1)
	{
		for (auto n=iNode.begin(); n!=iNode.end(); ++n)
		{
			if (*n == "unit" && n.get_attribute<std::string>("id") == unitName)
				return n;
			if (*n == "pyunit" && n.get_attribute<std::string>("id") == unitName)
				return n;
		// NOTE: do not exit with error here yet, we could still find
		//       a namespace 'without name', i.e. the name '/' below.
		}
	}

	XMLDom::iterator node = iNode;
	for (auto n=node.begin(); n!=node.end(); ++n)
	{
		if (*n == "namespace")
		{
			std::string name = n.get_attribute<std::string>("name");
			std::vector<std::string> nameParts;
			boost::split(nameParts, name, boost::is_from_range('/','/'));

			bool matchesAll=true;
			std::size_t parts=0;
			for(std::size_t i=0; i<nameParts.size(); ++i)
			{
				if(nameParts[i].empty())
					continue; //skip empty parts (even boost::token_compress_on may leave a single empty string)

				if(index+i>=ns.size()-1 || nameParts[i]!=ns[index+i]) {
					// have a missmatch at the i-th part
					matchesAll=false;
					break;
				}
				++parts;
			}

			if(matchesAll) {
				auto found = findNSUnit(ns, index+parts, unitName, n);
				if (found != iNode.end())
					return found;
			}
		}
		else if (*n == "process")
		{
			auto found = findNSUnit(ns, index, unitName, n);
			if (found != iNode.end())
				return found;
		}
	}
	return iNode.end();
}

XMLDom::iterator ParameterPreparer::findUnit(const std::string& name)
{
	std::vector<std::string> nsResult;
	boost::split(nsResult, name, boost::is_from_range('/','/'));
	std::vector<std::string> paramResult;
	boost::split(paramResult, *nsResult.rbegin(), boost::is_from_range('.','.'));

	XMLDom::iterator unit = findNSUnit(nsResult, 1, paramResult[0], mRoot);
	if (unit == mRoot.end())
		MIRA_THROW(XInvalidConfig, "Error in <parameter>. Cannot resolve unit "
		           << name << " at this position in config (current namespace='"
		           << mNamespace << "', maybe you forgot a /)");
	return unit;
}

///////////////////////////////////////////////////////////////////////////////

std::string subStringBetween(const std::string& str, char start, char end)
{
	std::string::size_type startPos = str.find_first_of(start);
	std::string::size_type endPos = str.find_last_of(end);
	if (startPos == std::string::npos || endPos == std::string::npos)
		return "";
	return str.substr(startPos+1, endPos-startPos-1);
}

std::string stripName(const std::string& str)
{
	std::string result;
	std::string::size_type i=0;
	while(i<str.size() && str[i] != '[' && str[i] != '{')
		result += str[i++];
	return result;
}

XMLDom::iterator searchChilds(XMLDom::iterator& searchRoot,
                              const std::vector<std::string>& childs,
                              std::size_t startIndex,
                              std::size_t searchStop)
{
	std::string paramEntry = childs[startIndex];
	std::string paramName = stripName(paramEntry);
	std::string className;
	std::string indexStr;
	XMLDom::iterator searchIt = searchRoot.begin();

	// parameter contains possibly a class attribute
	if (paramEntry.find('{') != std::string::npos)
	{
		className = subStringBetween(paramEntry, '{', '}');
		if (className.empty())
			MIRA_THROW(XInvalidConfig, "Error in <parameter>. Error while parsing class attribute");
	}
	if (paramEntry.find('[') != std::string::npos)
	{
		indexStr = subStringBetween(paramEntry, '[', ']');
		if (indexStr.empty())
			MIRA_THROW(XInvalidConfig, "Error in <parameter>. Access to a collection " << paramName << " has invalid syntax");
	}

	for (; searchIt != searchRoot.end(); ++searchIt)
	{
		if (*searchIt == paramName &&
				(className.empty() ||
				(searchIt.has_attribute("class") && searchIt.get_attribute<std::string>("class") == className)))
		{
			if (!indexStr.empty())
			{
				// index is a key -> collection is a map (key,item)
				if (indexStr.find('\'') != std::string::npos)
				{
					std::string key = subStringBetween(indexStr, '\'', '\'');
					if (key.empty())
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Access to collection " << paramName << " with (key) has invalid syntax");
					// We have <parameter name="/Unit.Collection['key']....">...
					// Look for an entry with <key>key<key>
					XMLDom::iterator keyIt = searchIt.begin();
					for (; keyIt != searchIt.end(); ++keyIt)
					{
						if (*keyIt == "key" &&
							keyIt.content_begin() != keyIt.content_end() &&
							*keyIt.content_begin() == key)
						break;
					}
					// No key was found in map
					if (keyIt == searchIt.end())
						return searchIt.end();
					// we found a key -> next child must be item
					searchIt = ++keyIt;
					if (*searchIt != "item")
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Key "
								   << key << " in collection " << paramName
								   << " has invalid or missing item/value");
				}
				else if (indexStr == "?")
				{
					// ignore
				}
				if (indexStr[0] == '+')
				{
					MIRA_THROW(XInvalidConfig, "Error in <parameter>. +end, +begin and +index operations are not allowed while searching a parameter path using [?]");
				}
				else
				{
					uint32 index = fromString<uint32>(indexStr);
					// now search for nth (index) child with the given name
					if (startIndex>=childs.size()-1)
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Parameter tag for access to items in collection "
														   << paramName << " must have syntax <parameter name=\"Unit.Collection[index].ItemName\">");
					std::string itemName = childs[startIndex+1];
					searchIt = searchIt.find(itemName, index);
					if (searchIt == searchIt.end())
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Item " << itemName
								   << " with index " << index << " not found in collection " << paramName);
					startIndex++;
				}
			}
			if (startIndex == childs.size()-1)
				break;
			XMLDom::iterator child = searchChilds(searchIt, childs, startIndex+1, searchStop);
			if (child == searchRoot.end())
				continue;
			searchIt = child;
			break;
		}
	}

	if (searchIt == searchRoot.end())
		return searchIt;
	if (startIndex >= searchStop)
		return searchRoot;
	return searchIt;
}

void ParameterPreparer::processParam(XMLDom::iterator& iNode, const std::string& name, XMLDom::iterator& destNode, std::vector<std::string>& params)
{
	bool lastWasItem = false;
	for(std::size_t i=1; i<params.size(); ++i)
	{
		std::string paramEntry = params[i];
		std::string paramName = stripName(paramEntry);
		// parameter contains possibly an index for a collection
		if (paramEntry.find('[') != std::string::npos)
		{
			std::string indexStr = subStringBetween(paramEntry, '[', ']');
			if (indexStr.empty())
				MIRA_THROW(XInvalidConfig, "Error in <parameter>. Access to a collection " << paramName << " has invalid syntax");
			if (!lastWasItem)
			{
				XMLDom::iterator parameterNode = destNode.find(paramName);
				if (parameterNode == destNode.end())
					destNode = destNode.add_child(paramName);
				else
					destNode = parameterNode;
				lastWasItem = false;
			}
			// index is a key -> collection is a map (key,item)
			if (indexStr.find('\'') != std::string::npos)
			{
				std::string key = subStringBetween(indexStr, '\'', '\'');
				if (key.empty())
					MIRA_THROW(XInvalidConfig, "Error in <parameter>. Access to collection " << paramName << " with (key) has invalid syntax");
				// We have <parameter name="/Unit.Collection['key']....">...
				// Look for an entry with <key>key<key>
				XMLDom::iterator keyIt = destNode.begin();
				for (; keyIt != destNode.end(); ++keyIt)
				{
					if (*keyIt == "key" &&
						keyIt.content_begin() != keyIt.content_end() &&
						*keyIt.content_begin() == key)
					break;
				}
				// No key was found in map -> create new one and an related item
				if (keyIt == destNode.end())
				{
					destNode.add_child("key").add_content(key);
					destNode = destNode.add_child("item");
				}
				else
				{
					// we found a key -> next child must be item
					destNode = ++keyIt;
					if (*destNode != "item")
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Key "
						           << key << " in collection " << paramName
						           << " has invalid or missing item/value");
				}
				lastWasItem = true;
			} // index is the search operator -> from here on we search for a child in the given path
			else if (indexStr == "?")
			{
				// If name="A.b[?].c.d" then the search operator works as follows:
				// we search for a child in the collection b that has a child c that has a child d
				// We then modify the parameters of d.
				// If the attribute hasChild is specified as hasChild="e.f" we additionally test if
				// d contains a child e that has a child f. But we still modify the parameters of d.
				std::string hasChild = iNode.get_attribute<std::string>("hasChild", "");
				std::vector<std::string> childResult;
				if (!hasChild.empty())
					boost::split(childResult, hasChild, boost::is_from_range('.','.'));
				// if we have no search path (neither in the name attribute nor in the hasChild attribute)
				// the search is invalid
				if (i==params.size()-1 && childResult.empty())
					MIRA_THROW(XInvalidConfig, "Error in <parameter>. When using the search operator "
					           "you need to specify a search path either in the name or in the hasChild attribute");
				// from this position in the search path on we only search for sub childs but do not descend any
				// more as we want to modify parameters relative to the child at searchStop
				std::size_t searchStop = params.size();
				// add the hasChild path to the search path
				params.insert(params.end(), childResult.begin(), childResult.end());
				// the child we want to modify is now the one that matches the search path in the
				// name attribute and contains (if specified) childs that match the search path
				// given in the hasChild attribute
				XMLDom::iterator child = searchChilds(destNode, params, i+1, searchStop);
				if (child == destNode.end())
					MIRA_THROW(XInvalidConfig, "Error in <parameter>. Search path not found: " << name << " , " << hasChild);
				destNode = child;
				break;
			}// index is a number or insert/del operation-> collection is a set, list or vector or
			// just a tag with many childs of the same name
			else
			{
				// we have either
				// 1. <parameter name="/Unit.Collection[index].ItemName....">... or
				// 2. <parameter name="/Unit.Collection[index]"><ItemName>.... or
				// 3. <parameter name="/Unit.Collection[+begin]"><ItemName>.... or
				// 4. <parameter name="/Unit.Collection[+end]"><ItemName>.... or
				// 5. <parameter name="/Unit.Collection[+index]"><ItemName>.... or
				// everything else is invalid.
				std::string itemName;
				// 1. name of item is next parameter in the name attribute
				if (i<params.size()-1)
					itemName = stripName(params[i+1]);
				else // 2. name of item is name of first and only child tag in the <parameter> node
				{
					XMLDom::iterator firstChild = iNode.begin();
					if (firstChild == iNode.end())
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Parameter tag for access to items in collection "
						           << paramName << " must either have syntax <parameter name=\"Unit.Collection[index].ItemName\">... or "
						           "<parameter name=\"Unit.Collection[index]\"><ItemName>...");
					itemName = *firstChild;
				}
				// is it an insert operation?
				if (indexStr[0] == '+')
				{
					if (indexStr == "+begin" || indexStr == "+0")
					{
						XMLDom::iterator tmpNode = destNode.add_child(itemName);
						destNode = destNode.begin().insert_before(tmpNode);
						tmpNode.remove();
					}
					else if (indexStr == "+end")
					{
						destNode = destNode.add_child(itemName);
					}
					else
					{
						uint32 index = fromString<uint32>(indexStr.substr(1, indexStr.size()-1));
						index--;
						// now search for nth (index) child with the given name
						XMLDom::iterator itemNode = destNode.find(itemName, index);
						if (itemNode == destNode.end())
							MIRA_THROW(XInvalidConfig, "Error in <parameter>. Inserting item '" << itemName
							           << "' at position " << index+1 << " to collection " << paramName
							           << " not possible since collection has < " << index+1 << " items ");
						XMLDom::iterator tmpNode = destNode.add_child(itemName);
						destNode = itemNode.insert_after(tmpNode);
						tmpNode.remove();
					}
				}
				// no its just an index
				else
				{
					uint32 index = fromString<uint32>(indexStr);
					// now search for nth (index) child with the given name
					XMLDom::iterator itemNode = destNode.find(itemName, index);
					if (itemNode == destNode.end())
						MIRA_THROW(XInvalidConfig, "Error in <parameter>. Item " << itemName
						           << " with index " << index << " not found in collection " << paramName);
					destNode = itemNode;
				}
				lastWasItem = true;
			}
		} // normal parameter
		else
		{
			if (lastWasItem)
			{
				lastWasItem = false;
				continue;
			}
			XMLDom::iterator parameterNode = destNode.find(paramEntry);
			if (parameterNode == destNode.end())
				destNode = destNode.add_child(paramEntry);
			else
				destNode = parameterNode;
		}
	}
	// if parameter has no childs then overwrite/create content of destination parameter
	if (iNode.begin() == iNode.end())
	{
		// Last parameter was collection item (destNode points to item node)
		// As parameter tag contains only content this is invalid
		if (lastWasItem)
			MIRA_THROW(XInvalidConfig, "Error in <parameter>. Parameter tag for access to items in collections "
			           "must either have syntax <parameter name=\"Unit.Collection[0].ItemName\">... or "
			           "<parameter name=\"Unit.Collection[0]\"><ItemName>...");
		auto it=destNode.content_begin();
		while(it != destNode.content_end())
			it = it.remove();
		for (auto it=iNode.content_begin(); it!=iNode.content_end(); ++it)
			destNode.add_content(*it);
	}
	else // otherwise create/overwrite all childs
	{
		// Last parameter was collection item (destNode points to item node)
		if (lastWasItem)
		{
			// parameter node must only contain a single child
			XMLDom::iterator firstChild = iNode.begin();
			if (++firstChild != iNode.end())
				MIRA_THROW(XInvalidConfig, "Error in <parameter>. Parameter tag for access to items in collections "
				           "with syntax <parameter name=\"Unit.Collection[0]\"><ItemName>... must only contain a single child");
			// this child replaces the item node
			destNode.replace(iNode.begin());
		}
		else // create/overwrite all childs
		{
			for (auto it=iNode.begin(); it!=iNode.end(); ++it)
			{
				auto node = destNode.find(*it);
				// if param does not exist create it
				if (node == destNode.end())
					node = destNode.add_child(*it);
				node.replace(it);
			}
		}
	}
}

void ParameterPreparer::processRemoveParam(XMLDom::iterator& iNode, const std::string& name, XMLDom::iterator& destNode, std::vector<std::string>& params)
{
	bool lastWasItem = false;
	for(std::size_t i=1; i<params.size(); ++i)
	{
		std::string paramEntry = params[i];
		std::string paramName = stripName(paramEntry);
		// parameter contains possibly an index for a collection
		if (paramEntry.find('[') != std::string::npos)
		{
			std::string indexStr = subStringBetween(paramEntry, '[', ']');
			if (indexStr.empty())
				MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. Access to a collection has invalid syntax");
			if (!lastWasItem)
			{
				XMLDom::iterator parameterNode = destNode.find(paramName);
				if (parameterNode == destNode.end())
					return;
				destNode = parameterNode;
				lastWasItem = false;
			}
			// index is a key -> collection is a map (key,item)
			if (indexStr.find('\'') != std::string::npos)
			{
				std::string key = subStringBetween(indexStr, '\'', '\'');
				if (key.empty())
					MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. Access to collection " << paramName << " with (key) has invalid syntax");
				// We have <parameter name="/Unit.Collection['key']....">...
				// Look for an entry with <key>key<key>
				XMLDom::iterator keyIt = destNode.begin();
				for (; keyIt != destNode.end(); ++keyIt)
				{
					if (*keyIt == "key" &&
						keyIt.content_begin() != keyIt.content_end() &&
						*keyIt.content_begin() == key)
					break;
				}
				// No key was found in map -> create new one and an related item
				if (keyIt == destNode.end())
					return;
				// we found a key -> next child must be item
				destNode = ++keyIt;
				if (*destNode != "item")
					MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. Key "
					           << key << " in collection " << paramName
					           << " has invalid or missing item/value");
				lastWasItem = true;
			} // index is the search operator -> from here on we search for a child in the given path
			else if (indexStr == "?")
			{
				// If name="A.b[?].c.d" then the search operator works as follows:
				// we search for a child in the collection b that has a child c that has a child d
				// We then modify the parameters of d.
				// If the attribute hasChild is specified as hasChild="e.f" we additionally test if
				// d contains a child e that has a child f. But we still modify the parameters of d.
				std::string hasChild = iNode.get_attribute<std::string>("hasChild", "");
				std::vector<std::string> childResult;
				if (!hasChild.empty())
					boost::split(childResult, hasChild, boost::is_from_range('.','.'));
				// if we have no search path (neither in the name attribute nor in the hasChild attribute)
				// the search is invalid
				if (i==params.size()-1 && childResult.empty())
					MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. When using the search operator "
					           "you need to specify a search path either in the name or in the hasChild attribute");
				// from this position in the search path on we only search for sub childs but do not descend any
				// more as we want to modify parameters relative to the child at searchStop
				std::size_t searchStop = params.size();
				// add the hasChild path to the search path
				params.insert(params.end(), childResult.begin(), childResult.end());
				// the child we want to modify is now the one that matches the search path in the
				// name attribute and contains (if specified) childs that match the search path
				// given in the hasChild attribute
				XMLDom::iterator child = searchChilds(destNode, params, i+1, searchStop);
				if (child == destNode.end())
					return;
				destNode = child;
				break;
			}// index is a number or insert/del operation-> collection is a set, list or vector or
			// just a tag with many childs of the same name
			else
			{
				// we have either
				// 1. <parameter name="/Unit.Collection[index].ItemName....">... or
				// 2. <parameter name="/Unit.Collection[index]"><ItemName>.... or
				// 3. <parameter name="/Unit.Collection[+begin]"><ItemName>.... or
				// 4. <parameter name="/Unit.Collection[+end]"><ItemName>.... or
				// 5. <parameter name="/Unit.Collection[+index]"><ItemName>.... or
				// everything else is invalid.
				std::string itemName;
				// 1. name of item is next parameter in the name attribute
				if (i<params.size()-1)
					itemName = stripName(params[i+1]);
				else // 2. name of item is name of first and only child tag in the <parameter> node
				{
					XMLDom::iterator firstChild = iNode.begin();
					if (firstChild == iNode.end())
						MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. Parameter tag for access to items in collection "
						           << paramName << " must either have syntax <parameter name=\"Unit.Collection[index].ItemName\">... or "
						           "<parameter name=\"Unit.Collection[index]\"><ItemName>...");
					itemName = *firstChild;
				}

				uint32 index = fromString<uint32>(indexStr);
				// now search for nth (index) child with the given name
				XMLDom::iterator itemNode = destNode.find(itemName, index);
				if (itemNode == destNode.end())
					return;
				destNode = itemNode;
				lastWasItem = true;
			}
		} // normal parameter
		else
		{
			if (lastWasItem)
			{
				lastWasItem = false;
				continue;
			}
			XMLDom::iterator parameterNode = destNode.find(paramEntry);
			if (parameterNode == destNode.end())
				return;
			destNode = parameterNode;
		}
	}
	if (iNode.begin() != iNode.end() || iNode.content_begin() != iNode.content_end())
		MIRA_THROW(XInvalidConfig, "Error in <remove_parameter>. Remove parameter tag must not have any children or content.");
	destNode.remove();
}

XMLDom::iterator ParameterPreparer::parseParam(XMLDom::iterator& iNode)
{
	auto iName = iNode.find_attribute("name");
	if (iName == iNode.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <" << *iNode << ">. "
		           "No name attribute specified. "
		           "Usage <parameter name=\"namespace/Unit.MyParam\">...</parameter>.");

	std::string name = iName.value();

	// resolve the unit name

	// split the name as follows:
	//    some/path/../with/dots/UnitName.parameter.subparam
	//    |----------- prefix -----------|----- suffix ----|
	std::size_t lastSlash = name.rfind('/');
	if(lastSlash==std::string::npos)
		lastSlash=0;
	// find first dot after the last /
	std::size_t dotPos = name.find('.',lastSlash);
	std::string prefix = name;
	std::string suffix;
	if(dotPos!=std::string::npos) {
		prefix = name.substr(0, dotPos);
		suffix = name.substr(dotPos);
	}
	// resolve the prefix according to the processed <using>s and append the suffix
	name = (std::string)mNameRegistry.resolve(prefix, mNamespace) + suffix;

	XMLDom::iterator unit = findUnit(name);
	// for backward compatibility with old instance tag
	XMLDom::iterator instance = unit.find("instance");
	if (instance != unit.end())
		unit = instance;

	std::vector<std::string> paramResult;
	boost::split(paramResult, name, boost::is_from_range('.','.'));

	if (*iNode == "parameter")
		processParam(iNode, name, unit, paramResult);
	else
		processRemoveParam(iNode, name, unit, paramResult);

	iNode = iNode.remove();
	return iNode;
}

void ParameterPreparer::preprocessParam(XMLDom::iterator& iNode)
{
	XMLDom::iterator node = iNode.begin();
	for (; node != iNode.end();)
	{
		if (*node == "parameter" || *node == "remove_parameter")
		{
			try {
				node = parseParam(node);
			}
			catch (XInvalidConfig& ex) {
				ex.addInfo("while parsing tag '" + *node + "' with ParameterPreparer");
				MIRA_RETHROW(ex, "while parsing tag starting at: " << node.uri() << " (line " << node.line() << ")");
			}
		}
		else 
		{
			if (*node == "namespace")
			{
					ResourceName ns = mNamespace;
					try {
						ResourceName innerNs = ns / ResourceName(node.get_attribute<std::string>("name"));
						mNamespace = innerNs;
					} catch(...) {        // ignore exceptions here, they will be triggered in NamespaceLoader with a better error message
						++node; continue; // but skip processing the node's content!
					}
					preprocessParam(node);
					mNamespace = ns;
			}
			else if (*node == "using")
			{
				try {
					ResourceName name   = node.get_attribute<std::string>("name");
					ResourceName asName = node.get_attribute<std::string>("as", name.leaf());
					ResourceName ns     = mNamespace;

					mNameRegistry.addAlias(asName, name, ns);
				} catch(...) {} // ignore exceptions here, they will be triggered in UsingLoader with a better error message
			}
			else if (*node == "process")
				preprocessParam(node);

			++node;
		}
	}
}

void ParameterPreparer::prepareDocument(XMLDom& xml)
{
	mRoot = xml.root();
	preprocessParam(mRoot);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::ParameterPreparer, mira::ConfigurationPreparePlugin)

