/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Loader.C
 *    Implementation of Loader.h.
 *
 * @author Tim Langner
 * @date   2010/09/15
 */

#include <loader/Loader.h>

#include <error/Logging.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ConfigurationLoader::ConfigurationLoader(bool unknownTagWarning /*= true*/)
:	mEnableUnknownTagWarnings(unknownTagWarning)
{
}

///////////////////////////////////////////////////////////////////////////////

void ConfigurationLoader::registerPreparePlugin(ConfigurationPreparePluginPtr plugin)
{
	if(!plugin)
		return;
	mPreparePlugins.insert(plugin);
}

void ConfigurationLoader::registerLoaderPlugin(ConfigurationLoaderPluginPtr plugin)
{
	if(!plugin)
		return;

	std::vector<std::string> tags = plugin->getTags();
	foreach(const std::string& tag, tags)
		mLoaderPluginsMap[tag].insert(plugin);
	mLoaderPlugins.insert(plugin);
}

void ConfigurationLoader::parseNode(const XMLDom::const_iterator& node)
{
	mAddedInfoToException = false;
	try {

		LoaderPluginMap::iterator p = mLoaderPluginsMap.find(*node);
		if ( p == mLoaderPluginsMap.end() ) {
			std::string tag = *node;
			if(mEnableUnknownTagWarnings && !tag.empty() && tag!="comment") { // TODO: if mira::XMLDom supports this feature, check if node is an element node instead
				MIRA_LOG(WARNING) << "Unknown Tag: '<" << tag << ">' at: "
						<<  node.uri() << "(" << node.line() << ")";
			}
			parse(node);
		} else {
			foreach(auto ptr, p->second)
			{
				try {
					ptr->parseNode(node, this);
				} catch (Exception& ex) {
					if(!mAddedInfoToException) {
						MIRA_RETHROW(ex, "while parsing tag '" << *node << "' with loader " << ptr->getClass().getName());
					} else throw;
				}
			}
		}

	} catch (Exception& ex) {
		if(!mAddedInfoToException) {
			mAddedInfoToException = true;
			MIRA_RETHROW(ex, "while parsing tag starting at: "
			             <<  node.uri() << "(" << node.line() << ")");
		} else
			throw; // just rethrow
	} catch (std::exception& ex) {
			MIRA_THROW(XInvalidConfig, "Exception while parsing tag starting at: "
			           <<  node.uri() << "(" << node.line() << ") " << ex.what());
	}
}

void ConfigurationLoader::parse(const XMLDom::const_iterator& iNode)
{
	XMLDom::const_iterator node = iNode.begin();
	for ( ; node != iNode.end(); ++node)
		parseNode(node);
}

void ConfigurationLoader::prepare(XMLDom& xml)
{
	foreach(auto plugin, mPreparePlugins)
		plugin->prepareDocument(xml);
}

void ConfigurationLoader::load(const XMLDom& xml)
{
	// reset our state
	mContext.clear();

	// tell all plugins, that we start parsing a document
	foreach(auto ptr, mLoaderPlugins)
		ptr->startDocument(this);

	parseNode(xml.croot());

	foreach(auto ptr, mLoaderPlugins)
		ptr->endDocument(this);

}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::ConfigurationPreparePlugin, mira::Object)
MIRA_CLASS_REGISTER(mira::ConfigurationLoaderPlugin, mira::Object)
