/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NameRegistry.C
 *    Implementation of NameRegistry.h.
 *
 * @author Erik Einhorn
 * @date   2010/09/13
 */

#include <fw/NameRegistry.h>
#include <error/Exceptions.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void NameRegistry::addAlias(const ResourceName& from,
                            const ResourceName& to,
                            const ResourceName& ns,
                            const std::string& annotation)
{
	ResourceName fullFrom = ResourceName::makeFullyQualified(from, ns);
	ResourceName fullTo = ResourceName::makeFullyQualified(to, ns);
	auto r = mAliases.insert(make_pair(fullFrom, make_pair(fullTo, annotation)));
	if(!r.second) { // existed already
		// throw exception if there is a different "to" target
		if(r.first->second.first != fullTo)
			MIRA_THROW(XInvalidParameter, "Cannot add alias: " << fullFrom << " -> " << fullTo << ", as a different alias already exists: " << fullFrom << " -> " << r.first->second.first);
	}
}

ResourceName NameRegistry::resolve(const ResourceName& name, const ResourceName& ns) const
{
	// generate the full qualified name
	ResourceName fullname = ResourceName::makeFullyQualified(name, ns);

	// look up, if we have an alias for it
	// - the aliases are looked up iteratively to resolve transitive aliases.
	// - the process stops, if no more alias can be found.
	//    OR if the max depth is reached which indicates a loop within the alias chain
	for(int depth=0; ; ++depth) {
		AliasMap::const_iterator it = mAliases.find(fullname);
		if(it!=mAliases.end())
			fullname = it->second.first; // use the alias
		else
			break; // we are done: fullname now contains the max. resolved name

		if(depth>64) {
			MIRA_THROW(XLogical, "The alias chain of '" <<
				ResourceName::makeFullyQualified(name, ns).str() <<
				"' contains a loop.");
		}
	}

	return fullname;
}


///////////////////////////////////////////////////////////////////////////////

}
