/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelManager.C
 *    Implementation of ChannelManager.h.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/09/09
 */

#include <fw/ChannelManager.h>
#include <fw/Framework.h>
#include <fw/RemoteModule.h>

#include <boost/bimap.hpp>
#include <boost/bimap/multiset_of.hpp>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class ChannelManager::Pimpl
{
public:
	typedef boost::bimap<
			boost::bimaps::multiset_of<std::string>,
			boost::bimaps::multiset_of<ChannelInfo>> ChannelToPublisherSubcriberMap;

	ChannelToPublisherSubcriberMap mPublishedChannels;
	ChannelToPublisherSubcriberMap mSubscribedChannels;
};

///////////////////////////////////////////////////////////////////////////////

ChannelManager::ChannelManager() : p(new Pimpl),
                                   mAutoPromoteChannels(true), mCheckChannelRegistrations(false)
{

}

ChannelManager::~ChannelManager()
{
	foreach(ChannelMap::value_type& v, mChannels)
	{
		MIRA_LOG(DEBUG) << "Destroying channel: '" << v.first;
		delete v.second;
	}
	mChannels.clear();

	delete p;
}

void ChannelManager::unpublish(const std::string& publisherID, bool internal)
{
	std::set<std::string> channels = getPublishedChannelsBy(publisherID);
	{
		boost::mutex::scoped_lock lock(mPublisherMutex);
		MIRA_LOG(DEBUG) << "'" << publisherID << "' no longer publishes any channel";
		p->mPublishedChannels.right.erase(ChannelInfo(publisherID));
	}
	// if publisher was internal we have no more work here
	if (internal)
		return;

	// check for all channels we had published if we have other publishers
	// if not let the remote frameworks know
	foreach(const std::string& channel, channels)
	{
		if (!hasPublisher(channel, false))
			remoteUnpublishChannel(channel);
	}
}

void ChannelManager::unpublish(const std::string& channelID,
                               const std::string& publisherID, bool internal)
{
	{
		boost::mutex::scoped_lock lock(mPublisherMutex);
		Pimpl::ChannelToPublisherSubcriberMap::right_iterator first = p->mPublishedChannels.right.lower_bound(ChannelInfo(publisherID));
		Pimpl::ChannelToPublisherSubcriberMap::right_iterator last = p->mPublishedChannels.right.upper_bound(ChannelInfo(publisherID));

		for ( ; first != last; ++first )
			if ( first->second == channelID )
			{
				// if found erase it
				p->mPublishedChannels.right.erase(first);
				MIRA_LOG(DEBUG) << "'" << publisherID
				                << "' no longer publishes channel '"
				                << channelID << "'";
				// we have only one entry for the publisher - channel combination. leave the loop
				break;
			}
	}

	// if publisher was internal we have no more work here
	if (internal)
		return;

	// if there are no more non-internal publishers of that channel
	// we can inform connected frameworks about this fact
	if (!hasPublisher(channelID, false))
		remoteUnpublishChannel(channelID);
}

void ChannelManager::unsubscribe(const std::string& subscriberID)
{
	std::set<std::string> channels = getSubscribedChannelsBy(subscriberID);
	{
		boost::mutex::scoped_lock lock(mSubscriberMutex);
		MIRA_LOG(DEBUG) << "'" << subscriberID << "' unsubscribed from all channels";
		p->mSubscribedChannels.right.erase(ChannelInfo(subscriberID));
	}

	foreach(const std::string& channel, channels)
	{
		if (!hasSubscriber(channel))
			remoteUnsubscribeChannel(channel);
	}
}

void ChannelManager::unsubscribe(const std::string& channelID,
                                 const std::string& subscriberID)
{
	{
		boost::mutex::scoped_lock lock(mSubscriberMutex);
		Pimpl::ChannelToPublisherSubcriberMap::right_iterator first = p->mSubscribedChannels.right.lower_bound(ChannelInfo(subscriberID));
		Pimpl::ChannelToPublisherSubcriberMap::right_iterator last = p->mSubscribedChannels.right.upper_bound(ChannelInfo(subscriberID));

		for ( ; first != last; ++first )
			if ( first->second == channelID )
			{
				// if found erase it
				p->mSubscribedChannels.right.erase(first);
				MIRA_LOG(DEBUG) << "'" << subscriberID 
					<< "' unsubscribed from channel '" << channelID << "'";
				// we have only one entry for the subscriber - channel combination.
				// leave the loop
				break;
			}
	}
	if (!hasSubscriber(channelID))
		remoteUnsubscribeChannel(channelID);
}

AbstractChannelPtr ChannelManager::getAbstractChannelNoLocking(const std::string& channelID)
{
	ChannelMap::iterator it = mChannels.find(channelID);
	if(it==mChannels.end())
		MIRA_THROW(XUnknownChannel, "Channel '" << channelID << "' does not exist");
	return it->second;
}

const AbstractChannel* ChannelManager::getAbstractChannelNoLocking(const std::string& channelID) const
{
	ChannelMap::const_iterator it = mChannels.find(channelID);
	if(it==mChannels.end())
		MIRA_THROW(XUnknownChannel, "Channel '" << channelID << "' does not exist");
	return it->second;
}

AbstractChannelPtr ChannelManager::getAbstractChannel(const std::string& channelID)
{
	boost::mutex::scoped_lock lock(mChannelsMutex);
	return getAbstractChannelNoLocking(channelID);
}

const AbstractChannel* ChannelManager::getAbstractChannel(const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mChannelsMutex);
	return getAbstractChannelNoLocking(channelID);
}

std::set<std::string> ChannelManager::getChannels() const
{
	boost::mutex::scoped_lock lock(mChannelsMutex);
	std::set<std::string> r;
	foreach(const ChannelMap::value_type& i, mChannels)
		r.insert(i.first);
	return r;
}

std::set<std::string> ChannelManager::getChannelList(bool publishedOnly,
                                                     bool subscribedOnly) const
{
	if (publishedOnly)
	{
		std::set<std::string> publishedChannelsSet;

		std::map<std::string, Typename> publishedChannels = getPublishedChannels();
		if (subscribedOnly)
		{
			// publishedOnly && subscribedOnly
			std::set<std::string> subscribedChannels = getSubscribedChannels();

			foreach(auto const & c, publishedChannels)
			{
				if (subscribedChannels.find(c.first) != subscribedChannels.end())
					publishedChannelsSet.insert(c.first);
			}
		}
		else
		{
			// publishedOnly && !subscribedOnly
			foreach(auto const & c, publishedChannels)
				publishedChannelsSet.insert(c.first);

			return publishedChannelsSet;
		}
	}
	else if (subscribedOnly)
		// !publishedOnly && subscribedOnly
		return getSubscribedChannels();

	// !publishedOnly && !subscribedOnly
	return getChannels();
}

std::map<std::string, Typename> ChannelManager::getPublishedChannels(bool includeInternal) const
{
	boost::mutex::scoped_lock lock(mPublisherMutex);
	std::map<std::string, Typename> r;
	foreach(const Pimpl::ChannelToPublisherSubcriberMap::value_type& i, p->mPublishedChannels)
	{
		if (!i.right.internal || includeInternal)
			r[i.left] = getTypename(i.left);
	}
	return r;
}

std::set<std::string> ChannelManager::getPublishedChannelsBy(const std::string& publisherID) const
{
	boost::mutex::scoped_lock lock(mPublisherMutex);
	std::set<std::string> r;
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator first = p->mPublishedChannels.right.lower_bound(ChannelInfo(publisherID));
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator last = p->mPublishedChannels.right.upper_bound(ChannelInfo(publisherID));

	for ( ; first != last; ++first )
		r.insert(first->second);
	return r;
}

bool ChannelManager::hasPublished(const std::string& publisherID,
                                  const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mPublisherMutex);

	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator first = p->mPublishedChannels.right.lower_bound(ChannelInfo(publisherID));
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator last = p->mPublishedChannels.right.upper_bound(ChannelInfo(publisherID));

	for ( ; first != last; ++first )
		if (first->second == channelID)
			return true;

	return false;
}

std::set<std::string> ChannelManager::getSubscribedChannels(bool includeInternal) const
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);
	std::set<std::string> r;
	foreach(const Pimpl::ChannelToPublisherSubcriberMap::value_type& i, p->mSubscribedChannels)
	{
		if (!i.right.internal || includeInternal)
			r.insert(i.left);
	}
	return r;
}

std::set<std::string> ChannelManager::getSubscribedChannelsBy(const std::string& subscriberID) const
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);
	std::set<std::string> r;
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator first = p->mSubscribedChannels.right.lower_bound(ChannelInfo(subscriberID));
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator last = p->mSubscribedChannels.right.upper_bound(ChannelInfo(subscriberID));

	for ( ; first != last; ++first )
		r.insert(first->second);
	return r;
}

bool ChannelManager::isSubscribedOn(const std::string& subscriberID,
                                    const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);

	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator first = p->mSubscribedChannels.right.lower_bound(ChannelInfo(subscriberID));
	Pimpl::ChannelToPublisherSubcriberMap::right_const_iterator last = p->mSubscribedChannels.right.upper_bound(ChannelInfo(subscriberID));

	for ( ; first != last; ++first )
		if (first->second == channelID)
			return true;

	return false;
}

std::list<std::pair<std::string, Typename>> ChannelManager::getChannelsOfType(const Typename& type) const
{
	boost::mutex::scoped_lock lock(mChannelsMutex);
	std::list<std::pair<std::string, Typename>> r;

	bool isVoid = type == "void" || type.empty();

	foreach(const ChannelMap::value_type& i, mChannels)
	{
		if(i.second->getTypename()==type || isVoid)
			r.push_back(std::make_pair(i.first, i.second->getTypename()));
	}
	return r;
}

bool ChannelManager::hasSubscriber(const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);
	return p->mSubscribedChannels.left.count(channelID) > 0;
}

bool ChannelManager::hasPublisher(const std::string& channelID, bool includeInternal) const
{
	boost::mutex::scoped_lock lock(mPublisherMutex);
	foreach(const Pimpl::ChannelToPublisherSubcriberMap::value_type& i, p->mPublishedChannels)
	{
		if ((!i.right.internal || includeInternal) && i.left == channelID)
			return true;
	}
	return false;
}

uint32 ChannelManager::getNrPublishers(const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mPublisherMutex);
	return p->mPublishedChannels.left.count(channelID);
}

uint32 ChannelManager::getNrSubscribers(const std::string& channelID) const
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);
	return p->mSubscribedChannels.left.count(channelID);
}

bool ChannelManager::hasChannel(const std::string& channelID) const
{
	return mChannels.count(channelID) > 0;
}

int ChannelManager::getTypeId(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getTypeId();
}

void ChannelManager::setTypename(const std::string& channelID, const Typename& typenam)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	Typename t = channel->getTypename();

	if(t.empty())
		channel->setTypename(typenam);
	else if(t!=typenam)
		MIRA_THROW(XLogical, "Channel '" << channelID << "' is already typed with '" << t << "'");
}

Typename ChannelManager::getTypename(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getTypename();
}

void ChannelManager::setTypeMeta(const std::string& channelID, TypeMetaPtr meta)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	channel->setTypeMeta(meta);
}


TypeMetaPtr ChannelManager::getTypeMeta(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getTypeMeta();
}


Time ChannelManager::getLastSlotTime(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getLastSlotTime();
}

std::size_t ChannelManager::getNrOfSlots(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getNrOfSlots();
}

uint64 ChannelManager::getNrOfDataChanges(const std::string& channelID) const
{
	return getAbstractChannel(channelID)->getNrOfDataChanges();
}

void ChannelManager::setStorageDuration(const std::string& channelID, const Duration& storageDuration)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	channel->getBuffer()->setStorageDuration(storageDuration);
}

void ChannelManager::setAutoIncreaseStorageDuration(const std::string& channelID, bool increase)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	channel->getBuffer()->setAutoIncreaseStorageDuration(increase);
}

void ChannelManager::setMinSlots(const std::string& channelID, std::size_t minSlots)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	channel->getBuffer()->setMinSlots(minSlots);
}

void ChannelManager::setMaxSlots(const std::string& channelID, std::size_t maxSlots)
{
	ConcreteChannel<void>* channel = obtainConcreteChannel<void>(channelID);
	channel->getBuffer()->setMaxSlots(maxSlots);
}

///////////////////////////////////////////////////////////////////////////////

void ChannelManager::writeChannel(const std::string& channelID,
                                  const json::Value& value,
                                  const Time& time)
{
	ConcreteChannel<void>* channel = getConcreteChannel<void>(channelID);
	ChannelWrite<void> cw = channel->write();
	cw->timestamp = time;
	cw.writeJSON(value);
}

json::Value ChannelManager::readChannel(const std::string& channelID)
{
	ConcreteChannel<void>* channel = getConcreteChannel<void>(channelID);
	ChannelRead<void> cr = channel->read();
	json::Value r;
	cr.readJSON(r);
	return r;
}

void ChannelManager::remotePublishChannel(const std::string& channelID,
                                          const Typename& type)
{
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote)
		remote->publishChannel(channelID, type);
}

void ChannelManager::remoteUnpublishChannel(const std::string& channelID)
{
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote)
		remote->unpublishChannel(channelID);
}

void ChannelManager::remoteSubscribeChannel(const std::string& channelID)
{
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote)
		remote->subscribeChannel(channelID);
}

void ChannelManager::remoteUnsubscribeChannel(const std::string& channelID)
{
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote)
		remote->unsubscribeChannel(channelID);
}

///////////////////////////////////////////////////////////////////////////////

void ChannelManager::insertPublishedChannel(const std::string& channelID,  const ChannelInfo& info)
{
	boost::mutex::scoped_lock lock(mPublisherMutex);
	p->mPublishedChannels.insert(Pimpl::ChannelToPublisherSubcriberMap::value_type(channelID,info));
}

void ChannelManager::insertSubscribedChannel(const std::string& channelID,  const ChannelInfo& info)
{
	boost::mutex::scoped_lock lock(mSubscriberMutex);
	p->mSubscribedChannels.insert(Pimpl::ChannelToPublisherSubcriberMap::value_type(channelID,info));
}

///////////////////////////////////////////////////////////////////////////////

}
