/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkMessage.h
 *    Message types exchanged between remote frameworks.
 *
 * @author Tim Langner
 * @date   2010/09/13
 */

#ifndef _MIRA_FRAMEWORKMESSAGE_H_
#define _MIRA_FRAMEWORKMESSAGE_H_

#ifndef Q_MOC_RUN
#include <boost/asio/buffer.hpp>
#endif

#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Remote framework message types
 */
enum FrameworkMessageType
{
	CONNECT_MSG = 1,
	SERVER_AUTH_MSG,
	CLIENT_AUTH_MSG,
	CONNECT_ACCEPT_MSG,
	CONNECT_DENIED_MSG,
	PTP_SYNC,
	PTP_FOLLOW_UP,
	PTP_DELAY_REQUEST,
	PTP_DELAY_RESPONSE,
	PTP_FINISH,
	PUBLISH_CHANNEL_MSG,
	UNPUBLISH_CHANNEL_MSG,
	PUBLISH_SERVICE_MSG,
	UNPUBLISH_SERVICE_MSG,
	PUBLISH_AUTHORITY_MSG,
	UNPUBLISH_AUTHORITY_MSG,
	SUBSCRIBE_CHANNEL_MSG,
	UNSUBSCRIBE_CHANNEL_MSG,
	WRITE_CHANNEL_MSG,
	RPC_REQUEST_MSG,
	RPC_RESPONSE_MSG,
	REQUEST_MIGRATION_MSG,
	MIGRATION_MSG,
	MIGRATION_SINK_SUCCESS_MSG,
	MIGRATION_SINK_FAILURE_MSG,
	MIGRATION_FINISHED_MSG,
	TYPE_METADB_MSG,
	CHANNEL_META_MSG,
	PING_MSG,

	MAX_MSG_COUNT
};

/**
 * Data that is sent as header in each message between remote frameworks.
 */
struct FrameworkMessageHeader
{
	/// Does not initialize the header.
	FrameworkMessageHeader() {}
	/// Initializes the call header with the specified values.
	FrameworkMessageHeader(uint32 size, FrameworkMessageType messagetype) :
		messageSize(size),
		messageType((uint8)messagetype)
	{}

	uint32 messageSize;
	uint32 messageType;

	/**
	 * Returns this message as boost asio buffer
	 */
	boost::asio::mutable_buffers_1 asBuffer()
	{
		return boost::asio::buffer(this, sizeof(FrameworkMessageHeader));
	}
};

// make sure, the compiler creates a header structure with the correct size
static_assert(sizeof(FrameworkMessageHeader)==8,
              "compiler created invalid message header size");

///////////////////////////////////////////////////////////////////////////////

}

#endif
